/* @(#) Copyright (c), 1987, 2001 Insightful Corp.  All rights reserved. */
/* @(#) $RCSfile: EvaluatorManager.h,v $: $Revision: #4 $, $Date: 2001/08/28 $  */
#ifndef EVALUATOR_MANAGER_H
#define EVALUATOR_MANAGER_H

#include "Thread.h"
#include "ThreadStructs.h"
#include "ThreadEnums.h"

/**
  This file defines the majority of the 
  Evaluator Manager's internal functionality.

  The routines called from S and used to communicate with the Manager
  are provided elsewhere (see EvaluatorManagerInterace.c).
**/


  /* Information that describes the status of a thread for use in ps and xps.
      Total CPU time used, total time since start, memory size, etc.
     These are to be added later.
   */
typedef struct {
  time_t start_time;
} ThreadRunTimeStatus;

  /* 
     A structure for handling messages/requests from threads 
     to the Evaluator Manager.
     These are used to notify the manager of an event such as 
     thread cancellation so that the manager can notify other 
     threads that are join()ing on that particular thread and 
     update the status of that thread, thread suspension.
     We use these messages also to handle all remote thread requests - in other words
     operations by one thread on another - such as querying the status of the
     other thread, its attributes, its related threads (children, parent), database,
     etc.
   */
typedef struct _ManagerMessage {
  s_boolean answered;
  ManagerMessageTypes type;      /* The type of message - see ManagerMessageTypes above */
  void *data;                    /* Any data provided by the requesting thread for the operation */

  /* Used to be void** - make sure all the changes are made */ 
  void *return_value;            /* where to put the return value. If this is NULL,
                                    the requesting thread is not waiting and no return value
                                    is required.
                                  */
  s_boolean you_free_it;           /* Whether the requesting thread should free the return value.
                                    The manager will probably allocate the return value's space
                                    in the threads permanent area.
                                  */
  EvalThread *source;            /* The thread that made the request and put this on the queue */

  struct _ManagerMessage *next;  /* Next element in the list */

     /* Necessary facilities for locking and notifying the queue correspondents */
  MUTEX *lock;
  COND_VAR  *lock_var;

} ManagerMessage, *ManagerMessageQueue;



typedef struct {
  s_object *expression;
  s_index  *threads;
  int num_threads;
  void *data;
} MessageData;

#define MAX_NUM_THREADS 64  
#define MAX_NUM_ACTIVE_THREADS 60

  /* The evaluator manager maintains a global list of
     locks currently in operation. (A mechanism can be used to cache these
     in the future).
     Also, it maintains a list of the threads currently in existence.
     The different types are held in different lists/tables so
     that we can get easy access to the different types (evaluator threads,
     reader threads, wait threads, and any others we may introduce)
   */

typedef struct _EvaluatorManager {

    /* We don't have the typedef yet as the EvaluatorManager passed in the second argument
       of the handler has not been defined yet.
     */
 s_boolean (**method_table)(ManagerMessage *msg, struct _EvaluatorManager *mgr, s_evaluator *S_evaluator);                          /* array of methods that correspond to the message types by index */
 long num_methods;                  /* Number of methods in the table */


 Thread thread;
 EvalThread *slave;    /* Similar to init process, this is the first evaluator 
                                thread created in the process.
                              */
               /* List of the currently active threads */
 int num_tables;
 ThreadTable thread_type_tables[NUM_TYPES_THREADS];

    /* A pool of available threads that can be used when asked to 
       create a new thread available_reader_threads,
       available_evaluators, available_timers */
 ThreadList thread_pools[NUM_TYPES_THREADS];

 ThreadTable waiting_threads;  /* Table of threads currently waiting on other threads to do something 
                                */

 MUTEX *queue_lock;
 COND_VAR *queue_lock_var;
 ManagerMessageQueue queue;  /* Events from threads are sent to the manager via this queue */


  /* Tables for handling thread locks.
     This consists of an index table with the keys being the database ids.
     The values are name tables indexed by the name of the object in that
     database for which events are being generated and waited on.
     The values in this name table are linked lists. These lists contain
     the ThreadLockCondition objects. See ThreadLock.c.
   */
 s_index_table *threadLock_table;

} EvaluatorManager;

extern EvaluatorManager *EvalManager;
typedef s_boolean (*ManagerMessageHandler)(ManagerMessage *msg, EvaluatorManager *mgr, s_evaluator *S_evaluator);
typedef ThreadStatus (*ActionMessageHandler)(EvalThread *, ManagerMessage *);


extern ActionMessageHandler ActionMessageHandlers[];


#define FAKE_MESSAGE_INDEX -1

  /* structure for registering/replacing an EvaluatorManager method */
typedef struct {
  ManagerMessageHandler handler;  
  long message_id;
} ManagerMethodData;

  /* structure for supplying a list of threads to a manager request. 
     This is used in more than just {S,G}ET_STATUS requests in which case
     the status field is usually ignored.
   */
typedef struct {
  long num_threads;
  long *threads;
  long status;
} StatusData;


typedef struct {
  s_index id;
  long num_children;
  long *children;
} ChildrenInfo;

typedef struct {
  s_index thread_id;
  ObjectID object_type;
} ObjectInfo;

typedef struct {
  s_index thread_id;
  s_object *expression;
} TaskData;


#define CURRENT_EVALUATOR_FRAME -999
typedef struct {
  s_index thread_id; 
  long frame_id;
  s_object *object;
} CopyObjectData;


#define RETURN_MESSAGE(msg) 

#define WAIT_MESSAGE_RESPONSE(msg) \
  if(msg->lock) { \
    LOCK(msg->lock);  \
    while(msg->answered != S_TRUE) { \
      COND_WAIT(msg->lock_var, msg->lock); \
    } \
    UNLOCK(msg->lock); \
  }


/**************************************************************************************/

       /* Internal C routines used to interface to the Pthread routines */
s_boolean yield_S_thread();
s_boolean join_S_thread();
s_boolean exit_S_thread();
s_boolean start_S_thread();
s_boolean suspend_S_thread();
s_boolean cancel_S_thread();


       /* S C interface functions for handling threads. */
s_object *S_createThread(s_object *s_thread, s_object *);

s_object *S_sendTask(s_object *s_thread, s_object *s_expression, s_object *s_background);
s_object *S_compareStatus(s_object *s_threads, s_object *s_type);
s_object *S_ThreadGetAttributes(s_object *s_threads);
s_object *S_ThreadSetAttributes(s_object *s_threads);

s_object *S_dbOf(s_object *s_threads);
s_object *S_valueOf(s_object *s_threads);
s_object *S_searchPath(s_object *s_threads);
s_object *S_threadOf(s_object *s_threads);

s_object *S_getParentThread();
s_object *S_getChildrenThreads();

s_object *S_onThreadExit(s_object *s_thread, s_object *s_expression);

s_object *S_getThreadList();
s_object *S_getThreadQueue();

 /*   */

s_boolean Manager_eval(ManagerMessage *msg, EvaluatorManager *mgr, s_evaluator *S_evaluator);

s_boolean set_thread_status(ManagerMessage *msg, EvaluatorManager *mgr);

EvalThread *get_evaluator_from_index(EvaluatorManager *mgr, s_index id);

EvalThread* get_self();


s_object *get_current_task(EvalThread *thread);
s_boolean suspend_thread(EvalThread *thread);
s_boolean continue_thread(EvalThread *thread);

s_index construct_thread(EvaluatorManager *mgr, ManagerMessage *msg, s_evaluator *S_evaluator);
s_boolean process_message(ManagerMessage *message, EvaluatorManager *mgr, s_evaluator *S_evaluator);
s_boolean get_thread_object(ManagerMessage *msg, EvaluatorManager *mgr,s_evaluator *S_evaluator);
s_boolean create_new_thread(ManagerMessage *msg, EvaluatorManager *mgr, s_evaluator *S_evaluator);
s_boolean send_task(ManagerMessage *msg, EvaluatorManager *mgr, s_evaluator *S_evaluator);
s_boolean eval_manager_quit(ManagerMessage *msg, EvaluatorManager *mgr, s_evaluator *S_evaluator);
s_boolean manager_status(ManagerMessage *msg, EvaluatorManager *mgr,s_evaluator *S_evaluator);
s_boolean get_threads_parent(ManagerMessage *msg, EvaluatorManager *mgr, s_evaluator *S_evaluator);
s_boolean check_access_request(ManagerMessage *msg, EvaluatorManager *mgr, s_evaluator *S_evaluator);
s_boolean get_thread_value(ManagerMessage *msg, EvaluatorManager *mgr);
s_boolean do_thread_action(ManagerMessage *msg, EvaluatorManager *mgr);
s_boolean get_thread_children(ManagerMessage *msg, EvaluatorManager *mgr,s_evaluator *S_evaluator);
s_boolean remove_task(ManagerMessage *msg, EvaluatorManager *mgr, s_evaluator *S_evaluator);
s_object *S_removeTask(s_object *s_expression, s_object *s_thread, s_object *s_background);


s_boolean check_status(s_evaluator *e);
void *EvaluatorManger_routine(void *arg);
s_boolean isDaemonThread(EvalThread *thread);
s_boolean RemoveTask(EvalThread *thread, SThreadTask *task);
void *EvaluatorManager_routine(void*);
s_evaluator *get_evaluator(s_index id, EvaluatorManager *mgr);
s_evaluator *get_s_evaluator(EvaluatorManager *mgr, s_index id);
int add_child_thread(s_index parent, s_index child, EvaluatorManager *mgr, s_evaluator *S_evaluator);

s_boolean evaluator_init(EvalThread *thread, s_index id, s_index parent_id);

s_boolean all_threads(ManagerMessage *msg, EvaluatorManager *mgr, s_evaluator *S_evaluator);
s_boolean get_thread_object(ManagerMessage *msg, EvaluatorManager *mgr, s_evaluator *S_evaluator);
s_boolean get_thread_db(ManagerMessage *msg, EvaluatorManager *mgr, s_evaluator *S_evaluator);

s_object *
remote_thread_searchPath(s_evaluator *local_thread, s_evaluator *remote_thread, s_db_purposes db_for);

s_boolean RemoveNextRequest(EvalThread *thread, s_object *expression);
s_boolean check_status(s_evaluator *e);
s_boolean removeEvaluatorThread(EvalThread *thread, s_evaluator *S_evaluator);
s_boolean manager_mem_exceeded(ManagerMessage *msg, EvaluatorManager *mgr, s_evaluator *S_evaluator);

s_boolean check_thread_access(EvalThread *src, EvalThread *target);
long initialize_manager_method_table(EvaluatorManager *mgr);

s_boolean
register_manager_method(ManagerMessage *msg, EvaluatorManager *mgr, s_evaluator *S_evaluator);

s_boolean set_manager_method(ManagerMessageHandler handler, long *id, s_evaluator *S_evaluator);


s_object *
request_manager_copy(s_object *object, s_index thread_id, long frame_id, s_evaluator *S_evaluator);

s_boolean
manager_copy_object(ManagerMessage *msg, EvaluatorManager *mgr,s_evaluator *S_evaluator);
#endif
