/* @(#) Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved. */
/* @(#) $RCSfile: S_obj.h,v $: $Revision: #8 $, $Date: 2006/06/26 $  */
/* This is a self-documenting doc++ file. */

#ifndef _IN_S_OBJ_H
#define _IN_S_OBJ_H

#include "libext.h"

/* This file contains declarations of functions used to
   access Splus objects passed to C functions via .Call.
*/


/*
**************************
Verification functions
**************************
*/


/** Verify that the input object has given class name.
 * This function looks at the actual class name of the
 * input object, and verifies that it matches the input name.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_verify_class(sobj, "numeric");#
 * @return 1 if the name matches, 0 if not or if there is an error.
 * @param obj     Splus object to check.
 * @param clname  Name to look for.
 */
LibExtern int S_STDCALL S_verify_class( s_object *obj, const char *clname );


/*
**************************
Functions for finding the data in Splus objects
for manipulating in C
**************************
*/


/** Extract data from an Splus character vector.
 * Find the length and a pointer to the data array in
 * an S character vector object, using macros from S.h.
 * Fails if the object type is incorrect, or if pointers are NULL.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_extract_character(&myobj, &len, &strings);#
 * @return 1/0 for success/failure.
 * @param input_vec  The Splus object to read.
 * @param ret_len    Length of the input object.
 * @param data_ptr   Pointer to the data array in input object.
 * @see S_extract_numeric
 * @see S_extract_integer
 * @see S_extract_logical
 * @see S_extract_complex
 * @see S_extract_datafrm_data
 */
LibExtern int S_STDCALL S_extract_character( s_object *input_vec, long *ret_len,
  char ***data_ptr );


/** Extract data from an Splus integer vector.
 * Find the length and a pointer to the data array in
 * an S integer vector object, using macros from S.h.
 * Fails if the object type is incorrect, or if pointers are NULL.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_extract_integer(&myobj, &len, &data);#
 * @return 1/0 for success/failure.
 * @param input_vec  The Splus object to read.
 * @param ret_len    Length of the input object.
 * @param data_ptr   Pointer to the data array in input object.
 * @see S_extract_numeric
 * @see S_extract_character
 * @see S_extract_logical
 * @see S_extract_complex
 * @see S_extract_datafrm_data
 */
LibExtern int S_STDCALL S_extract_integer( s_object *input_vec, long *ret_len,
  long **data_ptr );


/** Extract data from an Splus logical vector.
 * Find the length and a pointer to the data array in
 * an S logical vector object, using macros from S.h.
 * Fails if the object type is incorrect, or if pointers are NULL.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_extract_logical(&myobj, &len, &data);#
 * @return 1/0 for success/failure.
 * @param input_vec  The Splus object to read.
 * @param ret_len    Length of the input object.
 * @param data_ptr   Pointer to the data array in input object.
 * @see S_extract_numeric
 * @see S_extract_character
 * @see S_extract_integer
 * @see S_extract_complex
 * @see S_extract_datafrm_data
 */
LibExtern int S_STDCALL S_extract_logical( s_object *input_vec, long *ret_len,
  long **data_ptr );


/** Extract data from an Splus numeric vector.
 * Find the length and a pointer to the data array in
 * an S numeric vector object, using macros from S.h.
 * Fails if the object type is incorrect, or if pointers are NULL.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_extract_numeric(&myobj, &len, &data);#
 * @return 1/0 for success/failure.
 * @param input_vec  The Splus object to read.
 * @param ret_len    Length of the input object.
 * @param data_ptr   Pointer to the data array in input object.
 * @see S_extract_character
 * @see S_extract_integer
 * @see S_extract_logical
 * @see S_extract_complex
 * @see S_extract_datafrm_data
 */
LibExtern int S_STDCALL S_extract_numeric( s_object *input_vec, long *ret_len,
  double **data_ptr );


/** Extract data from an Splus complex vector.
 * Find the length and a pointer to the data array in
 * an S complex vector object, using macros from S.h.
 * Fails if the object type is incorrect, or if pointers are NULL.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_extract_complex(&myobj, &len, &data);#
 * @return 1/0 for success/failure.
 * @param input_vec  The Splus object to read.
 * @param ret_len    Length of the input object.
 * @param data_ptr   Pointer to the data array in input object.
 * @see S_extract_character
 * @see S_extract_integer
 * @see S_extract_logical
 * @see S_extract_numeric
 * @see S_extract_datafrm_data
 */
LibExtern int S_STDCALL S_extract_complex( s_object *input_vec, long *ret_len,
  S_complex **data_ptr );


/** Extract data from an Splus data frame.
 * Find the length and a pointer to the list data in
 * an S data frame object, using macros from S.h.
 * Fails if the object type is incorrect, or if pointers are NULL.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_extract_datafrm_data(&myobj, &len, &data);#
 * @return 1/0 for success/failure.
 * @param input_vec  The Splus object to read.
 * @param ret_len    Length of the input object.
 * @param data_ptr   Pointer to the data array in input object.
 * @see S_extract_character
 * @see S_extract_integer
 * @see S_extract_logical
 * @see S_extract_numeric
 * @see S_extract_complex
 */
LibExtern int S_STDCALL S_extract_datafrm_data( s_object *input_vec, long *ret_len,
  s_object **data_ptr );


/*
**************************
Functions for creating objects to fill
with data and return to Splus
**************************
*/


/** Create an Splus character vector.
 * Create a new Splus character vector object and return a
 * pointer to the new object and its internal data pointer.
 * The vector is created and pointers extracted using the macros in S.h.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_create_character(Nelem, &myobj, &strings);#
 * @return 1/0 for success/failure.
 * @param len      Length of the desired object.
 * @param ret_vec  Pointer for returning the created object.
 * @param data_ptr Pointer to the data array in created object.
 * @see S_create_numeric
 * @see S_create_integer
 * @see S_create_logical
 * @see S_create_complex
 * @see S_create_list
 */
LibExtern int S_STDCALL S_create_character( long len, s_object **ret_vec,
  char ***data_ptr );


/** Create an Splus integer vector.
 * Create a new Splus integer vector object and return
 * a pointer to the new object and its internal data pointer.
 * The vector is created and pointers extracted using the macros in S.h.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_create_integer(Nelem, &myobj, &data);#
 * @return 1/0 for success/failure.
 * @param len      Length of the desired object.
 * @param ret_vec  Pointer for returning the created object.
 * @param data_ptr Pointer to the data array in created object.
 * @see S_create_numeric
 * @see S_create_character
 * @see S_create_logical
 * @see S_create_complex
 * @see S_create_list
 */
LibExtern int S_STDCALL S_create_integer( long len, s_object **ret_vec, long **data_ptr );


/** Create an Splus logical vector.
 * Create a new Splus logical vector object and return
 * a pointer to the new object and its internal data pointer.
 * The vector is created and pointers extracted using the macros in S.h.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_create_logical(Nelem, &myobj, &data);#
 * @return 1/0 for success/failure.
 * @param len      Length of the desired object.
 * @param ret_vec  Pointer for returning the created object.
 * @param data_ptr Pointer to the data array in created object.
 * @see S_create_numeric
 * @see S_create_integer
 * @see S_create_character
 * @see S_create_complex
 * @see S_create_list
 */
LibExtern int S_STDCALL S_create_logical( long len, s_object **ret_vec, long **data_ptr );


/** Create an Splus numeric vector.
 * Create a new Splus numeric vector object and return
 * a pointer to the new object and its internal data pointer.
 * The vector is created and pointers extracted using the macros in S.h.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_create_numeric(Nelem, &myobj, &data);#
 * @return 1/0 for success/failure.
 * @param len      Length of the desired object.
 * @param ret_vec  Pointer for returning the created object.
 * @param data_ptr Pointer to the data array in created object.
 * @see S_create_character
 * @see S_create_integer
 * @see S_create_logical
 * @see S_create_complex
 * @see S_create_list
 */
LibExtern int S_STDCALL S_create_numeric( long len, s_object **ret_vec,
  double **data_ptr );


/** Create an Splus complex vector.
 * Create a new Splus complex vector object and return
 * a pointer to the new object and its internal data pointer.
 * The vector is created and pointers extracted using the macros in S.h.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_create_complex(Nelem, &myobj, &data);#
 * @return 1/0 for success/failure.
 * @param len      Length of the desired object.
 * @param ret_vec  Pointer for returning the created object.
 * @param data_ptr Pointer to the data array in created object.
 * @see S_create_character
 * @see S_create_integer
 * @see S_create_logical
 * @see S_create_numeric
 * @see S_create_list
 */
LibExtern int S_STDCALL S_create_complex( long len, s_object **ret_vec,
  S_complex **data_ptr );


/** Create an Splus list.
 * Create a new Splus list object and return
 * a pointer to the new object and its internal data pointer.
 * The list is created and pointers extracted using the macros in S.h.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_create_list(Nelem, &myobj, &rec_ptr);#
 * @return 1/0 for success/failure.
 * @param len      Length of the desired object.
 * @param ret_vec  Pointer for returning the created object.
 * @param data_ptr Pointer to the recursive data area in created object.
 * @see S_create_numeric
 * @see S_create_integer
 * @see S_create_logical
 * @see S_create_character
 * @see S_set_list_integer
 * @see S_set_list_numeric
 * @see S_set_list_logical
 * @see S_create_complex
 */
LibExtern int S_STDCALL S_create_list( long len, s_object **ret_vec, s_object ***data_ptr );


/*
**************************
Functions for filling Splus lists
with data
**************************
*/


/** Add an integer vector to a list.
 * Create a new Splus integer vector object and put it
 * into an existing Splus list.  The vector is created using
 * \Ref{S_create_integer}, and inserted using macros in S.h.  The
 * list must already be long enough to hold the new element.
 * Fails if the list object type is incorrect, or if pointers are NULL.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_set_list_integer(&listobj, elem, len, &data);#
 * @return 1/0 for success/failure.
 * @param lst      Splus list object to add to.
 * @param ele      Index of the element to put the vector in.
 * @param len      Length of the desired vector.
 * @param data_ptr Pointer to the data array in created vector.
 * @see S_create_list
 * @see S_create_integer
 * @see S_set_list_numeric
 * @see S_set_list_logical
 * @see S_set_list_complex
 */
LibExtern int S_STDCALL S_set_list_integer( s_object *lst, long ele, long len,
  long **data_ptr );


/** Add an numeric vector to a list.
 * Create a new Splus numeric vector object and put it
 * into an existing Splus list.  The vector is created using
 * \Ref{S_create_numeric}, and inserted using macros in S.h.  The
 * list must already be long enough to hold the new element.
 * Fails if the list object type is incorrect, or if pointers are NULL.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_set_list_numeric(&listobj, elem, len, &data);#
 * @return 1/0 for success/failure.
 * @param lst      Splus list object to add to.
 * @param ele      Index of the element to put the vector in.
 * @param len      Length of the desired vector.
 * @param data_ptr Pointer to the data array in created vector.
 * @see S_create_list
 * @see S_create_numeric
 * @see S_set_list_integer
 * @see S_set_list_logical
 * @see S_set_list_complex
 */
LibExtern int S_STDCALL S_set_list_numeric( s_object *lst, long ele, long len,
  double **data_ptr );


/** Add a logical vector to a list.
 * Create a new Splus logical vector object and put it
 * into an existing Splus list.  The vector is created using
 * \Ref{S_create_logical}, and inserted using macros in S.h.  The
 * list must already be long enough to hold the new element.
 * Fails if the list object type is incorrect, or if pointers are NULL.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_set_list_logical(&listobj, elem, len, &data);#
 * @return 1/0 for success/failure.
 * @param lst      Splus list object to add to.
 * @param ele      Index of the element to put the vector in.
 * @param len      Length of the desired vector.
 * @param data_ptr Pointer to the data array in created vector.
 * @see S_create_list
 * @see S_create_logical
 * @see S_set_list_integer
 * @see S_set_list_numeric
 * @see S_set_list_complex
 */
LibExtern int S_STDCALL S_set_list_logical( s_object *lst, long ele, long len,
  long **data_ptr );


/** Add an complex vector to a list.
 * Create a new Splus complex vector object and put it
 * into an existing Splus list.  The vector is created using
 * \Ref{S_create_complex}, and inserted using macros in S.h.  The
 * list must already be long enough to hold the new element.
 * Fails if the list object type is incorrect, or if pointers are NULL.
 *
 * Declaration included in S\_obj.h.
 *
 * @author Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved.
 * @usage  #ok = S_set_list_complex(&listobj, elem, len, &data);#
 * @return 1/0 for success/failure.
 * @param lst      Splus list object to add to.
 * @param ele      Index of the element to put the vector in.
 * @param len      Length of the desired vector.
 * @param data_ptr Pointer to the data array in created vector.
 * @see S_create_list
 * @see S_create_complex
 * @see S_set_list_integer
 * @see S_set_list_logical
 * @see S_set_list_complex
 */
LibExtern int S_STDCALL S_set_list_complex( s_object *lst, long ele, long len,
  S_complex **data_ptr );

#endif /* _IN_S_OBJ_H */





