#if !defined(_IM_OBJECTS_H_)
#define _IM_OBJECTS_H_

/* TODO: comment out the line below to undefine _EXPORT_IMCSRC so this header file can be included in other app. 
 Instead define it via compiler flag ie cl.exe /D_EXPORT_IMCSRC ...*/ 
#define _EXPORT_IMCSRC

#include <stdio.h>
#include <stdlib.h>

#undef IMCSRC_EXTERN
#undef IMCSRC_STDCALL
#undef IMCSRC_DLLAPI

#if defined(WIN32) && defined(_EXPORT_IMCSRC)
	#define IMCSRC_EXTERN  __declspec(dllexport)
	#define IMCSRC_STDCALL __stdcall
#elif defined(WIN32)
	#define IMCSRC_EXTERN  __declspec(dllimport)
	#define IMCSRC_STDCALL __stdcall
#else
	#define IMCSRC_EXTERN extern
	#define IMCSRC_STDCALL 
#endif 
#define IMCSRC_DLLAPI( returnType ) IMCSRC_EXTERN returnType IMCSRC_STDCALL

#define IMINER_IMCSRC_VERSION 1.10

/* enavle/disable hash lookup to speed up reading datasets */
#define USE_HASH_LOOKUP

/* used for maximum line length...set to 4000 characters by default */
#define MAX_LINE 8000

#define IMINER_FAIL             0L
#define IMINER_SUCCESS          1L
#define IMINER_BAD_INPUT_ARGS   2L
#define IMINER_OUT_OF_MEMORY    3L

/* mode for IMinerVector */
#define IMINER_MODE_UNKNOWN    -1L
#define IMINER_MODE_NULL        0L   
#define IMINER_MODE_BOOLEAN     1L
#define IMINER_MODE_INT         2L
#define IMINER_MODE_LONG        IMINER_MODE_INT
#define IMINER_MODE_FLOAT       3L
#define IMINER_MODE_DOUBLE      4L
#define IMINER_MODE_STRING      5L
#define IMINER_MODE_VECTOR_MAX  IMINER_MODE_STRING
#define IMINER_MODE_LIST        21L

/* mode for Factor */
#define IMINER_MODE_FACTOR 61L
/* mode for DoubleMatrix */
#define IMINER_MODE_DOUBLEMATRIX    1004L
/* mode for ModelMatrix */
#define IMINER_MODE_MODELMATRIX     1005L
/* mode for DataSet */
#define IMINER_MODE_DATASET         1006L
/* mode for a model */
#define IMINER_MODE_MODEL           2001L

#define IMINER_MAX_STRING_LEN 255L
#define IMINER_MAX_STRING_LEN_PLUS_1 256L

/* class names */
#define IMINER_CLASS_VECTOR        "Vector"
#define IMINER_CLASS_LIST          "List"
#define IMINER_CLASS_FACTOR        "Factor"
#define IMINER_CLASS_COLUMN        "Column"
#define IMINER_CLASS_MATRIX        "Matrix"
#define IMINER_CLASS_MODELMATRIX   "ModelMatrix"
#define IMINER_CLASS_DATASET       "DataSet"
#define IMINER_CLASS_UNKOWN        "Unknwon"

/* a factor has two members */
#define IMINER_FACTOR_DATA_MEMBER         0 /* data member,  vector class of mode INT  */
#define IMINER_FACTOR_LEVEL_MEMBER        1 /* level member, vector class of mode STRING */

/* a matrix has four members */
#define IMINER_MATRIX_DATA_MEMBER         0 /* data member,  vector class of any mode */
#define IMINER_MATRIX_DIM_MEMBER          1 /* dim member,   vector class of INT mode */
#define IMINER_MATRIX_ROWNAMES_MEMBER     2 /* row names member,  vector class of STRING mode */
#define IMINER_MATRIX_COLUMNNAMES_MEMBER  3 /* column names member, vector class of STRING mode */

/* a DataSet has four members */
#define IMINER_DATASET_DATA_MEMBER        0 /* data member, vector class of list mode: its length == number of columns*/
#define IMINER_DATASET_DIM_MEMBER         1 /* dim member,  vector class of INT mode */
#define IMINER_DATASET_ROWNAMES_MEMBER    2 /* row names member,  vector class of STRING mode */
#define IMINER_DATASET_COLUMNNAMES_MEMBER 3 /* column names member, vector class of STRING mode */

/* a ModelMatrix class has two members */
#define IMINER_MODELMATRIX_DATA_MEMBER    0 /* data member,  matrix class of DOUBLE mode */
#define IMINER_MODELMATRIX_LEVEL_MEMBER   1 /* dim member,   list class with length==ncolumns & each element is a vector of STRING mode*/

/* macros for mode checking */
#define IMINER_IS_VECTOR_MODE(nMode) (IMINER_MODE_NULL <= (nMode) && (nMode) <= IMINER_MODE_VECTOR_MAX)
#define IMINER_IS_FACTOR_MODE(nMode) ((nMode) == IMINER_MODE_FACTOR)
#define IMINER_IS_MATRIX_MODE(nMode) ((nMode) == IMINER_MODE_DOUBLEMATRIX) /* TODO: non-double */

#define IMINER_IS_VECTOR(pObject) IMINER_IS_VECTOR_MODE((pObject)->m_nMode)
#define IMINER_IS_FACTOR(pObject) IMINER_IS_FACTOR_MODE((pObject)->m_nMode)
#define IMINER_IS_MATRIX(pObject) IMINER_IS_MATRIX_MODE((pObject)->m_nMode)

#define IMINER_OBJECT_MODE(pIMinerObject)   ((IMinerObject*)pIMinerObject)->m_nMode
#define IMINER_OBJECT_LENGTH(pIMinerObject) (pIMinerObject)->m_nLen

/* macro to access a vector element from a vector object*/
#define IMINER_INT_PTR(pIMinerVector) (pIMinerVector)->m_uData.m_pLong
#define IMINER_LONG_PTR(pIMinerVector) (pIMinerVector)->m_uData.m_pLong
#define IMINER_FLOAT_PTR(pIMinerVector) (pIMinerVector)->m_uData.m_pFloat
#define IMINER_DOUBLE_PTR(pIMinerVector) (pIMinerVector)->m_uData.m_pDouble
#define IMINER_STRING_PTR(pIMinerVector) (pIMinerVector)->m_uData.m_ppString
#define IMINER_VOID_PTR(pIMinerVector) (pIMinerVector)->m_uData.m_pVoid
#define IMINER_LIST_PTR(pIMinerObject) (pIMinerObject)->m_uData.m_ppList
#define IMINER_LIST_NAMES_PTR(pIMinerObject) (pIMinerObject)->m_uData.m_ppList[(pIMinerObject)->m_nLen]

#define IMINER_INT_VALUE(pIMinerVector, i) (pIMinerVector)->m_uData.m_pLong[(i)]
#define IMINER_LONG_VALUE(pIMinerVector, i) (pIMinerVector)->m_uData.m_pLong[(i)]
#define IMINER_FLOAT_VALUE(pIMinerVector, i) (pIMinerVector)->m_uData.m_pFloat[(i)]
#define IMINER_DOUBLE_VALUE(pIMinerVector, i) (pIMinerVector)->m_uData.m_pDouble[(i)]
#define IMINER_STRING_VALUE(pIMinerVector, i) (pIMinerVector)->m_uData.m_ppString[(i)]
#define IMINER_VOID_VALUE(pIMinerVector, i) (pIMinerVector)->m_uData.m_pVoid[(i)]
#define IMINER_LIST_VALUE(pIMinerVector, i) (pIMinerVector)->m_uData.m_ppList[(i)]

/* macros to access IMinerFactor object */
#define IMINER_FACTOR_DATA_PTR(pObj)      IMINER_LIST_PTR((pObj))[0]
#define IMINER_FACTOR_LENGTH(pObj)        IMINER_OBJECT_LENGTH(IMINER_LIST_PTR((pObj))[0])
#define IMINER_FACTOR_DATA_VALUE(pObj, i) IMINER_LONG_VALUE(IMINER_LIST_PTR((pObj))[0], (i))

#define IMINER_FACTOR_LEVELS_PTR(pObj)      IMINER_LIST_PTR((pObj))[1]
#define IMINER_FACTOR_NLEVELS(pObj)         IMINER_OBJECT_LENGTH(IMINER_LIST_PTR((pObj))[1])
#define IMINER_FACTOR_LEVELS_VALUE(pObj, i) IMINER_STRING_VALUE(IMINER_LIST_PTR((pObj))[1], (i) )

/* macro to access a matrix element from DoubleMatrix object*/
#define IMINER_MATRIX_DATA_PTR(pObj) (pObj)->m_uData.m_ppList[IMINER_MATRIX_DATA_MEMBER]
#define IMINER_MATRIX_DIM_PTR(pObj) (pObj)->m_uData.m_ppList[IMINER_MATRIX_DIM_MEMBER]
#define IMINER_MATRIX_ROWNAME_PTR(pObj) (pObj)->m_uData.m_ppList[IMINER_MATRIX_ROWNAMES_MEMBER]
#define IMINER_MATRIX_COLUMNNAME_PTR(pObj) (pObj)->m_uData.m_ppList[IMINER_MATRIX_COLUMNNAMES_MEMBER]

#define IMINER_MATRIX_NROWS(pObj) IMINER_LONG_VALUE((pObj)->m_uData.m_ppList[IMINER_MATRIX_DIM_MEMBER], 0)
#define IMINER_MATRIX_NCOLUMNS(pObj) IMINER_LONG_VALUE((pObj)->m_uData.m_ppList[IMINER_MATRIX_DIM_MEMBER], 1)

#define IMINER_MODELMATRIX_LEVEL_PTR(pObj) (pObj)->m_uData.m_ppList[4]

/* macro to access a matrix element from IMinerDoubleMatrix object*/
#define IMINER_Aij(pIMinerMatrix, i, j) IMINER_DOUBLE_VALUE(IMINER_MATRIX_DATA_PTR((pIMinerMatrix)), (i)+(j)*IMINER_MATRIX_NROWS((pIMinerMatrix)) )

/* macro to access a matrix element from a Fortran-liked storage*/
#define Aij(pA, nRows, i, j) pA[(i)+(j)*(nRows)]

/* macros to access IMinerDataSet object */

#define IMINER_DATASET_DATA_PTR(pObj) (pObj)->m_uData.m_ppList[IMINER_DATASET_DATA_MEMBER]
#define IMINER_DATASET_COLUMN_PTR(pObj, j) IMINER_LIST_PTR(IMINER_DATASET_DATA_PTR(pObj))[(j)]
#define IMINER_DATASET_DIM_PTR(pObj) (pObj)->m_uData.m_ppList[IMINER_DATASET_DIM_MEMBER]
#define IMINER_DATASET_ROWNAME_PTR(pObj) (pObj)->m_uData.m_ppList[IMINER_DATASET_ROWNAMES_MEMBER]
#define IMINER_DATASET_COLUMNNAME_PTR(pObj) (pObj)->m_uData.m_ppList[IMINER_DATASET_COLUMNNAMES_MEMBER]

#define IMINER_DATASET_NROWS(pObj) IMINER_LONG_VALUE((pObj)->m_uData.m_ppList[IMINER_DATASET_DIM_MEMBER], 0)
#define IMINER_DATASET_NCOLUMNS(pObj) IMINER_LONG_VALUE((pObj)->m_uData.m_ppList[IMINER_DATASET_DIM_MEMBER], 1)

#define IMINER_DATASET_IS_VECTOR(pObj, j) IMINER_IS_VECTOR(IMINER_DATASET_COLUMN_PTR(pObj, (j)))
#define IMINER_DATASET_IS_FACTOR(pObj, j) IMINER_IS_FACTOR(IMINER_DATASET_COLUMN_PTR(pObj, (j)))
#define IMINER_DATASET_VECTOR_PTR(pObj, j) IMINER_DATASET_COLUMN_PTR(pObj, j)
#define IMINER_DATASET_FACTOR_PTR(pObj, j) IMINER_DATASET_COLUMN_PTR(pObj, j)

#define IMinerMetaData IMinerObject

#define IMINER_MD_COLUMN_DEF_NUM			0
#define IMINER_MD_TERM_DEF_NUM				1

#define IMINER_MD_LEVEL_NAMES_NUM			0
#define IMINER_MD_SPLIT_CATEGORY_NUM		1
#define IMINER_MD_NO_SPLIT					0
#define IMINER_MD_INDICATOR					1
#define IMINER_MD_TREATMENT					2

#define IMINER_MD_TERM_NAMES_NUM			0

#define IMINER_MD_PTR(pObj, i)							(pObj)->m_uData.m_ppList[i]

#define IMINER_MD_COLUMN_INFO(md)						IMINER_MD_PTR(md, IMINER_MD_COLUMN_DEF_NUM)
#define IMINER_MD_NUM_COLUMNS(md)						IMINER_OBJECT_LENGTH(IMINER_MD_COLUMN_INFO(md))
#define IMINER_MD_COLUMN_PTR(md, col)					IMINER_MD_PTR(IMINER_MD_COLUMN_INFO(md), col)
#define IMINER_MD_COLUMN_LEVEL_NAMES(md, col)			IMINER_MD_PTR(IMINER_MD_COLUMN_PTR(md, col), IMINER_MD_LEVEL_NAMES_NUM)
#define IMINER_MD_COLUMN_LEVEL_COUNT(md, col)			IMINER_OBJECT_LENGTH(IMINER_MD_COLUMN_LEVEL_NAMES(md, col))
#define IMINER_MD_COLUMN_LEVEL_NAME_STR(md, col, lev)	IMINER_STRING_VALUE(IMINER_MD_COLUMN_LEVEL_NAMES(md, col), lev)
#define IMINER_MD_COLUMN_SPLIT_CATEGORIES(md, col)		IMINER_MD_PTR(IMINER_MD_COLUMN_PTR(md, col), IMINER_MD_SPLIT_CATEGORY_NUM)
#define IMINER_MD_COLUMN_NUM_SPLIT_CATEGORIES(md, col)	IMINER_INT_VALUE(IMINER_MD_PTR(IMINER_MD_COLUMN_PTR(md, col), IMINER_MD_SPLIT_CATEGORY_NUM), 0)
#define IMINER_MD_GET_COLUMN_NAME(md, col)				IMINER_STRING_VALUE(IMINER_LIST_NAMES_PTR(IMINER_MD_COLUMN_INFO(md)), col)

#define IMINER_MD_TERM_INFO(md)							IMINER_MD_PTR(md, IMINER_MD_TERM_DEF_NUM)
#define IMINER_MD_NUM_TERMS(md)							IMINER_OBJECT_LENGTH(IMINER_MD_TERM_INFO(md))
#define IMINER_MD_TERM_PTR(md, term)					IMINER_MD_PTR(IMINER_MD_TERM_INFO(md), term)
#define IMINER_MD_NUM_TERM_NAMES(md, term)				IMINER_OBJECT_LENGTH(IMINER_MD_TERM_PTR(md, term))
#define IMINER_MD_GET_TERM_NAME(md, term, num)			IMINER_STRING_VALUE(IMINER_MD_TERM_PTR(md, term), num)


#ifdef __cplusplus
extern "C" {
#endif

/* support IMinerObject */
typedef union IMinerDataPointerUnion {
	void*   m_pVoid;
	long*   m_pLong;
  float*  m_pFloat;
  double* m_pDouble;
	/* TODO char*   m_pChar; */
  char**  m_ppString;
	struct IMinerObject** m_ppList;
} IMinerDataPointer;

/* object */
typedef	struct IMinerObject 
{
	long m_nMode;                     /* data type between IMINER_MODE_NULL to IMINER_MODE_VECTOR_MAX */
	long m_nLen;		                  /* vector length */
	IMinerDataPointer m_uData;        /* array of pointers to vector data */
	char m_szClass[64];               /* class name */
} IMinerObject;

#define IMinerVector IMinerObject
#define IMinerList IMinerObject
#define IMinerFactor IMinerObject
#define IMinerDoubleMatrix IMinerObject 
#define IMinerColumn IMinerObject 
#define IMinerDataSet IMinerObject

/************************************** IMinerObject ******************************************/

/* return 1L if true */
IMCSRC_DLLAPI(long) IMinerObject_isEmpty(const IMinerObject* pVector);

/* return 1L if valid */
IMCSRC_DLLAPI(long) IMinerObject_isValid(const IMinerObject* pObject /* in: object */);

/* alloc memory for type IMinerVector. */
IMCSRC_DLLAPI(long) IMinerObject_alloc(IMinerObject* pObject,  /* out: data object */
 long nLen,          /* in: length */
 long nMode         /* in: mode of the data */
);

/* destroy and free memory */
IMCSRC_DLLAPI(long) IMinerObject_destroy(IMinerObject* pObject  /* in/out: data object */);

/* Copy content from another IMinerVector object. */
IMCSRC_DLLAPI(long) IMinerObject_clone(IMinerObject* pObjDest,   /* in/out: data object to copy to */
 const IMinerObject* pObjSource  /* in: data object to copy from */
);

/* get mode */
IMCSRC_DLLAPI(long) IMinerObject_getMode(long* pnMode, /*out: mode of the data */
 const IMinerObject* pObject  /* in: data object */												 
);

/* write to an opened file */
IMCSRC_DLLAPI(long) IMinerObject_writeToStream(FILE* pFile, /* in: output file pointer */
 const IMinerObject* pObject /* in: data object */
);

/* create from from an opened file*/
IMCSRC_DLLAPI(long) IMinerObject_createFromStream(IMinerObject* pObject, /* out: data object */
 FILE* pFile/* in: input file pointer */      												
);

/* open, write to a file and close the file */
IMCSRC_DLLAPI(long) IMinerObject_writeToFile(const char* pszFile, /* in: output file name */
 const IMinerObject* pObject /* in: IMiner object */
);

/* open, create from from a file and close the file*/
IMCSRC_DLLAPI(long) IMinerObject_createFromFile(IMinerObject* pObject, /* out: IMiner object */
 const char* pszFile    /* in: file name */													
);

/**************************************  Vector ******************************************/

/* return 0L if not valid */
IMCSRC_DLLAPI(long) IMinerVector_isValid(const IMinerObject* pVector);

/* alloc memory for type IMinerVector. */
IMCSRC_DLLAPI(long) IMinerVector_create(IMinerObject* pVector,  /* out: data object */
 long nLen,          /* in: length */
 long nMode,         /* in: mode of the data */
 const void* pvValue /* in: NULL or pointer to content of data to be copied */
);

/* destroy and free memory */
IMCSRC_DLLAPI(long) IMinerVector_destroy(IMinerObject* pVector  /* in/out: data object */);

/* return 1L if true */
IMCSRC_DLLAPI(long) IMinerVector_isBoolean(const IMinerObject* pVector);
IMCSRC_DLLAPI(long) IMinerVector_isLong(const IMinerObject* pVector);
IMCSRC_DLLAPI(long) IMinerVector_isFloat(const IMinerObject* pVector);
IMCSRC_DLLAPI(long) IMinerVector_isDouble(const IMinerObject* pVector);
IMCSRC_DLLAPI(long) IMinerVector_isString(const IMinerObject* pVector);

/* Copy content from another IMinerVector object. */
IMCSRC_DLLAPI(long) IMinerVector_clone(IMinerObject* pVector,   /* in/out: data object to copy to */
 const IMinerObject* pvSource  /* in: data object to copy from */
);

/* add an element to IMinerVector.
*  Warning: the void* pointer must be a correct type
*/
IMCSRC_DLLAPI(long) IMinerVector_add(IMinerObject* pVector,  /* out: data object */
 const void* pElement /* in: a pointer to content of data to be copied.  It must be cast-able to type of pVector*/
);
IMCSRC_DLLAPI(long) IMinerVector_addLong(IMinerObject* pVector,  /* out: data object */
 const long lElement /* in: element to be added */
);
IMCSRC_DLLAPI(long) IMinerVector_addDouble(IMinerObject* pVector,  /* in/out: data object */
 const double dElement /* in: element to be added */
);
IMCSRC_DLLAPI(long) IMinerVector_addString(IMinerObject* pVector,  /* in/out: data object */
 const char* pszElement /* in: element to be added */
);

/* get a copy of the element at : caller is responsible for destroying the output object*/
IMCSRC_DLLAPI(long)  IMinerVector_getAt(void* pElement, /* out: a pointer to content of data to be copied.  It must be cast-able to type of pVector*/
 const IMinerObject* pVector,  /* in: data object */
 long i                  /* in: zero-index */
);
IMCSRC_DLLAPI(long)  IMinerVector_getLongAt(long* plElement, /* out: pointer to address of result */
 const IMinerObject* pVector,  /* in: data object */
 long i                        /* in: zero-index */
);
IMCSRC_DLLAPI(long)  IMinerVector_getDoubleAt(double* pdElement, /* out: pointer to address of result */
 const IMinerObject* pVector,  /* in: data object */
 long i                        /* in: zero-index */
);
IMCSRC_DLLAPI(long)  IMinerVector_getStringAt(char* pszBuff, /* out: pointer to buffer with length nBufLen */
 size_t nBuffLen,              /* in: length of buffer to recieve content */
 const IMinerObject* pVector,  /* in: data object */
 long i                        /* in: zero-index */
);

/* set element at */
IMCSRC_DLLAPI(long) IMinerVector_setAt(IMinerObject* pVector,  /* out: data object */
 long i,              /* in: zero-index */
 const void* pElement /* in: a pointer to content of data to be copied.  It must be cast-able to type of pVector*/
);
IMCSRC_DLLAPI(long) IMinerVector_setLongAt(IMinerObject* pVector,  /* out: data object */
 long i,              /* in: zero-index */
 long lElement /* in: value to set*/
);
IMCSRC_DLLAPI(long) IMinerVector_setDoubleAt(IMinerObject* pVector,  /* out: data object */
 long i,         /* in: zero-index */
 double dElement /* in: value to set*/
);IMCSRC_DLLAPI(long) IMinerVector_setStringAt(IMinerObject* pVector,  /* out: data object */
 long i,              /* in: zero-index */
 const char* pszElement /* in: value to set*/
);

/* get copy of data as string array: caller must free the pointes */
IMCSRC_DLLAPI(long) IMinerVector_getAsStringAt(char* pszBuff, /* out: pointer to buffer for reciving the data*/
 size_t nBuffLen,             /* in: length of buffer */
 const IMinerObject* pVector, /* in: the vector object */
 long i                       /* in: zero-index */
);
/* print one elem to stdout */
IMCSRC_DLLAPI(long) IMinerVector_printElement(const IMinerObject* pVector,  /* in: data object */
 long i /* zero-based index */
);

/* print one elem to stdout */
IMCSRC_DLLAPI(long) IMinerVector_print(const IMinerObject* pVector  /* in: data object */);

/* write to an opened file */
IMCSRC_DLLAPI(long) IMinerVector_writeToStream(FILE* pFile, /* in: output file pointer */
 const IMinerObject* pVector /* in: data object */
);

/* create from an opened file*/
IMCSRC_DLLAPI(long) IMinerVector_createFromStream(IMinerObject* pVector, /* out: data object */
 FILE* pFile/* in: input file pointer */      												
);

/**************************************  List ******************************************/

/* return 1L if valid */
IMCSRC_DLLAPI(long) IMinerList_isValid(const IMinerObject* pList);

/* alloc memory for class List and its family. */
IMCSRC_DLLAPI(long) IMinerList_create(IMinerObject* pList,  /* out: data object */
 long nLen, /* len of list to be created */
 long nMode /* class ID of any list-liked class*/
);

/* destroy and free memory */
IMCSRC_DLLAPI(long) IMinerList_destroy(IMinerObject* pList  /* in/out: data object */);

/* create a new object and copy content from the source object */
IMCSRC_DLLAPI(long) IMinerList_clone(IMinerObject* pList,   /* in/out: data object to copy to */
 const IMinerObject* pSource  /* in: data object to copy from */
);

/* set names */
IMCSRC_DLLAPI(long) IMinerList_setNamesFromStrings(IMinerObject* pList,   /* in/out: data object */
 const char** ppszNames  /* in: names of the list elements.  length must be the same as pList->m_nLen */
);

/* set names */
IMCSRC_DLLAPI(long) IMinerList_setNames(IMinerObject* pList,   /* in/out: data object */
 const IMinerObject* pvszNames  /* in: names of the list elements.  length must be the same as pList->m_nLen */
);

/* set name at */
IMCSRC_DLLAPI(long) IMinerList_setNameAt(IMinerObject* pList,   /* in/out: data object */
 long i,              /* in: zero-index */
 const char* pszName  /* in: name */
);

/* get a copy of names as a string vector */
IMCSRC_DLLAPI(long) IMinerList_getNames( IMinerObject* pvszNames,  /* out: names of the list elements.  length must be the same as pList->m_nLen */
 const IMinerObject* pList   /* in: data object */
);

/* append an element to the list */
IMCSRC_DLLAPI(long) IMinerList_add(IMinerObject* pList,  /* out: data object */
 const IMinerObject* pElement, /* in: the element object */
 const char* pszName     /* in: element name */
);

/* get a copy of element at */
IMCSRC_DLLAPI(long)  IMinerList_getAt(IMinerObject* pElement, /* out: a pointer to content of data to be copied.  It must be cast-able to type of pVector*/
 const IMinerObject* pList, /* in: data object */
 long i                     /* in: zero-index */
);

/* set element at */
IMCSRC_DLLAPI(long) IMinerList_setAt(IMinerObject* pList,  /* out: data object */
 long i,              /* in: zero-index */
 const IMinerObject* pElement /* in: a pointer to content of data to be copied.  It must be cast-able to type of pVector*/
);

/* print one elem to stdout */
IMCSRC_DLLAPI(long) IMinerList_printElement(const IMinerObject* pList,  /* in: data object */
 long i /* zero-based index */
);

/* print one elem to stdout */
IMCSRC_DLLAPI(long) IMinerList_print(const IMinerObject* pList  /* in: data object */);

/* write to an opened file */
IMCSRC_DLLAPI(long) IMinerList_writeToStream(FILE* pFile, /* in: output file pointer */
 const IMinerObject* pList /* in: data object */
);
/* create from from an opened file*/
IMCSRC_DLLAPI(long) IMinerList_createFromStream(IMinerObject* pList, /* out: data object */
 FILE* pFile/* in: input file pointer */      												
);

/************************************* Factor : List ******************************************/

/* return 0L if not valid */
IMCSRC_DLLAPI(long) IMinerFactor_isValid(const IMinerObject* pFactor);

/* alloc memory for type IMinerFactor. */
IMCSRC_DLLAPI(long) IMinerFactor_create(IMinerObject* pFactor,  /* out: data object */
 long nLen,            /* in: length of pnValue*/
 const long*  pnValue, /* in: value of length nLen */
 long nLevels,         /* in: length of ppcLevel */
 const char** ppcLevel /* in: level */
);

IMCSRC_DLLAPI(long) IMinerFactor_createFromStrings(IMinerObject* pFactor,  /* out: data object */
 long nLen,             /* in: length of ppszValue*/
 const char** ppszValue /* in: data */
);

/* clone an object */
long IMCSRC_STDCALL IMinerFactor_clone(IMinerObject* pFactor,   /* out: data object to copy to */
 const IMinerObject* pfSource  /* in: data object to copy from */
);

/* destroy and free memory */
IMCSRC_DLLAPI(long) IMinerFactor_destroy(IMinerObject* pFactor  /* in/out: data object */);

/* print one elem to stdout */
IMCSRC_DLLAPI(long) IMinerFactor_printElement(const IMinerObject* pFactor,  /* in: data object */
 long i /* zero-based index */
);

/* print one elem to stdout */
IMCSRC_DLLAPI(long) IMinerFactor_print(const IMinerObject* pFactor  /* in: data object */);

/************************************ DoubleMatrix : List ******************************************/

IMCSRC_DLLAPI(long) IMinerDoubleMatrix_isValid(const IMinerDoubleMatrix* pdMatrix);

/* alloc memory for the object. */
IMCSRC_DLLAPI(long) IMinerDoubleMatrix_create(IMinerDoubleMatrix* pdMatrix,  /* out: data object */
 long nRows,                 /* in:  nunber of rows */
 long nColumns,              /* in:  nunber of columns */
 const double* pdValue,      /* in: NULL or pointer to content of data to be copied (column dominant)*/
 const char** ppszRowName,   /* in: row names if not NULL */
 const char** ppszColumnName /* in: column names if not NULL */
);

/* create and copy content from the source */
long IMCSRC_STDCALL IMinerDoubleMatrix_clone(IMinerDoubleMatrix* pObject,   /* out: data object to copy to */
 const IMinerDoubleMatrix* pSource  /* in: data object to copy from */
);

/* alloc memory for the object and copy content from the source*/
IMCSRC_DLLAPI(long) IMinerDoubleMatrix_createFromDataSet(IMinerDoubleMatrix* pdMatrix,  /* out: data object */
 const IMinerObject* pDataSet /* in: source */
);

/* destroy and free memory */
IMCSRC_DLLAPI(long) IMinerDoubleMatrix_destroy(IMinerDoubleMatrix* pdMatrix  /* in/out: data object */);

/* set content */
IMCSRC_DLLAPI(long) IMinerDoubleMatrix_set(IMinerDoubleMatrix* pdMatrix, /* out: data object */
 const double* pdValue        /* in:  data as double of length (nClusters*nColumns); */
);

/* get column names as a new IMinerVector.  You must destroy this object using IMinerVector_destroy() to avoid mem leak*/
IMCSRC_DLLAPI(long) IMinerDoubleMatrix_getColumnNames(IMinerObject* pvszColumnName,  /* out: string vector of the column */
 const IMinerDoubleMatrix* pdMatrix /* in: data object */
);

/* set column names from an IMinerVector object */
IMCSRC_DLLAPI(long) IMinerDoubleMatrix_setColumnNames(IMinerDoubleMatrix* pdMatrix, /* out: data object */ 
 const IMinerObject* pvszColumnName   /* in: string vector of the column */
);

/* set column names from char** */
long IMCSRC_STDCALL IMinerDoubleMatrix_setColumnNamesFromStrings(IMinerDoubleMatrix* pdMatrix, /* out: data object */ 
 const char** ppszColumnName   /* in: string vector of the column */
);

/* get column names as a new IMinerVector.  You must destroy this object using IMinerVector_destroy() to avoid mem leak*/
IMCSRC_DLLAPI(long) IMinerDoubleMatrix_getRowNames(IMinerObject* pvszColumnName,  /* out: names */
 const IMinerDoubleMatrix* pdMatrix /* in: data object */
);

/* set row names from an IMinerVector object */
IMCSRC_DLLAPI(long) IMinerDoubleMatrix_setRowNames(IMinerDoubleMatrix* pdMatrix, /* out: data object */ 
 const IMinerObject* pvszColumnName   /* in: string vector of the column */
);

/* set row names from char ***/
long IMCSRC_STDCALL IMinerDoubleMatrix_setRowNamesFromStrings(IMinerDoubleMatrix* pdMatrix, /* out: data object */ 
 const char** ppszRowName   /* in: string vector of the column */
);

/* get number of rows */
IMCSRC_DLLAPI(long) IMinerDoubleMatrix_getNRows(long* pnNumberOfRows, /* out: number of rows */
 const IMinerDoubleMatrix* pdMatrix /* in: data object */
);

/* get number of columns */
IMCSRC_DLLAPI(long) IMinerDoubleMatrix_getNColumns(long* pnNumberOfColumns, /* out: number of columns */
 const IMinerDoubleMatrix* pdMatrix /* in: data object */
);

/* print to standard out */
IMCSRC_DLLAPI(long) IMinerDoubleMatrix_print(const IMinerDoubleMatrix* pdMatrix );

/******************************** ModelMatrix : DoubleMatrix : List ****************************/

IMCSRC_DLLAPI(long) IMinerModelMatrix_isValid(const IMinerObject* pdMatrix);

/* alloc memory for the object. */
IMCSRC_DLLAPI(long) IMinerModelMatrix_create(IMinerObject* pdMatrix,  /* out: data object */
 long nRows,                 /* in:  nunber of rows */
 long nColumns,              /* in:  nunber of columns */
 const double* pdValue,      /* in: NULL or pointer to content of data to be copied (column dominant)*/
 const char** ppszRowName,   /* in: row names if not NULL */
 const char** ppszColumnName /* in: column names if not NULL */
);

/* create and copy content from the source */
long IMCSRC_STDCALL IMinerModelMatrix_clone(IMinerObject* pObject,   /* out: data object to copy to */
 const IMinerObject* pSource  /* in: data object to copy from */
);

/* alloc memory for the object and copy content from the source*/
IMCSRC_DLLAPI(long) IMinerModelMatrix_createFromDataSet(IMinerObject* pdMatrix,  /* out: data object */
 const IMinerObject* pDataSet /* in: source */
);

/* print to standard out */
IMCSRC_DLLAPI(long) IMinerModelMatrix_print(const IMinerObject* pdMatrix );

/* open, write to a file and close the file */
IMCSRC_DLLAPI(long) IMinerModelMatrix_writeToFile(const char* pszFile, /* output file name */
 const IMinerObject* pdMatrix /* in: data object */
);

/*********************************  IMinerDataSet : List ******************************************/

/* returns 1L if this is a valid object */
IMCSRC_DLLAPI(long) IMinerDataSet_isValid(const IMinerObject* pDataSet);
/* return 1L if the object can be a column of DataSet */
IMCSRC_DLLAPI(long) IMinerColumn_isValid(const IMinerObject* pColumn);

/* alloc memory for type IMinerDataSet. */
long IMCSRC_STDCALL IMinerDataSet_create(IMinerObject* pDataSet,  /* out: data object */
 long nRows,                /* in:  number of rows */
 long nColumns,             /* in:  number of cols */ 
 const long* pnColumnsModes /* in:  types of columns desired.  If NULL, must use IMinerDataSet_setXXX() to set data */
);

/* get modes of the data columns */
IMCSRC_DLLAPI(long) IMinerDataSet_getColumnMode(long* pnMode, /* out: mode(type) of the data */
 const IMinerObject* pDataSet, /* in: data object */
 long j /* in: j-th column */																								
);

/* set column data. */
long IMCSRC_STDCALL IMinerDataSet_setColumnAt(IMinerObject* pDataSet, /* in/out: data object */
 long j,      /* in:  jth column to set the data */
 const IMinerColumn* pColumn   /* in: the column, length of its length must be long enough */
);

/* set column data. Data type (mode) of pVector must match the column type (mode)*/
IMCSRC_DLLAPI(long) IMinerDataSet_setVectorColumnAt(IMinerObject* pDataSet, /* in/out: data object */
 long j,                        /* in:  jth column to set the data */ 
 const IMinerObject* pVector    /* in:  vector of the column, length of pVector must be long enough */
);

/* set column data */
IMCSRC_DLLAPI(long) IMinerDataSet_setFactorColumnAt(IMinerObject* pDataSet, /* in/out: data object */
 long j,                        /* in:  jth column to set the data */ 
 const IMinerObject* pFactor    /* in:  vector of the column, length of pVector must be long enough */
);

/* get column names as a new IMinerVector.  You must destroy this object using IMinerVector_destroy() to avoid mem leak*/
IMCSRC_DLLAPI(long) IMinerDataSet_getColumnNames(IMinerObject* pvszColumnName,  /* out: string vector of the column */
 const IMinerObject* pDataSet /* in: data object */
);
/* set column names from an IMinerVector object */
IMCSRC_DLLAPI(long) IMinerDataSet_setColumnNames(IMinerObject* pDataSet, /* out: data object */ 
 const IMinerObject* pvszColumnName   /* in: string vector of the column */
);
/* set column names from char** */
long IMCSRC_STDCALL IMinerDataSet_setColumnNamesFromStrings(IMinerObject* pDataSet, /* out: data object */ 
 const char** ppszColumnName   /* in: string vector of the column */
);

/* get row names */
IMCSRC_DLLAPI(long) IMinerDataSet_getRowNames(IMinerObject* pvszColumnName,  /* out: names */
 const IMinerObject* pDataSet /* in: data object */
);
/* set row names*/
IMCSRC_DLLAPI(long) IMinerDataSet_setRowNames(IMinerObject* pDataSet, /* out: data object */ 
 const IMinerObject* pvszColumnName   /* in: string vector of the column */
);
long IMCSRC_STDCALL IMinerDataSet_setRowNamesFromStrings(IMinerObject* pDataSet, /* out: data object */ 
 const char** ppszRowName   /* in: string vector of the column */
);

/* print to standard out */
IMCSRC_DLLAPI(long) IMinerDataSet_print(const IMinerObject* pDataSet	);

/* a simple version of import from text files */
IMCSRC_DLLAPI(long) IMinerDataSet_createFromTextFile(IMinerObject* pDataSet, /* out: data object */
 const char* pszFile                      /* in: file name */													
);

#ifdef USE_HASH_LOOKUP
IMCSRC_DLLAPI(long) IMinerDataSet_createWithModel(
							IMinerObject* pModel, /* out: data object */
							IMinerObject* pDataSet, /* out: data object */
							const char* dataFile     /* in: file name */
							);
#endif /*USE_HASH_LOOKUP*/

/* a simple retrieval function that gets the value of a cell */
IMCSRC_DLLAPI(double) IMinerDataSet_getNonStringValue(IMinerDataSet *ds, int col, int row);

IMCSRC_DLLAPI(long) IMinerDataSet_createFromInput(
 IMinerObject *rectDataInput,	/* out: dataset object */
 long numberOfRows,				/* in:  number of rows */
 long numberOfColumns,			/* in:  number of columns */
 long *columnTypes,				/* in:  column types; must either be 
										IMINER_MODE_DOUBLE or IMINER_MODE_STRING */
 char **columnNames,			/* in:  column names */
 void **data					/* in:  data; each column must be either 
										(double*) or (char**) */
);

/**************************************  MetaData ******************************************/

IMCSRC_DLLAPI(long) IMinerMetaData_createInputDescription(
 IMinerObject* metaData,	    /* out: data object */
 long nColumns,					/* in:  number of columns */
 char **colNames,				/* in:  column names of length (nColumns) */
 long *numLevels,				/* in:  number of levels of length (nColumns) */
 long nMaxLevels,				/* in:  max number of levels */
 char ***levelNames				/* in:  level names of length (nColumns*nMaxLevels) */
);

IMCSRC_DLLAPI(long) IMinerMetaData_createTerms(
 IMinerObject* metaData,	    /* out: data object */
 long nColumns,					/* in:  number of columns */
 long nTerms,					/* in:  number of terms */
 char **colNames,				/* in:  column names of length (nColumns) */
 long *numLevels,				/* in:  number of levels of length (nColumns) */
 char **termNames,				/* in:  names of the terms of length (nTerms) */
 long **colUse,					/* in:  matrix of column use per term of length (nColumns * nTerms) */
 long *splitCategories,			/* in:  split categorical column into multiple double columns of 0/1 */
 long nMaxLevels,				/* in:  max number of levels */
 char ***levelNames				/* in:  level names of length (nColumns*nMaxLevels) */
);

IMCSRC_DLLAPI(long) IMinerMetaData_create(
 IMinerObject* metaData,	    /* out: data object */
 long nColumns,					/* in:  number of columns */
 char **colNames,				/* in:  column names of length (nColumns) */
 long *numLevels,				/* in:  number of levels of length (nColumns) */
 long *splitCategories,			/* in:  split categorical column into multiple double columns of 0/1 */
 long nMaxLevels,				/* in:  max number of levels */
 char ***levelNames				/* in:  level names of length (nColumns*nMaxLevels) */
);

IMCSRC_DLLAPI(long) IMinerMetaData_isValid(const IMinerObject* pModel);
IMCSRC_DLLAPI(long) IMinerMetaData_destroy(IMinerObject* pModel);
IMCSRC_DLLAPI(long) IMinerMetaData_print(const IMinerObject* pModel);

IMCSRC_DLLAPI(long) IMinerMetaData_InputConvert(
 IMinerObject* pConvertedInput,		/* out: data object */
 const IMinerObject* modelMetaData,	/* in:  meta data stored in model */
 const IMinerObject* pInput,		/* in:  input data object */
 IMinerObject* inputMetaData	/* in:  input data meta data */
);

IMCSRC_DLLAPI(long) IMinerMetaData_createFromDataSet(
 IMinerObject *createMD,		/* out: metadata */
 const IMinerDataSet * pInput	/* in: input dataset */
);

/**************************************  Misc. ******************************************/

/* open IMCSRC a data file for write*/
IMCSRC_DLLAPI(long) IMinerDataFile_openForWrite(FILE** ppFile, /* out: output file pointer */
 const char* pszFile    /* in: file name */													
);
/* open IMCSRC a data file for read*/
IMCSRC_DLLAPI(long) IMinerDataFile_openForRead(FILE** ppFile, /* out: output file pointer */
 const char* pszFile    /* in: file name */													
);

/* write to an opened file */
IMCSRC_DLLAPI(long) IMinerVersion_writeToStream(FILE* pFile /* in: output file pointer */);

/* read from an opened file and validate it*/
IMCSRC_DLLAPI(long) IMinerVersion_readFromStreamAndValidate(FILE* pFile/* in: input file pointer */);     												

/* write header info to an opened file. */
IMCSRC_DLLAPI(long) IMinerObjectHeader_writeToStream(FILE* pFile, /* in/out: input file pointer */
 const char* pszClass, /* in: object type if not NULL.  If NULL, use result from IMiner_getClass()*/
 long nMode /* in: mode of an IMiner object which is a unique type (ID) */
);

/* read header info from an opened file. */
IMCSRC_DLLAPI(long) IMinerObjectHeader_readFromStream(long* pnMode, /* out: mode of an IMiner object which is a unique type (ID) */
 FILE* pFile /* in: input file pointer */
);

/* return object type as char* given a mode */
IMCSRC_DLLAPI(long) IMiner_getClassFromMode(char* pszBuf, /*in/out: object type */
 size_t nBufLen, /* in: length of the buffer, pszBuf */
 long nMode      /* in: mode of an IMiner object which is a unique type (ID) */
);

/* some simple test */
void IMinerTest(void);

/* print to stderr and dealing with ellipse ... */
IMCSRC_DLLAPI(long) IMiner_error(const char* pszFormat, ...);


/* code used for hashing */
#ifdef USE_HASH_LOOKUP

typedef	struct IMinerHashEntry {
         char *key;						/** key */
         unsigned int length;			/** length of key */
         unsigned int value;			/** value */
         struct IMinerHashEntry *collision;	/** collision entry */
} IMinerHashEntry;
 
typedef	struct IMinerHashTable {
         unsigned int capacity;
         unsigned int used;
         struct IMinerHashEntry **objects;
} IMinerHashTable;

struct IMinerHashTable *IMinerHashTable_createHash();
void IMinerHashTable_freeHash(struct IMinerHashTable *ht);
unsigned int IMinerHashTable_put(struct IMinerHashTable *ht, const char *key, int value);
unsigned int IMinerHashTable_get(struct IMinerHashTable *ht, const char *key);

#endif /*USE_HASH_LOOKUP*/


#ifdef __cplusplus
}
#endif

#endif /* _IM_OBJECTS_H_ */


