/**
 * GraphServlet is a servlet that receives S-PLUS expression
 *   for creating a Graphlet, passes the expression on
 *   to S-PLUS to create the Graphlet, and responds to the
 *   client with an HTML page that loads the new Graphlet.
 *
 * author Gary Nelson, gnelson@insightful.com
 * version 24 April 2001
 */

import java.io.*;
import java.util.*;
import java.rmi.RemoteException;
import javax.servlet.*;         // comes with Tomcat
import javax.servlet.http.*;    // comes with Tomcat
import com.insightful.splus.*;  // comes with S+AS


public class GraphServlet extends HttpServlet
{
	/* directory in which to store Graphlet files */
	protected static String s_strPath;

	/* S-PLUS connection manager for this Web site */
	protected SpConnectionManager m_mgr = null;

	/* Initialize the servlet. */
	public void init()
	throws ServletException
	{
		// Decide where to store the Graphlet files.
		ServletContext ctx = getServletContext();
		s_strPath = ctx.getRealPath("/applets/");
		// Get a connections manager for S+AS 2.0.
		m_mgr = SpConnectionManager.getManager(ctx);
	}

	/* Shut down the servlet. */
	public void destroy() {
		m_mgr.release();
	}

	/* Respond to a GET request. */
	protected void doGet(HttpServletRequest req,
		HttpServletResponse resp)
	throws ServletException, IOException
	{
		// Create a file on the local machine for the Graphlet.
		File file = File.createTempFile("gt_", ".spj",
			new File(s_strPath));
		String strLocalFilename = file.getName();
		file.deleteOnExit();

		// Reconstruct the query string as an S-PLUS expression.
		String strExp = parseQuery(req);

		String strTitle = null;
		SplusDataResult spdr = null;

		boolean bSucceeded = false;
		while (!bSucceeded) { // If the first connection fails, try another.
			SpConnection con = m_mgr.getConnection();
			if (con == null) {
				sendErrorPage(resp, strExp, 
					"No S-PLUS connections available");
				return;
			}

			try {
				// Call Splus; bale out if it returns an error.
				spdr = con.eval(strExp,
					false, true, true, true, false);
				if (spdr.hasError()) {
					m_mgr.releaseConnection(con);
					sendErrorPage(resp, strExp, spdr.getError());
					return;
				}

				// The S-PLUS function GraphletCreate will return a vector
				//   of two strings.  The first is the title for the Web
				//   page; the second is the filename for the Graphlet file
				//   on the server.
				String strRemoteFilename;
				try {
					String[] vstrResults = spdr.getStringData();
					strTitle = vstrResults[0];
					strRemoteFilename = vstrResults[1];
				}
				catch (Exception ex) {
					m_mgr.releaseConnection(con);
					sendErrorPage(resp, strExp, 
						"S-PLUS function GraphletCreate not responding properly");
					return;
				}

				// Transfer the Graphlet file to the local machine.
				try {
					con.getRemoteFile(strRemoteFilename, 
						s_strPath + File.separator + strLocalFilename, true);
				}
				catch (IOException ex) {
					if (ex instanceof RemoteException) {
						throw(ex);
					}
					m_mgr.releaseConnection(con);
					sendErrorPage(resp, strExp, 
						"Error transfering file: " + ex.getMessage());
					return;
				}

				m_mgr.releaseConnection(con);
				bSucceeded = true;
			}
			catch (RemoteException ex) {
				m_mgr.recycleConnection(con);
			}
		}

		// Return a standard HTML file.
		sendGraphlet(resp, strLocalFilename,
			strTitle, spdr.getWarning());
	}

	/* Respond to a POST request. */
	protected void doPost(HttpServletRequest req,
		HttpServletResponse resp)
	throws ServletException, IOException
	{
		doGet(req, resp);
	}

	/* Parse the query string and construct an S-PLUS expression. */
	protected static String parseQuery(HttpServletRequest req)
	{
		// No special treatment is necessary for the "cmd"
		//   parameter; it's passed to GraphletCreate just
		//   like any other.
		String strParams = "";
		Enumeration enumNames = req.getParameterNames();
		while (enumNames.hasMoreElements()) {
			if (strParams.length() > 0) {
				strParams += ',';
			}
			String strName = (String) enumNames.nextElement();
			String strValue = req.getParameter(strName);
			strParams += strName + '=' + strValue;
		}

		return "GraphletCreate(" + strParams + ')';
	}

	/* Send an HTML page with error information. */
	protected void sendErrorPage(HttpServletResponse resp,
		String strExp, String strError)
	throws IOException
	{
		// Using a sting buffer allows us to report
		//   the length of the page.
		StringBuffer buf = new StringBuffer();
		buf.append("<HTML>\n");
		buf.append("<HEAD>\n");
		buf.append("<TITLE>S-PLUS Error</TITLE>\n");
		buf.append("</HEAD>\n");
		buf.append("<BODY>\n");

		buf.append("<H1>S-PLUS Error</H1>\n");

		buf.append("<H2>S-PLUS Expression</H2>\n");
		buf.append("<BLOCKQUOTE><PRE>\n" + strExp +
			"\n</PRE></BLOCKQUOTE>\n");
		buf.append("<H2>S-PLUS Error Message</H2>\n");
		buf.append("<BLOCKQUOTE><PRE>\n" + strError +
			"\n</PRE></BLOCKQUOTE>\n");

		buf.append("<P>\n");
		buf.append("Use your browser's Back button ");
		buf.append("to go back to the last good page.\n");

		buf.append("</BODY>\n");
		buf.append("</HTML>");

		// Pass the response to the client.
		resp.setContentType("text/html");
		resp.setContentLength(buf.length());
		resp.getOutputStream().print(buf.toString());
	}

	/* Send an HTML page to load the Graphlet. */
	protected void sendGraphlet(HttpServletResponse resp,
		String strFilename,
		String strTitle, String[] vstrWarnings)
	throws IOException
	{
		// Using a sting buffer allows us to report
		//   the length of the page.
		StringBuffer buf = new StringBuffer();
		buf.append("<HTML>\n");
		buf.append("<HEAD>\n");
		buf.append("<TITLE>" + strTitle + "</TITLE>\n");
		buf.append("</HEAD>\n");
		buf.append("<BODY>\n");

		buf.append("<APPLET code=\"spjgraph.class\" ");
		buf.append("archive=\"spjgraph.jar\"\n");
		buf.append("   codebase=\"../applets\" " +
			"WIDTH=720 HEIGHT=540>\n");
		buf.append("   <PARAM NAME=spjgraph.filename ");
		buf.append("VALUE=" + strFilename + ">\n");
		buf.append("</APPLET>\n");

		buf.append("</BODY>\n");
		buf.append("</HTML>");

		// Pass the response to the client.
		resp.setContentType("text/html");
		resp.setContentLength(buf.length());
		resp.getOutputStream().print(buf.toString());
	}
}
