/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
*/

// aparray.cxx: implementation of the CSParray class.
//
//////////////////////////////////////////////////////////////////////
#include "speval.h"
#include "sparray.h"
#include "spint.h"
#include "spexcept.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

//Default constructor
CSParray::CSParray()
: CSPstructure()
{
}
//Copy constructor 
CSParray::CSParray(const CSParray& sobject)
: CSPstructure()
{
	Attach(&sobject, TRUE);
}
//Construct from a base class object
CSParray::CSParray(const CSPobject& sObject)
: CSPstructure()
{	
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a valid S-expression
CSParray::CSParray(const char* pszExpression)
: CSPstructure()
{
	Create(pszExpression);
}
//Construct from an S object 
CSParray::CSParray(s_object* ps_object, BOOL bTryToFreeOnDetach)
: CSPstructure()
{
	Attach(ps_object, bTryToFreeOnDetach);
}

//Assigment operator
CSParray& CSParray::operator=(const CSParray& sobject)
{
	Attach(&sobject, TRUE);
	return *this;
}
//Assigment from the base class
CSParray& CSParray::operator=(const CSPobject& sObject)
{
	Attach(&sObject, TRUE);
	return *this;
}
//Assigment operator
CSParray& CSParray::operator=(s_object* ps_object)
{
	Attach(ps_object, FALSE);
	return *this;
}

CSParray::~CSParray() //The base class CSPobject is responsible for release/detact the object
{
}

//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////

//Return TRUE if the object is a valid 'array' class
BOOL CSParray::IsValid(void) const
{
	if(CSPobject::IsValid())
		return IS(GetPtr(), s_array_class);

	return FALSE;
}

// Return pointer to the .Dim slot
CSPinteger CSParray::GetDim(BOOL bValidate) const
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
	long i = get_slot_offset(s_array_class, SPL_DOT_DIM, S_evaluator);
	if ( i < 0 )
		SCONNECT_ThrowException("Object of class array is missing its dimensions slot");
	return CSPinteger((*this)->value.tree[i]);
}

// Return pointer to the .Dimnames slot
CSPlist CSParray::GetDimnames(BOOL bValidate) const
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
	long i = get_slot_offset(s_array_class, SPL_DOT_DIMNAMES, S_evaluator);
	if ( i < 0 )
		SCONNECT_ThrowException("Missing dimnames slot");
	return CSPlist((*this)->value.tree[i]);
}

int CSParray::GetDataMode(BOOL bValidate) const
{
	return GetData(bValidate)->mode;
}

//SetMode() same as mode(obj) <- "mode" from S
void CSParray::SetDataMode(int lSMode)
{
	if(GetData()->mode != lSMode)
	{
		char* pszMode = token_name(lSMode,  S_evaluator);

		//Clone the whole object before modifying it.
		s_object* ps_newObject = Clone();

		::set_mode2(ps_newObject, lSMode,  pszMode, S_evaluator);

		// Re-attach and re-assign if needed
		ReAttachAndAssign(ps_newObject);
	}
}

//////////////////////////////////////////////////////
// Operations
//////////////////////////////////////////////////////

// Proxy implementation

CSParray::CProxy::CProxy(CSParray* psArray, long lZeroBasedIndex):
CSPproxy(psArray, lZeroBasedIndex)
{
}

CSParray::CProxy::CProxy(const CSParray::CProxy& rhs): //copy constructor
CSPproxy(rhs)
{
}

//WARNING: CSParray::operator () supports only 5 dimensions
CSParray::CProxy CSParray::operator () (long lIndex1, long lIndex2, long lIndex3, long lIndex4, long lIndex5)
{
	long lIndexes[] = {lIndex1-1, lIndex2-1 , lIndex3-1 , lIndex4-1 , lIndex5-1};
	CSPinteger siDim = GetDim(FALSE);
	long lDimIndex = siDim.GetLength();
	if(lDimIndex < 1 || lDimIndex > 5)
		SCONNECT_ThrowException("Index out of range for object of class ", s_array_class->name);  

	long lZeroBasedIndex = 0;
	while(--lDimIndex > 0)
	{
		lZeroBasedIndex += lIndexes[lDimIndex];
		lZeroBasedIndex *= INTEGER_DATA(siDim.GetPtr())[lDimIndex-1];
	}
	lZeroBasedIndex += lIndexes[0];

	return CSParray::CProxy(this, lZeroBasedIndex);
}


///////////////////////////// Assignment operators ////////////////////////////

CSParray::CProxy& CSParray::CProxy::operator=(int nRhs)
{
        long lRhs = (long)nRhs;
	SetElementAs((void*)&lRhs, S_MODE_INT);
	return *this;
}

CSParray::CProxy& CSParray::CProxy::operator=(long lRhs)
{
	SetElementAs((void*)&lRhs, S_MODE_INT);
	return *this;
}

CSParray::CProxy& CSParray::CProxy::operator=(float fRhs)
{
	SetElementAs((void*)&fRhs, S_MODE_REAL);
	return *this;
}

CSParray::CProxy& CSParray::CProxy::operator=(double dRhs)
{
	SetElementAs((void*)&dRhs, S_MODE_DOUBLE);
	return *this;
}

CSParray::CProxy& CSParray::CProxy::operator=(const CSPproxy& proxyRhs)
{
	SetElementAs(proxyRhs.GetElementPtr(), proxyRhs.mode(FALSE));
	return *this;
}

void CSParray::CProxy::SetElementAs( void* pRhs, long lMode, BOOL bCopyIfNeeded)
{
	long lReplacedMode = lMode;

	s_object* ps_object = m_pSPobject->GetPtr();
	s_object* ps_data   = DATA_SLOT(ps_object);
	long lLhsMode = ps_data->mode;
	long lRhsMode = lMode;

	if(bCopyIfNeeded)
	{
		CSPevaluator sEvaluator; 

		//Coerce to mode lRhsMode if needed
		if((lRhsMode!=-1) //Do not try to coerce?
		&&(lLhsMode != lRhsMode))
		{
			lReplacedMode = coerce_to(lLhsMode, lRhsMode, S_evaluator);
			if(lReplacedMode != lLhsMode)
				ps_object = coeves(ps_object, lReplacedMode, FALSE, FALSE, &ps_data, S_evaluator); 
		}

		if(::multiple_ref_count(ps_object, S_evaluator) || multiple_ref_count(DATA_SLOT(ps_object), S_evaluator))
			 ps_object = COPY_ALL(ps_object);

		SCONNECT_SetElement(DATA_SLOT(ps_object), m_lIndex, lRhsMode, pRhs);

		if(m_pSPobject->GetPtr() != ps_object)
		{
			if(m_pSPobject->GetObjectName()==NULL) //Was it a non-persistent object?
				ps_object=sEvaluator.CloneIfNeeded(ps_object);
			static_cast<CSParray*>(m_pSPobject)->ReAttachAndAssign(ps_object); 
		}
	}
	else
		SCONNECT_SetElement(DATA_SLOT(ps_object), m_lIndex, lRhsMode, pRhs);
}

void CSParray::Attach(s_object *ps_object, BOOL bTryToFreeOnDetach)
{
	BaseCoerceAttach(ps_object, s_array_class, bTryToFreeOnDetach);
}
