/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001

	speval.h: interface for the CSPevaluator class wrapping S-PLUS object of structure "s_evaluator".
*/

#if !defined(__SCONNECT_SPEVAL_H_INCLUDED__)
#define __SCONNECT_SPEVAL_H_INCLUDED__

#if _MSC_VER > 1000
#pragma once
#endif //_MSC_VER > 1000

#include <setjmp.h>
#include <string.h>
#include "S.h"
#include "sqpe.h"

#include "spdefs.h"
#include "spexcept.h"

#ifdef __cplusplus
extern "C" {
#endif

SCONNECT_DLLAPI(BOOL) SPL_GetWarningList( s_object **warningList, BOOL bResetAfterGet );

SCONNECT_DLLAPI(s_object*) SPL_ParseEval_GetWarnings(const char* pszExpression, s_object **warningList);
SCONNECT_DLLAPI(s_object*) SPL_ParseEval(const char* pszExpression);

//Generic binary element-by-element operation
SCONNECT_DLLAPI(s_object*) SPL_BinaryOp( s_object* ps_e1, s_object* ps_e2, long lBinOp);

//Clone an object (permanent or not) 
SCONNECT_DLLAPI(s_object*) SPL_Clone(s_object *ps_orig_object, BOOL bPermanent);

//Routine to allocate a string in the same frame as s_object passed in (CRD)
SCONNECT_DLLAPI(char *) SPL_AllocateStringInFrame( s_object *ps_object, const char *pszString );

SCONNECT_DLLAPI(s_object*) SPL_GetFramePointer(long lFrame);

//Used by CSPobject::BaseCreate() to eval expressions for initialization
//without the overhead of SPL_ParseEval_GetWarnings - eliminates output
//and warning list return.  
SCONNECT_DLLAPI(s_object *) SPL_DirectParseEval( const char *pszExpression, BOOL bTry=TRUE );

SCONNECT_DLLAPI(long) SPL_GetSearchPathPosition( s_object *ps_object );

//Utilty for extending the lengths of vectors 
SCONNECT_DLLAPI(s_object*) SPL_ExtendVector(s_object* ps_vector, long n, long index);
SCONNECT_DLLAPI(s_object*) SPL_ShrinkVector(s_object* ps_vector, long n, long index);

//Obtain vector's length. If it is not an atomic vector (recursive?), then
//we use the S evaluator to obtain the length
SCONNECT_DLLAPI(long) SPL_GetVectorLength(s_object* ps_object);

//Query whether a pszClass1 extends from pszClass2. 
SCONNECT_DLLAPI(BOOL) SPL_Extends(const char* pszClass1, const char* pszClass2);
//Get all classes that are direct extensions of pszClass
SCONNECT_DLLAPI(s_object*) SPL_ExtendsDirect(const char* pszClass);

//Retieve an attached object by its name or position
SCONNECT_DLLAPI(s_object*) SPL_AttachedObjectFromPos(long lPos, s_db_purposes dbPurpose);
SCONNECT_DLLAPI(s_object*) SPL_AttachedObjectFromName(const char* pszName, s_db_purposes dbPurpose);

//Retieve the search path used by the engine. 
//dbPurpose = REGULAR_DB or META_DB to retieve the names of the
//regular or meta databases, respectively.
SCONNECT_DLLAPI(s_object*) SPL_DatabaseSearch(s_db_purposes dbPurpose);
//Retieve the directory search path used by the engine
SCONNECT_DLLAPI(s_object*) SPL_SearchPaths(void);

//Boot a chapter
SCONNECT_DLLAPI(wchar_t) SPL_BootResore(void);
SCONNECT_DLLAPI(void) SPL_NaSet(void* p, int mode);
SCONNECT_DLLAPI(BOOL) SPL_IsNa(void* p, int mode);

#ifdef __cplusplus
}
#endif

//////////////////////////////////////////////////////
//CSPevaluator is useful for setting allocated frame via its default constructor.
//And the destructor automatically reset it.
//Commonly use in a scope such as 
//	{
//		CSPevaluator s; //set allocated frame to 1 and open top-level-eval if needed.
//    ...
//  } //reset to previously allocated frame.
//
//IMPORTANT NOTE: The constructor and destructor must be inline in order fo longjmp (signaling)
//to be cought and converted to an exception handling. 

class CSPevaluator  
{
//////////////////////////////////////////////////////
//Construction/Destruction
//////////////////////////////////////////////////////
public:
	enum mode {none, topLevelFrame, localFrame};
	//Default constructor: open the top-evaluator if it is not already open.
	//Also set allocation frame to desired frame.  The default allocation frame is lAllocFrame=1.
	//lAllocFrame must be less than or equal to 1.  If it is greater than 1, we set it to 1 so
	//that any created object can live with the life time of this CSPevaluator object. 
	//Note that this constructor should be inline because it call Open().  See comment for Open().
	inline CSPevaluator(enum mode enMode=CSPevaluator::topLevelFrame)
	{	
		Init();
		//Don't want to open evaluator or want top level eval but it already is open?
		if(enMode==CSPevaluator::none || (IsOpen() && enMode==CSPevaluator::topLevelFrame))
			return;
		else
			Open(enMode);
	}
	//Destructor
	SCONNECT_LIB_EXTERN virtual ~CSPevaluator();
//////////////////////////////////////////////////////
//Operations
//////////////////////////////////////////////////////
public:
	//Open the top-level-evaluator.  Note we setjmp() in this member function which (at least for VC++)
	//requires Open() to be an inline function to correctly recieve the longjump from the engine.
	inline BOOL Open(enum mode enMode=CSPevaluator::topLevelFrame)
	{
		//Reset the long jump to here.
#ifdef WIN32
		int nError ;
		if((nError = ::setjmp(GetErrorJmp())!=0))  //errors in the engine?
#else
		int nError = 0 ;
		if(getenv("S_CALL_SETJMP_IN_CSPEVALUATOR_OPEN") && (nError = ::setjmp(GetErrorJmp())!=0))  //errors in the engine?
#endif
		{
			CSPevaluator::eval_close(nError);		//bailout: invalidate local objects and close all eval frames		
			throw CSPengineException();
		}

		if((enMode==CSPevaluator::none)                       //don't want to open the eval?
		|| (enMode==CSPevaluator::topLevelFrame && IsOpen())) //want topLevel and it already open?
			return FALSE;
		
		//Open the top-eval-frame or push a new empty eval frame onto the stack.
		if(enMode==CSPevaluator::topLevelFrame || enMode==CSPevaluator::localFrame) 
			PushFrame();
	
		return TRUE;
	}
	//Close all evaluators 
	SCONNECT_LIB_EXTERN BOOL Close(const int nErrorCode=0);	
  //Return TRUE if currently open
	SCONNECT_LIB_EXTERN BOOL IsOpen(void) const;                 
	//Validate the evaluator
	SCONNECT_LIB_EXTERN BOOL IsValid(void) const; 
	//If not valid evaluator, throw execption
	SCONNECT_LIB_EXTERN void Validate(void) const; 
	//Casting operator to s_object*
	operator s_evaluator*() const 
	{return m_ps_evaluator;}
	//Get pointer to the s_evaluator pointer
	inline s_evaluator* GetPtr(void) const 
	{return m_ps_evaluator;} 
  //Get/set the current eval. frame
	SCONNECT_LIB_EXTERN s_object* GetCurrentEvalFramePtr(void) const;
  SCONNECT_LIB_EXTERN long GetCurrentEvalFrame(void) const;     
	SCONNECT_LIB_EXTERN long SetCurrentEvalFrame(long lFrame);   
	//Push/pop frames. Pair of PushFrame()/PopFrame() must be called before the destructor
	SCONNECT_LIB_EXTERN BOOL PushFrame(s_object* ps_list=NULL);
	SCONNECT_LIB_EXTERN s_object* PopFrame(s_object* ps_object=NULL);
	//Get/set the current allocation frame
  SCONNECT_LIB_EXTERN long GetCurrentAllocFrame(void) const;    
	SCONNECT_LIB_EXTERN long SetCurrentAllocFrame(long lFrame);  
	//Set an element in current eval frame 
	SCONNECT_LIB_EXTERN s_object* SetInFrame(s_object* ps_object, const char* pszName);  
	//Set an element in the target frame, nTargetFrame
	SCONNECT_LIB_EXTERN s_object* SetInFrame(s_object* ps_object, const char* pszName, long nTargetFrame);  
	//Find an element in current eval frame 
	SCONNECT_LIB_EXTERN s_object* FindInFrame(const char* pszName) const;  
	//Find an element in the target frame, nTargetFrame
	SCONNECT_LIB_EXTERN s_object* FindInFrame(const char* pszName, long nTargetFrame) const;  
	//Get the persistant object in the S databases: the first one in the search list will be returned.
	SCONNECT_LIB_EXTERN s_object* Get(const char* pszName, BOOL bLocal=FALSE, BOOL bGetData=TRUE);
	inline s_object* get(const char* pszName, BOOL bLocal=FALSE, BOOL bGetData=TRUE)
	{	return Get(pszName, bLocal, bGetData);}
	//Same as selectMethod<- function(f, sig = character(), coerce = F): pszSig="class1, class2,..."
	//The returned object is 'function' class which can be attached to CSPfunction 
	SCONNECT_LIB_EXTERN s_object* selectMethod(s_object* ps_charName, s_object* ps_charSig=NULL,  BOOL bCoerce=FALSE)const;
	SCONNECT_LIB_EXTERN s_object* selectMethod(const char* pszName, const char* pszSig=NULL,  BOOL bCoerce=FALSE)const;

	//Same as calling getFunction(name, generic = TRUE, "must find" = TRUE, where)
	SCONNECT_LIB_EXTERN s_object* getFunction(s_object* ps_charName, BOOL bGeneric=TRUE, BOOL bMustFind=TRUE, long lWhere=-1) const;
	SCONNECT_LIB_EXTERN s_object* getFunction(const char* pszName, BOOL bGeneric=TRUE, BOOL bMustFind=TRUE, long lWhere=-1) const;

	//Evaluate S expresion/object in the current evaluation frame
	SCONNECT_LIB_EXTERN s_object* Eval(const char* pszExpression, BOOL bTry=TRUE);
	SCONNECT_LIB_EXTERN s_object* Eval(s_object* ps_object);
	inline s_object* eval(const char* pszExpression, BOOL bTry=TRUE){return Eval(pszExpression, bTry);}
	inline s_object* eval(s_object* ps_object){return Eval(ps_object);}

	//Evaluate S expresion/object in a more permanent evaluation frame than the current one
// note: the following method is not yet implemented nor employed except by the associated inline. rwc 9/27/01
//	SCONNECT_LIB_EXTERN s_object* Eval(const char* pszExpression, long lFrame) ;
	SCONNECT_LIB_EXTERN s_object* Eval(s_object* ps_object, long lFrame);
	inline s_object* eval(s_object* ps_object, long lFrame){return Eval(ps_object, lFrame);}
//	inline s_object* eval(const char* pszExpression, long lFrame){return Eval(pszExpression, lFrame);}

	//Clone to the default perm storage frame if needed
	SCONNECT_LIB_EXTERN s_object* CloneIfNeeded(s_object* ps_object) const;
	//Comparing two objects: same as all.equal()
	SCONNECT_LIB_EXTERN BOOL allEqual(s_object* ps_arg1, s_object* ps_arg2) const;
  //Return TRUE if this object will close the top-level-evaluation frame (frame 1)
	SCONNECT_LIB_EXTERN BOOL WillCloseEvaluator(void) const; 

	//Get the entry point of a C/Fortran routine: search based on the S search list
	SCONNECT_LIB_EXTERN void* GetEntry(const char* pszRoutineName /* in: routine name */) const;
	SCONNECT_LIB_EXTERN void* GetEntry(s_object* ps_chRoutineName /* in: routine name as a scalar char obj*/) const;
	//if bRecursive=FALSE free the parent only, else try to free children and leave the parent alone 
	SCONNECT_LIB_EXTERN void TryToFree(s_object* ps_object, BOOL bRecursive=TRUE /*in: */) const;
	//Accessing ref. count
	SCONNECT_LIB_EXTERN void IncrRef(s_object *ps_object, BOOL bRecursive=TRUE) const;
	SCONNECT_LIB_EXTERN void DecrRef(s_object *ps_object, BOOL bRecursive=TRUE, BOOL bTryToFree=FALSE /* in: if not NULL don't free */) const;
	SCONNECT_LIB_EXTERN int GetRefCount(s_object* ps_object) const;//Only the perent is set
	SCONNECT_LIB_EXTERN void SetRefCount(s_object* ps_object, int nCount) const;//Only the perent is set
	SCONNECT_LIB_EXTERN void SetPermRefCount(s_object* ps_object, s_object* ps_frame) const;
	//Create a new header for the specified object in the current eval frame 
	SCONNECT_LIB_EXTERN s_object* CreateNewHeader(const s_object *ps_object) const;
	////Copy the header and the first level of arena: same as the global function copy_lev(obj, level=0,...)
	SCONNECT_LIB_EXTERN s_object* CopyForWrite(const s_object *ps_object) const;
	//Return TRUE if the object has ref. count <0 or > 1
	SCONNECT_LIB_EXTERN BOOL IsSharing(const s_object *ps_object) const;
	//Create a new vector
	SCONNECT_LIB_EXTERN s_object*	alcvec(long lMode, long len) const;
	//Exactly the same as the global function coerce() 
	SCONNECT_LIB_EXTERN s_object*	coevec(s_object* ps_vector, long lMode) const;
	//Similar to the macro AS wrapping the global function as_class() with additional protection
	SCONNECT_LIB_EXTERN s_object* As(s_object* ps_object, s_class* ps_class) const ;
	//remove an object from a data
	SCONNECT_LIB_EXTERN void Remove(const char* pszName, long lDataBase=1) const;
	//free the header.
	SCONNECT_LIB_EXTERN void FreeHeader(s_object* ps_object) const;
	//ReplaceElement of a list or recursive object: decr. ref count the old child and incr. the new one.
	SCONNECT_LIB_EXTERN void ReplaceElement(s_object* ps_parent, long lZeroIndex, s_object* ps_child) const;

	//////////////////////////////////////////////////////
//No copy constructor nor assignment operator for this class: too dangerous!
//////////////////////////////////////////////////////
private:
	//Initialize all members.
	SCONNECT_LIB_EXTERN void Init(void);
	//Copy constructor
	CSPevaluator(const CSPevaluator& sEvaluator);
	//Assignment operators
	CSPevaluator& operator = (const CSPevaluator& sEvaluator);
	//Restricted interfaces to the engine C-codes
	SCONNECT_LIB_EXTERN jmp_buf& GetErrorJmp(void) const;
	SCONNECT_LIB_EXTERN BOOL eval_close(int nError);
//////////////////////////////////////////////////////
//Data Members
//////////////////////////////////////////////////////
private:
	s_evaluator* m_ps_evaluator; //The evaluator
	long m_lPushFrames[64];      //All frames pushed into the evaluator stack by this object.
	long m_lPushedCount;		     //Currently pushed into this evaluator
	long m_lPrevAllocFrame;      //Previous allocation frame
	long m_lPrevEvalFrame;       //Eval frame set by this object
	static long m_lInstanceCount;//Counts of CSPevaluator instances.
};

class CSPsetjmp
{
public:
	inline CSPsetjmp()
	{
		m_prev_jmp_buf_is_valid = 0 ;
		m_ps_evaluator = Sqpe_GetCurrentEvaluator() ;
		(void)memcpy((void*)&m_prev_jmp_buf, (void*)&m_ps_evaluator->_S_error_jmp, sizeof(jmp_buf));
		m_prev_jmp_buf_is_valid = 1 ;
		// SetTheJump(); see comment below about why we don't call it from constructor
	}
	inline ~CSPsetjmp(){
		if (m_prev_jmp_buf_is_valid) {
			(void)memcpy((void*)m_ps_evaluator->_S_error_jmp, (void*)m_prev_jmp_buf, sizeof(jmp_buf)); 
			m_prev_jmp_buf_is_valid = 0;
		}
	}
	// we make this public so user can make CSPsetjmp object
	// outside of try clause, then call SetTheJump inside it.
	// Makeing CSPsetjmp object in try can cause premature
	// call to destructor in longjmp (problem occurs in release
	// builds using VC++ 6.0).
	inline void SetTheJump(){
		int nError;
		if((nError = ::setjmp(m_ps_evaluator->_S_error_jmp))!=0) {
			// We would like to use m_ps_evaluator as 2nd argument
			// to eval_close, but when we get here (via longjmp)
			// 'this' is set to a random address.  Hence we use the
			// global current evaluator, which is fine as long as
			// the engine is single threaded.  WWD.  Oct 2002.
			::eval_close(nError, Sqpe_GetCurrentEvaluator());
			throw CSPengineException();
		}
	}
private:
		jmp_buf m_prev_jmp_buf ;
		int m_prev_jmp_buf_is_valid;
		s_evaluator *m_ps_evaluator;
};

#endif //!defined(__SCONNECT_SPEVAL_H_INCLUDED__)
