/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
*/

#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#ifdef __hppa
#include <stdarg.h>
#endif

#include "S.h"
#include "s_newio.h"
#include "spexcept.h"
#include "speval.h"
#include "sqpe.h"

extern "C" void do_error_message(char *header, char *message, s_object *ent, char *S_fun_name, FILE *file, s_evaluator *S_evaluator);

static const size_t spInvalidMessageId = (size_t) -1;

static const char* spExceptionMessages[] = {
	"Unknown",
	SCONNECT_INVALID_SOBJECT,

	SCONNECT_INVALID_ENGINE_CONNECTION,

	SCONNECT_ENGINE_EXCEPTION,

	SCONNECT_EVALUATOR_CLOSED,

	SCONNECT_EVALUATION_FAILURE,

	SCONNECT_OBJECT_CREATION_FAILURE,

	SCONNECT_INVALID_DATABASE,

	SCONNECT_COERCION_FAILURE,

	SCONNECT_MEM_ALLOCATION_FALURE,

	SCONNECT_INVALID_CLASS,

	NULL};

static char* SCONNECT_sprintf(const char* pszFormat, va_list args)
{
	char* pszResult = NULL;
	va_list args1 ;
#if defined(S_USE_VA_COPY) || defined(LINUX) || defined(SUNOS)
	va_copy(args1, args); /* va_copy() should be in stdarg.h for C99 standard C */
#else
	args1 = args ;
#endif
	try
	{
		int maxLength = 0;
		for (const char* pszBuf = pszFormat; *pszBuf != '\0'; pszBuf = ++pszBuf)
		{
			if (*pszBuf != '%' || *(++pszBuf) == '%')
			{
				++maxLength;
				continue;
			}

			int length = 0;

			int nWidth = 0;
			for (; *pszBuf != '\0'; ++pszBuf)
			{
				if (*pszBuf == '#')
					maxLength += 2;   
				else if (*pszBuf == '*')
					nWidth = va_arg(args, int);
				else if (*pszBuf != '-' && *pszBuf != '+' && *pszBuf != '0' &&	*pszBuf != ' ')
					break;
			}
			if (nWidth == 0)
			{
				nWidth = atoi(pszBuf);
				for (; *pszBuf != '\0' && isdigit(*pszBuf); pszBuf = ++pszBuf)	;
			}

			int nPrec = 0;
			if (*pszBuf == '.')
			{
				++pszBuf;
				if (*pszBuf == '*')
				{
					nPrec = va_arg(args, int);
					++pszBuf;
				}
				else
				{
					nPrec = atoi(pszBuf);
					for (; *pszBuf != '\0' && isdigit(*pszBuf); ++pszBuf);
				}
			}

			switch (*pszBuf)
			{
				case 'c':
					length = 2;
					va_arg(args, int); //char is promoted to int in ...
					break;

				case 's':
				{
					const char* pszArg = va_arg(args, const char*);
					if (pszArg == NULL)
						length = 6;  // "(null)"
					else
					{
						length = strlen(pszArg);
						if ( length < 1 )
							length = 1;
					}
					if ( length < nPrec )
						nPrec = length;
					if ( length < nWidth )
						length = nWidth;
					break;
				}
				case 'd':
				case 'i':
				case 'u':
				case 'x':
				case 'X':
				case 'o':
				{
					int n = (va_arg(args, int)<0?1:0);
					if ( nWidth < nPrec + n )
						n += nWidth + nPrec;
					else 
						n = nWidth + nPrec;

					length = 32;
					if ( length < n ) 
						length = n;
					break;
				}	
				case 'e':
				case 'g':
				case 'G':
				{
					int n = 7 + (va_arg(args, double)<0?1:0);
					if ( nWidth < nPrec + n )
						n += nWidth + nPrec;
					else 
						n = nWidth + nPrec;

					length = 128;
					if ( length < n ) 
						length = n;
					break;
				}	
				case 'f':
				{
					double d = va_arg(args, double);
					char buffer[512];

					if ( nWidth > 0 && nPrec > 0 )
						length = sprintf(buffer, "%*.*f", nWidth, nPrec, d);
					else if ( nWidth > 0 )
						length = sprintf(buffer, "%*f", nWidth, d);
					else if ( nPrec > 0 )
						length = sprintf(buffer, "%.*f", nPrec, d);
					break;
				}	
				case 'p':
				{
					va_arg(args, void*);
					length = 32;
					if ( length < nWidth+nPrec ) 
						length = nWidth + nPrec;
					break;
				}	
				case 'n':
				{
					va_arg(args, int*);
					break;
				}	
				default:
					break;
			}
			maxLength += length;
		}
		pszResult = new char[maxLength+1];
		int n = vsprintf(pszResult, pszFormat, args1);
		if ( n > maxLength )
			n = maxLength;
		pszResult[n] = '\0';
	}
	catch(...)
	{
	}

	va_end(args);
#if defined(S_USE_VA_COPY) || defined(LINUX) || defined(SUNOS)
	va_end(args1);
#endif

	return pszResult;
}

void SCONNECT_ThrowException(const char* pszFormat, ...)
{
	va_list args;
	va_start(args, pszFormat);

	CSPexception spExcept(CSPexception::Error);
	spExcept.SetMessageDirect(SCONNECT_sprintf(pszFormat, args));

	throw spExcept;
}

void SCONNECT_Warning(const char* pszFormat, ...)
{
	va_list args;
	va_start(args, pszFormat);

	char* pszMsg = SCONNECT_sprintf(pszFormat, args);
	if ( !pszMsg )
		return;

	SCONNECT_DoWarning(pszMsg);
	delete pszMsg;
}

void SCONNECT_Print(const char* pszFormat, ...)
{
	va_list args;
	va_start(args, pszFormat);

	char* pszMsg = SCONNECT_sprintf(pszFormat, args);
	if ( !pszMsg )
		return;

	SCONNECT_DoPrint(pszMsg);
	delete pszMsg;
}

void SCONNECT_DoPrint(const char* pszBuf)
{
	S_EVALUATOR
	if ( pszBuf && *pszBuf )
	{
		if ( S_evaluator && S_stdout )
		{
			// Use stdout (newio version) since stderr is sometimes ignored.
			S_newio_fputs((char*)pszBuf, S_stdout);
			ok_flush(S_stdout, S_evaluator);
		}
		else
		{
			S_newio_puts((char*)pszBuf);
//			S_newio_fflush(S_newio_stdout);
		}
	}
}

void SCONNECT_DoWarning(const char* pszMessage)
{
	if ( !pszMessage || !*pszMessage )
		return;

	do_error_message((char*)"Warning", (char*)pszMessage, NULL, NULL, S_stderr, S_evaluator);
}


/********************************************************************
	CSPexception 
		Base class for all SPL exceptions
********************************************************************/

CSPexception::CSPexception() :
				  m_severity(CSPexception::None),
				  m_uiMessage(CSPexception::Unknown),
				  m_pszMessage(NULL)
{
}

CSPexception::CSPexception(CSPexception& except)
				: m_pszMessage(NULL)
{
	m_severity = except.m_severity;
	m_uiMessage = except.m_uiMessage;
	SetMessage(except.m_pszMessage);
}

CSPexception::CSPexception(CSPexception::TSeverity severity) :
				  m_severity(severity),
				  m_uiMessage(CSPexception::Unknown),
				  m_pszMessage(NULL)
{
}

CSPexception::CSPexception(const char* pszMessage, CSPexception::TSeverity severity) :
				  m_severity(severity),
				  m_uiMessage(spInvalidMessageId),
				  m_pszMessage(NULL)
{
	SetMessage(pszMessage);
}

CSPexception::CSPexception(const char* pszMessage, long lAuxValue, TSeverity severity) :
				  m_severity(severity),
				  m_uiMessage(spInvalidMessageId),
				  m_pszMessage(NULL)
{
	SetMessage(pszMessage, lAuxValue);
}

// Prints "message: dput(ps_object)"
CSPexception::CSPexception(const char* pszMessage, s_object* ps_object, TSeverity severity) :
				  m_severity(severity),
				  m_uiMessage(spInvalidMessageId),
				  m_pszMessage(NULL)
{
	SetMessage(pszMessage, ps_object);
}

CSPexception::CSPexception(const char* pszMessage, const char* pszAuxMsg, TSeverity severity) :
				  m_severity(severity),
				  m_uiMessage(spInvalidMessageId),
				  m_pszMessage(NULL)
{
	SetMessage(pszMessage, pszAuxMsg);
}

CSPexception::CSPexception(TSPExceptionID idMessage, TSeverity severity) :
				  m_severity(severity),
				  m_uiMessage(idMessage),
				  m_pszMessage(NULL)
{
}
CSPexception::CSPexception(TSPExceptionID idMessage, const char* pszAuxMsg, TSeverity severity) :
				  m_severity(severity),
				  m_uiMessage(idMessage),
				  m_pszMessage(NULL)
{
	SetMessage(idMessage, pszAuxMsg);
}

CSPexception::~CSPexception()
{
	SetMessage(NULL);
}

wchar_t CSPexception::ValidMessageId()
{
	if ( m_uiMessage == spInvalidMessageId ) 
		return 0;
	if ( m_uiMessage >= sizeof(spExceptionMessages)/sizeof(char*)-1 )
		return 0;
	return 1;
}

size_t CSPexception::GetMsgLen()
{
	size_t len = 0;
	if ( !m_pszMessage || strlen(m_pszMessage) == 0 )
	{
		if ( ValidMessageId() )
			len = strlen(spExceptionMessages[m_uiMessage]);
	}
	else
	{
		len = strlen(m_pszMessage);
	}
	return len;
}

size_t CSPexception::GetMsg(char* pszMessage, size_t nMaxCh)
{
	if ( !pszMessage || nMaxCh<=0 )
		return 0;
	pszMessage[0] = '\0';
	if ( !m_pszMessage || strlen(m_pszMessage) == 0 )
	{
		if ( ValidMessageId() )
		{
			strncpy(pszMessage, spExceptionMessages[m_uiMessage], nMaxCh-1);
			pszMessage[nMaxCh-1] = '\0';
		}
	}
	else if ( strncpy(pszMessage, m_pszMessage, nMaxCh-1) )
	{
		pszMessage[nMaxCh-1] = '\0';
	}
	return strlen(pszMessage);
}

size_t CSPexception::SetMessage(const char* pszMessage)
{
	if ( m_pszMessage )
	{
		delete[] m_pszMessage;
		m_pszMessage = NULL;
	}
	int n;
	if ( !pszMessage || (n=strlen(pszMessage)) == 0 )
		return 0;
	try
	{
		m_pszMessage = new char[n+1];
		strcpy(m_pszMessage, pszMessage);
	}
	catch(...)
	{
		return 0;
	}
	return strlen(m_pszMessage);
}

size_t CSPexception::SetMessage(const char* pszMessage, s_object* ps_object)
{
	if ( m_pszMessage )
	{
		delete[] m_pszMessage;
		m_pszMessage = NULL;
	}
	if ( ps_object && !::SqpeIsValidAddress(ps_object, sizeof(s_object)) )
		ps_object = NULL;

	int n;
	if ( !pszMessage || (n=strlen(pszMessage)) == 0 )
		return 0;
	try
	{
		//Set allocated frame to be m_lGUIFrame and save previous allocated frame
		CSPevaluator s;

		int nStr = 0;
		s_object* ps_deparse = NULL;
		if ( ps_object )
		{
			ps_deparse = deparse(ps_object, NULL, S_evaluator);
			if ( ps_deparse && ::SqpeIsValidAddress(ps_deparse, sizeof(s_object)) &&
				  IS_CHARACTER(ps_deparse) )
			{
				nStr += 2;
				for ( int i=0; i<GET_LENGTH(ps_deparse); i++ )
					nStr += strlen(CHARACTER_POINTER(ps_deparse)[i]) + 2;
			}
			else 
				ps_deparse = NULL;

			n += nStr;
		}
		m_pszMessage = new char[n+1];
		strcpy(m_pszMessage, pszMessage);

		if ( ps_deparse )
		{
			strcat(m_pszMessage, ": ");
			for ( int i=0; i<GET_LENGTH(ps_deparse); i++ )
			{
				strcat(m_pszMessage, CHARACTER_POINTER(ps_deparse)[i]);
				strcat(m_pszMessage, "\r\n");
			}
		}
	}
	catch(...)
	{
		return 0;
	}
	return strlen(m_pszMessage);
}

size_t CSPexception::SetMessage(size_t idMessage, const char* pszAuxMsg)
{
	SetMessage(NULL);
	size_t n;
	if ( !pszAuxMsg || (n=strlen(pszAuxMsg))==0 )
		return 0;

	if ( !ValidMessageId() )
		return SetMessage(pszAuxMsg);
	try
	{
		n += strlen(spExceptionMessages[idMessage])+2; // +2 for " :"
		m_uiMessage = idMessage;	
		m_pszMessage = new char[n+1]; // " +1 for '\0'
		sprintf(m_pszMessage, "%s: %s", spExceptionMessages[idMessage], pszAuxMsg);
	}
	catch(...)
	{
		return 0;
	}
	return n;
}

size_t CSPexception::SetMessage(const char* pszMessage, const char* pszAuxMsg)
{
	SetMessage(NULL);
	size_t n, n1;
	if ( !pszMessage || (n=strlen(pszMessage))==0 )
		return 0;
	 
	if ( !pszAuxMsg || (n1=strlen(pszAuxMsg))==0 )
		return SetMessage(pszMessage);

	try
	{
		n += n1+2;
		m_pszMessage = new char[n+1]; // " +1 for '\0'
		sprintf(m_pszMessage, "%s: %s", pszMessage, pszAuxMsg);
	}
	catch(...)
	{
		return 0;
	}
	return n;
}

size_t CSPexception::SetMessage(const char* pszMessage, long lAuxValue)
{
	SetMessage(NULL);
	size_t n;
	if ( !pszMessage || (n=strlen(pszMessage))==0 )
		return 0;

	try
	{
		n += 22;
		m_pszMessage = new char[n+1]; // " +1 for '\0'
		sprintf(m_pszMessage, "%s: %ld", pszMessage, lAuxValue);
	}
	catch(...)
	{
		return 0;
	}
	return n;
}

size_t CSPexception::SetMessageDirect(char* pszMessage)
{
	SetMessage(NULL);
	if ( !pszMessage || strlen(pszMessage)==0 )
		return 0;

	m_pszMessage = pszMessage;

	return strlen(m_pszMessage);
}

void CSPexception::Print(void)
{
	char* pszMessage = m_pszMessage;
	if ( !pszMessage )
	{
		if ( ValidMessageId() )
			pszMessage = (char*)spExceptionMessages[m_uiMessage];
		else
			pszMessage = (char*)spExceptionMessages[0];
	}
	char* pszFun = cur_fun_name(S_evaluator->_cur_frame, 0, S_evaluator);
	s_object* ps_object = NULL;

	switch (m_severity)
	{
		// Use stderr (newio version) since stderr is sometimes ignored
		case Note:
		{
			do_error_message((char*)"Note", pszMessage, ps_object, pszFun, S_stderr, S_evaluator);
			break;
		}
		case Warning:
		{
			do_error_message((char*)"Warning", pszMessage, ps_object, pszFun, S_stderr, S_evaluator);
			break;
		}
		case Error:
		{
			do_error_message((char*)"Problem", pszMessage, ps_object, pszFun, S_stderr, S_evaluator);
			break;
		}
		case None:
		{
			do_error_message((char*)"", pszMessage, ps_object, pszFun, S_stderr, S_evaluator);
			break;
		}
	}
}




/********************************************************************
	CSPengineException
		Thrown from CSPeval when Connect/C++ catches the engine long jump
********************************************************************/

CSPengineException::CSPengineException()
:CSPexception(CSPexception::EngineLongJump, CSPexception::Error)
{
}

CSPengineException::CSPengineException(const CSPengineException& except)
:CSPexception(CSPexception::EngineLongJump, CSPexception::Error)
{
}
