/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
 
	splang.cxx: implementation of the CSPlanguage class wrapping S-PLUS object of class "language".
*/

#include <string>

#ifdef __hppa
#include <rw/exception.h>
#endif

#include "spnamed.h"
#include "sconnect.h"
#include "S_y_tab.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CSPlanguage::CSPlanguage()
: CSPobject()
{
}

CSPlanguage::~CSPlanguage()
{
}

//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////

//Return TRUE if the object is a valid class
BOOL CSPlanguage::IsValid(void) const
{
	if(CSPobject::IsValid())
	{
		return S_LANGUAGE_TYPE(GetMode(FALSE));
	}
	return FALSE;
}

//////////////////////////////////////////////////////
// Operations
//////////////////////////////////////////////////////

//Eval in lFrame 
CSPobject CSPlanguage::Eval(long lFrame) const
{
	if(!IsValid())
		throw SCONNECT_INVALID_CLASS;

	CSPobject sValue;
	try
	{ 
		CSPevaluator sEvaluator; //If frame 1 does not exist, open it.
		sValue = sEvaluator.Eval(&(*this), lFrame);
	}
	catch(...)
	{
		sValue.Attach(NULL);
	}

	if(!sValue.IsValid())
		throw SCONNECT_EVALUATION_FAILURE;

	return sValue;
}

//Eval in a new temporary eval frame
CSPobject CSPlanguage::EvalInLocalFrame(void) const
{
	if(!IsValid())
		throw SCONNECT_INVALID_CLASS;

	CSPobject sValue;
	try
	{ 
		CSPevaluator sEvaluator; //If frame 1 does not exist, open it.
		sEvaluator.PushFrame();
		s_object* ps_value = sEvaluator.Eval(&(*this));
		sValue = sEvaluator.PopFrame(ps_value); //pop current frame and move result to its parent frame.
	}
	catch(...)
	{
		sValue.Attach(NULL);
	}

	if(!sValue.IsValid())
		throw SCONNECT_EVALUATION_FAILURE;

	return sValue;
}

//Same as above but add or replace objects with names into the evaluation frame.
CSPobject CSPlanguage::Eval(const CSPnamed& sArgs) const
{
	if(!IsValid())
		throw SCONNECT_INVALID_CLASS;

	CSPobject sValue;
	try
	{ 
		CSPevaluator sEvaluator; //If frame 1 does not exist, open it.

		TSPnamed<s_object*, S_MODE_LIST> sNamedList(sArgs); //Got to have named object with list elements

		//Add objects into the current evaluation frame.
		CSPcharacter sNames(sNamedList.GetNames(FALSE));
		for(long n=0; n< sArgs.GetLength(FALSE); n++)
		{
			s_object* ps_object = sNamedList.GetAt(n);
			const char* pszName = sNames.GetAt(n);
			sEvaluator.SetInFrame(ps_object, pszName);
		}

		sValue = sEvaluator.Eval(&(*this));
	}
	catch(...)
	{
		sValue.Attach(NULL);
	}

	if(!sValue.IsValid())
		throw SCONNECT_EVALUATION_FAILURE;

	return sValue;	
}

//Eval in lFrame 
CSPobject CSPlanguage::Eval(const CSPnamed& sArgs, long lFrame) const
{
	if(!IsValid())
		throw SCONNECT_INVALID_CLASS;

	CSPobject sValue;
	try
	{ 
		CSPevaluator sEvaluator; //If frame 1 does not exist, open it.

		if(lFrame > sEvaluator.GetCurrentEvalFrame())
			throw SCONNECT_EVALUATION_FAILURE;		

		TSPnamed<s_object*, S_MODE_LIST> sNamedList(sArgs); //Got to have named object with list elements

		//Add objects into the current evaluation frame.
		CSPcharacter sNames(sNamedList.GetNames(FALSE));
		for(long n=0; n< sArgs.GetLength(FALSE); n++)
		{
			s_object* ps_object = sNamedList.GetAt(n);
			const char* pszName = sNames.GetAt(n);
			sEvaluator.SetInFrame(ps_object, pszName);
		}
		sValue.Attach(sEvaluator.Eval(&(*this), lFrame), TRUE);
	}
	catch(...)
	{
		sValue.Attach(NULL);
	}

	if(!sValue.IsValid())
		throw SCONNECT_EVALUATION_FAILURE;

	return sValue;
}

//Eval in a new temporary eval frame
CSPobject CSPlanguage::EvalInLocalFrame(const CSPnamed& sArgs) const
{
	if(!IsValid())
		throw SCONNECT_INVALID_CLASS;

	CSPobject sValue;
	try
	{ 
		CSPevaluator sEvaluator; //If frame 1 does not exist, open it.
		const TSPnamed<s_object*, S_MODE_LIST> slnArgs(sArgs); //Got to have named object with list elements
		const CSPcharacter schNames(slnArgs.GetNames());
		sEvaluator.PushFrame();
		for(long n=0; n< schNames.GetLength(); n++)
		{
			const char* pszName = schNames.GetAt(n);
			if(pszName && *pszName)
				sEvaluator.SetInFrame(const_cast<s_object*>(slnArgs.GetAt(n)), pszName);
		}
		s_object* ps_value = sEvaluator.Eval(&(*this));
		sValue.Attach(sEvaluator.PopFrame(ps_value), TRUE);
	}
	catch(...)
	{
		sValue.Attach(NULL);
	}

	if(!sValue.IsValid())
		throw SCONNECT_EVALUATION_FAILURE;

	return sValue;
}

// Returns a character object containing the deparse
// of the object
CSPobject CSPlanguage::Deparse(BOOL bValidate)
{
   if(bValidate)
      Validate();

   CSPevaluator sEvaluator;

   s_object* ps_arg = s_header_for_object(GetPtr(), NULL, S_evaluator);
   s_copy_if_needed(ps_arg, S_FALSE, S_evaluator);
   ps_arg->name = (char *)make_name("expr", S_evaluator)->text;
   ps_arg = alcf((char*)"substitute", ps_arg, S_evaluator);
   ps_arg->name = (char *)make_name("expr", S_evaluator)->text;
   s_object* ps_deparse = alcf((char*)"deparse", ps_arg, S_evaluator);
   s_object* ps_result =  sEvaluator.Eval(ps_deparse);
   sEvaluator.TryToFree(ps_deparse);

   return CSPobject(sEvaluator.CloneIfNeeded(ps_result), TRUE);
}

