/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
 
	spnamed.h: interface for the CSPnamed wrapping S-PLUS object of class "named".
*/

#if !defined(__SCONNECT_SPNAMED_H_INCLUDED__)
#define __SCONNECT_SPNAMED_H_INCLUDED__

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include "spstruct.h"
#include "spvector.h"
#include "sqpe.h"
#include "eval.h"
#include "spchar.h"

class SCONNECT_LIB_EXTERN CSPnamed : public CSPstructure  
{
//////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////
public:
	//Default constructor
	CSPnamed();
	//Copy constructor: copy appropriate data members
	CSPnamed(const CSPnamed& sObject);
	//Construct from object of a base class
	CSPnamed(const CSPobject& sObject);
	//Construct from a valid S-expression
	explicit CSPnamed(const char* pszExpression);
	//Construct frmo an S object pointer  
	explicit CSPnamed(s_object* ps_object, BOOL bTryToFreeOnDetach=FALSE);
	//Assign from the same class
	CSPnamed& operator=(const CSPnamed& sObject);
	//Assign from the base class
	CSPnamed& operator=(const CSPobject& sObject);
	//Assign from an S object
	CSPnamed& operator=(s_object* ps_object);
	//The destructor
	virtual ~CSPnamed();

	//Construct named object with specified length.
	explicit CSPnamed(long lLength);
	explicit CSPnamed(int  nLength);
//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////
public:
  //Return TRUE if the object is a valid numeric class
	virtual BOOL IsValid(void) const;

	//If not valid object, throw execption
	void Validate(void) const;

	//Attach an S object and coerce to appropriate class that override this method
	virtual void Attach(s_object *ps_object, BOOL bTryToFreeOnDetach=FALSE);

	//Get and set/re-set ength, number of columns
	virtual long GetLength(BOOL bValidate=TRUE) const;
	virtual void SetLength(long lLength, BOOL bValidate=TRUE);

	//Get mode
	virtual int GetDataMode(BOOL bValidate=TRUE) const;

	BOOL SetNames(s_object *names, BOOL bValidate=TRUE);
	s_object* GetNames(BOOL bValidate=TRUE) const;

};

// TEMPLATE CLASS "named" with Required pair of parameters:
// (long, S_MODE_LGL), (long,S_MODE_INT), (float, S_MODE_REAL)
// (double, S_MODE_DOUBLE), (char*, S_MODE_CHAR), (CSPobject, S_MODE_LIST) 

template<class TSP_TYPE, long TSP_MODE>
class TSPnamed : public  CSPnamed, protected TSPmodeVector<TSP_TYPE, TSP_MODE>  
{
public:
	//Default constructor
	TSPnamed(){};
	//Copy constructor, also copy appropriate data member
	TSPnamed(const CSPobject& sObject):CSPnamed()
	{
		Attach(&sObject, TRUE);
	}
	//Construct from a valid S-expression
	explicit TSPnamed(const char* pszExpression):CSPnamed()
	{
		CSPevaluator sEvaluator;
		s_object* ps_value = sEvaluator.Eval(pszExpression);
		Attach(sEvaluator.CloneIfNeeded(ps_value), TRUE);
	}
	//Construct from a valid S-object
	explicit TSPnamed(s_object* ps_object, BOOL bTryToFreeOnDetach=FALSE):CSPnamed()
	{
		Attach(ps_object, bTryToFreeOnDetach);
	}

	//Construct named object with specified length.
	explicit TSPnamed(long lLength)
	{
		CSPevaluator sEvaluator;
		s_object* ps_object= ::new_S_object(s_named_class, ::Sqpe_GetCurrentEvaluator());
		s_object* ps_data = this->NewMyType(lLength);
		CSPcharacter schNames(lLength); 
		LIST_POINTER(ps_object)[0] = ps_data;
		LIST_POINTER(ps_object)[1] = schNames.Detach();
		Attach(ps_object, FALSE);
	}

	//The assignment operators
	TSPnamed& operator=(const CSPobject& sObject)
	{
		Attach(&sObject, TRUE);
		return *this;
	}
	TSPnamed& operator=(const TSPnamed<TSP_TYPE, TSP_MODE>& sObject)
	{
		Attach(&sObject, TRUE);
		return *this;
	}
	TSPnamed& operator=(s_object* ps_object)
	{
		Attach(ps_object, FALSE);
		return *this;
	}
	virtual ~TSPnamed()
	{
	}

//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////
public:
  //Return TRUE if the object is a valid numeric class
	BOOL IsValid(void) const
	{
		return (CSPnamed::IsValid() && IsMyMode(CSPnamed::GetData(FALSE)));
	}
	void Validate(void) const
	{
		if (!IsValid())
			throw SCONNECT_INVALID_SOBJECT;
	}
	//Attach an S object and coerce to appropriate class that override this method
	virtual void Attach(s_object *ps_object, BOOL bTryToFreeOnDetach=FALSE)
	{
		if(SPL_NotThere(ps_object))
			CSPobject::Attach(NULL); 
		CSPnamed sNamed(ps_object);
		try
		{
			CSPevaluator sEvaluator;
			if(!this->IsMyMode(sNamed.GetData(FALSE)))
			{
				BaseCoerceAttach(ps_object, s_named_class, bTryToFreeOnDetach);
				return;
			}
		}
		catch(...)
		{
			sNamed.Attach(NULL);
		}

		CSPobject::Attach(sNamed.Detach(), bTryToFreeOnDetach); 
	}

	//Return the address to the content of a cell (element), zero-based index.
	const TSP_TYPE GetAt(long lZeroIndex) const
	{
		return *MyTypePointer(DATA_SLOT(GetPtr()), lZeroIndex);		
	}
	TSP_TYPE& GetAt(long lZeroIndex)
	{
		return *MyTypePointer(DATA_SLOT(GetPtr()), lZeroIndex);		
	}
	//Return the address to the content of a cell (element), zero-based index.
	const TSP_TYPE* GetElementPtr(long lZeroIndex) const
	{
		return MyTypePointer(DATA_SLOT(GetPtr()), lZeroIndex);		
	}
	TSP_TYPE* GetElementPtr(long lZeroIndex)
	{
		return MyTypePointer(DATA_SLOT(GetPtr()), lZeroIndex);		
	}
	//Set directly at the location without calling any synchonization code
	void SetAtDirect(long lZeroIndex, TSP_TYPE element)
	{
		if(TSP_MODE==S_MODE_CHAR)
		{
			CSPcharacter schData(DATA_SLOT(GetPtr()));
			schData.SetAtDirect(lZeroIndex, (const char*)element);
			DATA_SLOT(GetPtr()) = schData.Detach();
		}
		else						
			*GetElementPtr(lZeroIndex) = element;
	}
	void SetAtDirect(const char* pszName, TSP_TYPE element)
	{
		CSPcharacter sNames(GetNames(FALSE));
		long lZeroIndex = -1;
		if(sNames.IsValid())
			lZeroIndex = sNames.Find(pszName, FALSE);
		if(lZeroIndex < 0)
		{
			lZeroIndex = sNames.Add(pszName, FALSE);			 
			LIST_POINTER(GetPtr())[1] = sNames.Detach();
		}
		if(TSP_MODE==S_MODE_CHAR)
		{
			CSPcharacter schData(DATA_SLOT(GetPtr()));
			schData.SetAtDirect(lZeroIndex, (const char*)element);
			DATA_SLOT(GetPtr()) = schData.Detach();
		}
		else						
			*GetElementPtr(lZeroIndex) = element;
	}
	//Set element at lZeroIndex
	void SetAt(long lZeroIndex, TSP_TYPE element)
	{
		CSPevaluator sEvaluator;
		s_object* ps_object= sEvaluator.CopyForWrite(GetPtr()); //new header or copy first level
		//Fixup the data part
		s_object* ps_data = sEvaluator.CopyForWrite(DATA_SLOT(ps_object)); //new header or copy first level
		if(GET_LENGTH(ps_data) < lZeroIndex+1)
			SET_LENGTH(ps_data, lZeroIndex+1);
		switch(TSP_MODE)
		{
		case S_MODE_LIST:
			*this->MyTypePointer(ps_data, lZeroIndex) = sEvaluator.CreateNewHeader(element);
			break;
		case S_MODE_CHAR:
		{
			CSPcharacter schData(DATA_SLOT(GetPtr()));
			schData.SetAtDirect(lZeroIndex, (const char*)element);
			DATA_SLOT(ps_object) = schData.Detach();
			break;
		}
		default:
			*this->MyTypePointer(ps_data, lZeroIndex) = element;
			break;
		}
		DATA_SLOT(ps_object) = ps_data;
		ReAttachAndAssign(ps_object);
	}
	//Same as above, but added here to reduce ambiguility
	void SetAt(int nZeroIndex, TSP_TYPE element)
	{
		SetAt((long) nZeroIndex, element);
	}
	//Set element at pszName
	void SetAt(const char* pszName, TSP_TYPE element, BOOL bValidate=TRUE)
	{
		CSPevaluator sEvaluator;

		if(GetPtr()==NULL) 
		{
			s_object* ps_object= ::new_S_object(s_named_class, ::Sqpe_GetCurrentEvaluator());
			s_object* ps_data = this->NewMyType(1L);
			*this->MyTypePointer(ps_data, 0) = element;		
			CSPcharacter schNames((char**)&pszName, 1L); //TODO: fix CSPcharacter constructor!
			LIST_POINTER(ps_object)[0] = ps_data;
			LIST_POINTER(ps_object)[1] = schNames.Detach();
			ReAttachAndAssign(ps_object);
			return;
		}
		else if(bValidate)
			Validate();
		
		s_object* ps_object= sEvaluator.CopyForWrite(GetPtr()); //new header or copy first level

		//Fixup names and get the numerical index.
		CSPcharacter schNames(LIST_POINTER(ps_object)[1]);
		long lZeroIndex = -1;
		if(schNames.IsValid())
			lZeroIndex = schNames.Find(pszName, FALSE);
		if(lZeroIndex < 0)
			lZeroIndex = schNames.Add(pszName, FALSE);
		//Fixup the data part
		s_object* ps_data = sEvaluator.CopyForWrite(DATA_SLOT(ps_object)); //new header or copy first level
		if(GET_LENGTH(ps_data) < lZeroIndex+1)
			SET_LENGTH(ps_data, lZeroIndex+1);		
		switch(TSP_MODE)
		{
		case S_MODE_LIST:
			*this->MyTypePointer(ps_data, lZeroIndex) = sEvaluator.CreateNewHeader(element);
			break;
		case S_MODE_CHAR:
		{
			CSPcharacter schData(DATA_SLOT(GetPtr()));
			schData.SetAtDirect(lZeroIndex, (const char*)element);
			DATA_SLOT(ps_object) = schData.Detach();
			break;
		}
		default:
			*this->MyTypePointer(ps_data, lZeroIndex) = element;
			break;
		}
		LIST_POINTER(ps_object)[0] = ps_data;
		LIST_POINTER(ps_object)[1] = sEvaluator.CreateNewHeader(schNames.Detach());

		ReAttachAndAssign(ps_object);
	}

	//A vector class to temporary hold the data slot in assisting the (i, j) operator
	class CSPvectorData : public TSPvector<TSP_TYPE, TSP_MODE>
	{
	public:
		//Default counstructor
		CSPvectorData():m_pnamedParent(NULL)
		{};
		//Copy constructors 
		CSPvectorData(TSPnamed<TSP_TYPE, TSP_MODE>* pnamedParent): 
		m_pnamedParent(pnamedParent)
		{};
		virtual ~CSPvectorData()
		{ 
			this->Detach(); 
			this->m_pnamedParent = NULL;
		}
		//Only these friends can access private members
		friend class TSPnamed<TSP_TYPE, TSP_MODE>;
	private:
		void SetParent(TSPnamed<TSP_TYPE, TSP_MODE>* pnamedParent)
		{
			m_pnamedParent = pnamedParent;
			Attach( DATA_SLOT(&(*pnamedParent)));  //no checking assume always valid
		}
		virtual void SetAt(long lZeroIndex, const TSP_TYPE rhs /* TSP_TYPE& rhs?*/, BOOL bValidate=TRUE)
		{
			CSPevaluator sEvaluator;
			s_object* ps_data = &(*this);
			if(bValidate)
			{
				this->Validate();
				if(lZeroIndex < 0L )
					SCONNECT_ThrowException("Index is out of range: %d", lZeroIndex);
				else if(this->GetLength(TRUE) <= lZeroIndex)
					SET_LENGTH(ps_data, lZeroIndex+1L);
			}			

			if(sEvaluator.IsSharing(ps_data))
				ps_data = sEvaluator.CopyForWrite(ps_data);
			if(TSP_MODE==S_MODE_CHAR)
				S_set_arena_why(ps_data, ATOMIC_WHY, sEvaluator.GetPtr());
			this->MyTypeValue(ps_data, lZeroIndex)= rhs;

			if(&(*this) != ps_data) //new data?
			{
				s_object* ps_new_parent = sEvaluator.CopyForWrite(&(*m_pnamedParent));
				SET_ELEMENT(ps_new_parent, 0L, ps_data);
				//Inefficient but necessary for permanent and persistent (with name) objects
				m_pnamedParent->ReAttachAndAssign(ps_new_parent);
			}
		}
		//named parent of this vector
		TSPnamed<TSP_TYPE, TSP_MODE>* m_pnamedParent; 
	};
	//...continue TSPmatrix

	//Indexing: C style, zero-based (i)
	const TSP_TYPE operator[](int nZeroBasedIndex) const
	{
		return GetAt(nZeroBasedIndex);
	}

        //Warning: l-value can be modified.  For example, A(i,j)= rhs would change the content of A
	//without copy first if needed.
	TSP_TYPE& operator[](int nZeroBasedIndex)
	{
		return GetAt(nZeroBasedIndex);
	}

	const TSP_TYPE operator[](long lZeroBasedIndex) const
	{
		return GetAt(lZeroBasedIndex);
	}

	//Warning: l-value can be modified.  For example, A(i,j)= rhs would change the content of A
	//without copy first if needed.
	TSP_TYPE& operator[](long lZeroBasedIndex)
	{
		return GetAt(lZeroBasedIndex);		
	}

	//Indexing: Fortran style, one-based (i)
	const TSP_TYPE operator()(int nOneBasedIndex) const
	{
		return GetAt(nOneBasedIndex-1);
	}

	//Warning: l-value can be modified.  For example, A(i,j)= rhs would change the content of A
	//without copy first if needed.
	TSP_TYPE& operator()(int nOneBasedIndex)
	{
	        return GetAt(nOneBasedIndex-1);
	}
	const TSP_TYPE operator()(long lOneBasedIndex) const
	{
		return GetAt(lOneBasedIndex-1);
	}

	//Warning: l-value can be modified.  For example, A(i,j)= rhs would change the content of A
	//without copy first if needed.
	TSP_TYPE& operator()(long lOneBasedIndex)
	{
	        return GetAt(lOneBasedIndex-1);
	}


	friend class CSPvectorData;
private:
	CSPvectorData m_ps_vData; //temp variable used for indexing via operator() above
};

//Convenient and reserved class for future develement work
typedef TSPnamed<long,      S_MODE_LGL    > CSPlogicalNamed;
typedef TSPnamed<long,      S_MODE_INT    > CSPintegerNamed;
typedef TSPnamed<float,     S_MODE_REAL   > CSPsingleNamed;
typedef TSPnamed<double,    S_MODE_DOUBLE > CSPnumericNamed;
typedef TSPnamed<char*,     S_MODE_CHAR   > CSPcharacterNamed;
typedef TSPnamed<s_complex, S_MODE_COMPLEX> CSPcomplexNamed;
typedef TSPnamed<s_object*, S_MODE_LIST   > CSPlistNamed;
typedef TSPnamed<char,      S_MODE_RAW    > CSPrawNamed;

#endif // !defined(__SCONNECT_SPNAMED_H_INCLUDED__)
