/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
*/

// spproxy.cpp: implementation of the CSPproxy class.
//
//////////////////////////////////////////////////////////////////////
#ifdef __hppa
#include <stdlib.h>
#endif

#include "assert.h"
#include "spproxy.h"
#include "spvector.h"
#include "string.h"
#include "spalcfrm.h"
#include "S_y_tab.h"

///////////////////////////////////////
//Helper class for CSPproxy

class CSPvectorTemp : public CSPvector  
{
public:
	CSPvectorTemp(s_object* ps_object):CSPvector(ps_object){};
	virtual ~CSPvectorTemp(){};
};
///////////////////////////////////////


CSPproxy::CSPproxy():
m_lIndex(0), m_pSPobject(NULL), m_bTempSPobject(FALSE)
{
}

CSPproxy::CSPproxy(CSPobject* pSPObject, long lZeroBasedIndex):
m_lIndex(lZeroBasedIndex), m_pSPobject(pSPObject), m_bTempSPobject(FALSE)
{
}

CSPproxy::CSPproxy(const CSPproxy& rhs): //copy constructor
m_lIndex(rhs.m_lIndex), m_pSPobject(rhs.m_pSPobject)
{
}

//this will create CSPvectorTemp* and the CSPproxy::~CSPproxy will delete it
CSPproxy::CSPproxy(s_object* ps_object, long lZeroBasedIndex):
m_lIndex(lZeroBasedIndex), m_bTempSPobject(TRUE)
{
	m_pSPobject = (CSPobject*) new CSPvectorTemp(ps_object); 
}


CSPproxy::~CSPproxy()
{
	if(m_bTempSPobject && m_pSPobject)
		delete m_pSPobject;
	m_pSPobject = NULL;
}

/*
const CSPproxy& CSPproxy::operator=(const CSPproxy& rhs) //copy constructor
{
	m_pSPObject= rhs.m_pSPObject;
	m_pSPobject->GetVector()= rhs.m_pSPobject->GetVector();
	m_lIndex= rhs.m_lIndex;
	return *this;
}
*/

//CopyIfNeeded() Copy data if needed. It is used before a replacement
//to protect a shared object.
//Return S_MODE that can be used to assign elements of this object.

long CSPproxy::CopyIfNeeded(long lRhsMode)
{ 
	s_object* ps_object=m_pSPobject->GetPtr();
	long lLhsMode = m_pSPobject->GetVector()->mode;
	long lReplacedMode = lRhsMode;

	try
	{
		CSPallocFrame sAllocFrame; //Set allocated frame to a perm frame if top-eval is not openned.

		//Coerce to mode lRhsMode if needed
		if((lRhsMode!=-1) //Do not try to coerce?
		&&(lLhsMode != lRhsMode))
		{
			lReplacedMode = coerce_to(lLhsMode, lRhsMode, S_evaluator);
			if(lReplacedMode != lLhsMode)
				// this depends on vec modified in place (from precious=FALSE)
				ps_object = coeves(m_pSPobject->GetPtr(), lReplacedMode, FALSE, FALSE, &ps_object, S_evaluator); 
		}

		//Copy if multiple references; if we know that data is a ref
		//   (in a frame), we want to keep the ref. count in vec
		//   consistent, for structures, even though vec->Type != REF_TYPE 
		
		if(ps_object==m_pSPobject->GetPtr())
			ps_object= ::s_header_for_object(m_pSPobject->GetPtr(), &Nframe, S_evaluator);

		::s_copy_if_needed(ps_object, S_FALSE, //don't do ref count, we'll do it below
						S_evaluator);

		//Reset and re-attach if the data pointer change 
		if(m_pSPobject->GetPtr()->value.tree != ps_object->value.tree)
		{
			//TODO: For permanent object, m_pSPobject->Attach() call CSPobject::ObjectMap_Add()
			//And m_pSPobject->Assign() will call it again. Inefficient!

			m_pSPobject->Attach(ps_object); 

			const char* pszObjectName = m_pSPobject->GetObjectName();
			if(pszObjectName && (*pszObjectName))
				m_pSPobject->Assign(pszObjectName);
		}
		else
			::free_header(ps_object, S_evaluator); //free the header because it will not be needed 
	}
	catch(...)
	{
	}
					
	return lReplacedMode;
}

void* CSPproxy::GetElementPtr(void) const
{
	void* pReturn=NULL;

	long lDataMode = m_pSPobject->GetVector()->mode;
	switch(lDataMode) 
	{
		case S_MODE_LGL:
		case S_MODE_INT:
			pReturn = &INTEGER_POINTER(m_pSPobject->GetVector())[m_lIndex];
			break;
		case S_MODE_REAL:
			pReturn = &SINGLE_POINTER(m_pSPobject->GetVector())[m_lIndex];
			break;
		case S_MODE_DOUBLE:
			pReturn = &NUMERIC_POINTER(m_pSPobject->GetVector())[m_lIndex];
			break;
		case S_MODE_CHAR:
			pReturn =	CHARACTER_POINTER(m_pSPobject->GetVector())[m_lIndex]; 
			break;
		case S_MODE_COMPLEX:
			pReturn =	&COMPLEX_POINTER(m_pSPobject->GetVector())[m_lIndex] ;
			break;
		default: 
			return NULL;
	}
	return pReturn;
}

void CSPproxy::SetElementAs( void* pRhs, long lMode, BOOL bCopyIfNeeded)
{
	long lReplacedMode = lMode;
	
	//Limit it to just a vector
	assert(m_pSPobject->Is("vector"));

	s_object* ps_object = m_pSPobject->GetPtr();
	long lLhsMode = ps_object->mode;
	long lRhsMode = lMode;

	if(bCopyIfNeeded)
	{
		CSPevaluator sEvaluator; 

		//Coerce to mode lRhsMode if needed
		if((lRhsMode!=-1) //Do not try to coerce?
		&&(lLhsMode != lRhsMode))
		{
			if ( S_atomic_type(lLhsMode) && S_NOT_RECURSIVE(lLhsMode) )
				lReplacedMode = lLhsMode;
			else 
				lReplacedMode = coerce_to(lLhsMode, lRhsMode, S_evaluator);
			if(lReplacedMode != lLhsMode)
				ps_object = coeves(ps_object, lReplacedMode, FALSE, FALSE, &ps_object, S_evaluator); 
		}

		if(multiple_ref_count(ps_object, S_evaluator))
			ps_object = COPY_ALL(ps_object);

		SCONNECT_SetElement(ps_object, m_lIndex, lRhsMode, pRhs);

		if(m_pSPobject->GetPtr() != ps_object)
		{
			if(m_pSPobject->GetObjectName()==NULL) //Was it a non-persistent object?
				sEvaluator.CloneIfNeeded(ps_object);
			m_pSPobject->ReAttachAndAssign(ps_object); 
		}
	}
	else
		SCONNECT_SetElement(ps_object, m_lIndex, lRhsMode, pRhs);
}

template<class T>
T GetScalarPtrAs(void* pvScalar, long lScalarMode, T dummy) 
{
	switch(lScalarMode) 
	{
		case S_MODE_LGL:
		case S_MODE_INT:
			return (T) *((long*) pvScalar);	
			break;
		case S_MODE_REAL:
			return (T) *((float*) pvScalar);	
			break;
		case S_MODE_DOUBLE:
			return (T) *((double*) pvScalar);	
			break;
		case S_MODE_COMPLEX:
//TODO:
//				
//			s_complex z = COMPLEX_POINTER(m_pSPobject->GetVector())[m_lIndex];		
//			return (T) z;
//			break;
		case S_MODE_CHAR:
//TODO:
//			return (T) CHAR_POINTER(m_pSPobject->GetVector())[m_lIndex]; 
//			break;
		default: 
			return (T) 0;
/*
		value = coevec(ent,DOUBLE,MAYBE,PRECIOUS(ent), S_evaluator);
		if(IS_VOID(value) || (n=value->length) < 1)
			PROBLEM "Can't evaluate as a number (from mode %s)",
			token_name(mode, S_evaluator) ERROR;
		is_na_value = is_na_DOUBLE(value->value.Double);
		return = *(value->value.Double);
*/
	}
	return (T) 0;
}

template<class T>
T SCONNECT_GetElementAs(s_object* ps_object, long lIndex) 
{
	switch(ps_object->mode) 
	{
		case S_MODE_LGL:
		case S_MODE_INT:
			return static_cast<T> (INTEGER_POINTER(ps_object)[lIndex]);	
		case S_MODE_REAL:
			return static_cast<T> (SINGLE_POINTER(ps_object)[lIndex]);	
		case S_MODE_DOUBLE:
			return static_cast<T> (NUMERIC_POINTER(ps_object)[lIndex]);	
		case S_MODE_COMPLEX:		
//			s_complex z = COMPLEX_POINTER(ps_object)[lIndex];		
//			return dynamic_cast<T>(z);
//		case S_MODE_CHAR:
//			return static_cast<T> CHARACTER_POINTER(ps_object)[lIndex]; 
		default: 
//			return static_cast<T> RAW_POINTER(ps_object)[lIndex]; 
			return (T) 0;
	}
}

s_object* SPL_BinaryOp( 
	void* pvLhs, long lLhsMode, //pointer to left-hand-side of op and its mode
	void* pvRhs, long lRhsMode, //pointer to right-hand-side of op and its mode
	long lBinOp,							 //operation: S_BINOP_PLUS, S_BINOP_MINUS, etc...
	BOOL bCopyIfNeeded)				 //TRUE allow copy if needed
{
	s_object* ps_lhs= NULL;

	//Determine what mode to be used for the operation
	long lOperationalMode = coerce_to(lLhsMode, lRhsMode, S_evaluator);

	//Copy if the object is multiply reference before any change to its element via SetElementAs.
	switch(lOperationalMode) 
	{
		case S_MODE_DOUBLE:
		{
			double dLhs = 0.0;
			dLhs = (double)GetScalarPtrAs(pvLhs, lLhsMode, dLhs );
			switch(lBinOp)
			{
				case S_BINOP_PLUS:
					dLhs += (double)GetScalarPtrAs(pvRhs, lRhsMode, dLhs );
					break;
				case S_BINOP_MINUS:
					dLhs -= (double)GetScalarPtrAs(pvRhs, lRhsMode, dLhs );
					break;
				case S_BINOP_MULTIPLY:
					dLhs *= (double)GetScalarPtrAs(pvRhs, lRhsMode, dLhs );
					break;
				case S_BINOP_DIVIDE:
					dLhs /= (double)GetScalarPtrAs(pvRhs, lRhsMode, dLhs );
					break;
				default:
					SCONNECT_ThrowException("Error in SPL_BinaryOp: Unknown operator, %d", lBinOp);
			}
			ps_lhs = 	NEW_NUMERIC(1);
			NUMERIC_POINTER(ps_lhs)[0] = dLhs;
			break;
		}
		case S_MODE_REAL:
		{
			float fLhs = 0.0;
			fLhs = (float)GetScalarPtrAs(pvLhs, lLhsMode, fLhs );
			switch(lBinOp)
			{
				case S_BINOP_PLUS:
					fLhs += (float)GetScalarPtrAs(pvRhs, lRhsMode, fLhs );
					break;
				case S_BINOP_MINUS:
					fLhs -= (float)GetScalarPtrAs(pvRhs, lRhsMode, fLhs );
					break;
				case S_BINOP_MULTIPLY:
					fLhs *= (float)GetScalarPtrAs(pvRhs, lRhsMode, fLhs );
					break;
				case S_BINOP_DIVIDE:
					fLhs /= (float)GetScalarPtrAs(pvRhs, lRhsMode, fLhs );
					break;
				default:
					SCONNECT_ThrowException("Error in SPL_BinaryOp: Unknown operator, %d", lBinOp);
			}
			ps_lhs = 	NEW_SINGLE(1);
			SINGLE_POINTER(ps_lhs)[0] = fLhs;
			break;
		}
		case S_MODE_LGL:
		case S_MODE_INT:
		{
			long lLhs = 0;
			lLhs = (long)GetScalarPtrAs(pvLhs, lLhsMode, lLhs );
			switch(lBinOp)
			{
				case S_BINOP_PLUS:
					lLhs += (long)GetScalarPtrAs(pvRhs, lRhsMode, lLhs );
					break;
				case S_BINOP_MINUS:
					lLhs -= (long)GetScalarPtrAs(pvRhs, lRhsMode, lLhs );
					break;
				case S_BINOP_MULTIPLY:
					lLhs *= (long)GetScalarPtrAs(pvRhs, lRhsMode, lLhs );
					break;
				case S_BINOP_DIVIDE:
					lLhs /= (long)GetScalarPtrAs(pvRhs, lRhsMode, lLhs );
					break;
				default:
					SCONNECT_ThrowException("Error in SPL_BinaryOp: Unknown operator, %d", lBinOp);
			}
			ps_lhs = 	NEW_INTEGER(1);
			INTEGER_POINTER(ps_lhs)[0] = lLhs;
			break;
		}
		case S_MODE_COMPLEX:
	//TODO:
	//			break;
			case S_MODE_CHAR:
	//TODO:
	//			break;
			default: 
					SCONNECT_ThrowException("Error in SPL_BinaryOp: unsupported mode, %d", lOperationalMode);
	}
	return ps_lhs;
}


//Binary operations involving (CSPproxy) op (CSPproxy)
s_object* CSPproxy::BinaryOp( const CSPproxy& proxyRhs, long lBinOp) const
{
	return SPL_BinaryOp(GetElementPtr(), mode(FALSE), 
                      proxyRhs.GetElementPtr(), proxyRhs.mode(FALSE), lBinOp, TRUE);
}

//Binary operations involving (CSPproxy) op (double)  
s_object* CSPproxy::BinaryOp(double dRhs, long lBinOp) const
{
	return SPL_BinaryOp(GetElementPtr(), mode(FALSE), &dRhs, S_MODE_DOUBLE, lBinOp, TRUE);
}
//Binary operations involving (CSPproxy) op (long)  
s_object* CSPproxy::BinaryOp(long lRhs, long lBinOp) const
{
	return SPL_BinaryOp(GetElementPtr(), mode(FALSE), &lRhs, S_MODE_INT, lBinOp, TRUE);
}
//Binary operations involving (CSPproxy) op (float)  
s_object* CSPproxy::BinaryOp(float fRhs, long lBinOp) const
{
	return SPL_BinaryOp(GetElementPtr(), mode(FALSE), &fRhs, S_MODE_REAL, lBinOp, TRUE);
}

//Binary operations involving (double) op (CSPproxy)
s_object* SPL_BinaryOp(double dLhs, const CSPproxy& proxyRhs,  long lBinOp) 
{
	return SPL_BinaryOp(&dLhs, S_MODE_DOUBLE,
	                    proxyRhs.GetElementPtr(), proxyRhs.mode(FALSE), lBinOp, TRUE);
}
//Binary operations involving (double) op (CSPproxy)
s_object* SPL_BinaryOp(long lLhs, const CSPproxy& proxyRhs,  long lBinOp) 
{
	return SPL_BinaryOp(&lLhs, S_MODE_INT,
	                    proxyRhs.GetElementPtr(), proxyRhs.mode(FALSE), lBinOp, TRUE);
}
//Binary operations involving (double) op (CSPproxy)
s_object* SPL_BinaryOp(float fLhs, const CSPproxy& proxyRhs,  long lBinOp) 
{
	return SPL_BinaryOp(&fLhs, S_MODE_REAL,
	                    proxyRhs.GetElementPtr(), proxyRhs.mode(FALSE), lBinOp, TRUE);
}

/////////////////////////////////////////////////////////////
//Assigment operators

CSPproxy& CSPproxy::operator=(const CSPproxy& proxyRhs)
{
	try
	{
		void* pRhs=proxyRhs.GetElementPtr();
		long lDataMode = proxyRhs.m_pSPobject->GetVector()->mode;
		switch(lDataMode) 
		{
			case S_MODE_LGL:
			case S_MODE_INT:
				SetElementAs(pRhs, S_MODE_INT);
				break;
			case S_MODE_REAL:
				SetElementAs(pRhs, S_MODE_REAL);
				break;
			case S_MODE_DOUBLE:
				SetElementAs(pRhs, S_MODE_DOUBLE);
				break;
			case S_MODE_CHAR:
				SetElementAs(pRhs, S_MODE_CHAR);
				break;
			case S_MODE_COMPLEX:
	//TODO:
			default: 
				SCONNECT_ThrowException("Error in CSPproxy::operator=(): unsupported mode, %d", lDataMode);
		}
	}
	catch(...)
	{
		throw; //re-throw
	}
	return *this;
}

/////////////////////////////////////////////////////////////
//Boolean operators

BOOL CSPproxy::equal(double dRhs) const 
{
	if(canbeDouble())
	{ 
		double dLhs = SCONNECT_GetElementAs<double>(m_pSPobject->GetVector(), m_lIndex);
		return dLhs == dRhs ;
	}
	else
		return FALSE;
}

BOOL CSPproxy::lessThan(double dRhs) const 
{
	if(canbeDouble())
	{ 
		double dLhs = SCONNECT_GetElementAs<double>(m_pSPobject->GetVector(), m_lIndex);
		return dLhs < dRhs ;
	}
	else
		return FALSE;
}

BOOL CSPproxy::lessThanEqual(double dRhs) const 
{
	if(canbeDouble())
	{ 
		double dLhs = SCONNECT_GetElementAs<double>(m_pSPobject->GetVector(), m_lIndex);
		return dLhs <= dRhs ;
	}
	else
		return FALSE;
}

BOOL CSPproxy::greaterThan(double dRhs) const 
{
	if(canbeDouble())
	{ 
		double dLhs = SCONNECT_GetElementAs<double>(m_pSPobject->GetVector(), m_lIndex);
		return dLhs > dRhs ;
	}
	else
		return FALSE;
}
BOOL CSPproxy::greaterThanEqual(double dRhs) const 
{
	if(canbeDouble())
	{ 
		double dLhs = SCONNECT_GetElementAs<double>(m_pSPobject->GetVector(), m_lIndex);
		return dLhs >= dRhs ;
	}
	else
		return FALSE;
}

BOOL CSPproxy::operator==(const char* pszRhs) const 
{
	if(mode() != S_MODE_CHAR)
		return FALSE;
	size_t len = strlen(pszRhs);
	char* pszLhs = (char*)(*this);
	if(pszLhs == pszRhs)
		return TRUE;
	if(strlen(pszLhs) != len)
		return FALSE;
	else
		return !strncmp(pszLhs, pszRhs, len);
}

BOOL CSPproxy::operator==(const CSPproxy& proxyRhs) const
{
	try
	{
		// unused : void* pRhs=proxyRhs.GetElementPtr();
		long lDataMode = proxyRhs.m_pSPobject->GetVector()->mode;
		if(lDataMode != this->m_pSPobject->GetVector()->mode)
			return FALSE;

		switch(lDataMode) 
		{
			case S_MODE_LGL:
			case S_MODE_INT:
				return	SCONNECT_GetElementAs<long>(m_pSPobject->GetVector(), m_lIndex) == 
								SCONNECT_GetElementAs<long>(proxyRhs.m_pSPobject->GetVector(), proxyRhs.m_lIndex);
				break;
			case S_MODE_REAL:
				return	SCONNECT_GetElementAs<float>(m_pSPobject->GetVector(), m_lIndex) == 
								SCONNECT_GetElementAs<float>(proxyRhs.m_pSPobject->GetVector(), proxyRhs.m_lIndex);
				break;
			case S_MODE_DOUBLE:
				return	SCONNECT_GetElementAs<double>(m_pSPobject->GetVector(), m_lIndex) == 
								SCONNECT_GetElementAs<double>(proxyRhs.m_pSPobject->GetVector(), proxyRhs.m_lIndex);
				break;
			case S_MODE_CHAR:
			{
				char* pszRhs = (char*) proxyRhs;
				char* pszLhs = (char*)(*this);
				size_t len = strlen(pszRhs);
				if(pszLhs == pszRhs)
					return TRUE;
				else if(strlen(pszLhs) != len)
					return FALSE;
				else
					return !strncmp(pszLhs, pszRhs, len);
				break;
			}
			case S_MODE_COMPLEX:
	//TODO:
			default: 
				SCONNECT_ThrowException("Error in CSPproxy::operator=(): unsupported mode, %d", lDataMode);
		}
	}
	catch(...)
	{
		throw; //re-throw
	}
	return FALSE;
}

BOOL CSPproxy::operator!=(const char* pszRhs) const
{
	if(mode() != S_MODE_CHAR)
		return TRUE;
	size_t len = strlen(pszRhs);
	char* pszLhs = (char*)(*this);
	if(pszLhs == pszRhs)
		return FALSE;
	if(strlen(pszLhs) != len)
		return TRUE;
	else
		return strncmp(pszLhs, pszRhs, len);
}

BOOL CSPproxy::operator!=(const CSPproxy& proxyRhs) const
{
	return !operator==(proxyRhs);
}

//Set directly into a vector's element
s_boolean SCONNECT_SetElement(s_object* ps_object, long nIndex, long lRhsMode, void* pRhs)
{
	long lLhsMode = ps_object->mode;
	switch(lLhsMode) 
	{
		case S_MODE_LGL:
		case S_MODE_INT:
			switch( lRhsMode )
			{
				case S_MODE_LGL:
				case S_MODE_INT:
					INTEGER_POINTER(ps_object)[nIndex] = *((long*) pRhs);
				break;
				case S_MODE_REAL:
					INTEGER_POINTER(ps_object)[nIndex] =  (long)(*((float*) pRhs));
				break;
				case S_MODE_DOUBLE:
					INTEGER_POINTER(ps_object)[nIndex] =  (long)(*((double*) pRhs));
				break;
				case S_MODE_COMPLEX:
					INTEGER_POINTER(ps_object)[nIndex] = (long)(((S_complex*) pRhs)->re);
				break;
				case S_MODE_CHAR:
				{
					char* pszRhs = *(char**) pRhs;
					long lValue = atol( pszRhs );
					INTEGER_POINTER(ps_object)[nIndex] =  lValue;
				}
				break;
			}
			break;
		case S_MODE_REAL:
			switch( lRhsMode )
			{
				case S_MODE_LGL:
				case S_MODE_INT:
					SINGLE_POINTER(ps_object)[nIndex] = (float)(*((long*) pRhs));
				break;
				case S_MODE_REAL:
					SINGLE_POINTER(ps_object)[nIndex] = *((float*) pRhs);
				break;
				case S_MODE_DOUBLE:
					SINGLE_POINTER(ps_object)[nIndex] = (float)(*((double*) pRhs));
				break;
				case S_MODE_COMPLEX:
					SINGLE_POINTER(ps_object)[nIndex] = (float)(((S_complex*) pRhs)->re);
				break;
				case S_MODE_CHAR:
				{
					char* pszRhs = *(char**) pRhs;
					double dValue = atof( pszRhs );
					SINGLE_POINTER(ps_object)[nIndex] =  (float)dValue;
				}
				break;
			}
			break;
		case S_MODE_DOUBLE:
			switch( lRhsMode )
			{
				case S_MODE_LGL:
				case S_MODE_INT:
					NUMERIC_POINTER(ps_object)[nIndex] = (double)(*((long*) pRhs));
				break;
				case S_MODE_REAL:
					NUMERIC_POINTER(ps_object)[nIndex] = (double)(*((float*) pRhs));
				break;
				case S_MODE_DOUBLE:
					NUMERIC_POINTER(ps_object)[nIndex] = *((double*) pRhs);
				break;
				case S_MODE_COMPLEX:
					NUMERIC_POINTER(ps_object)[nIndex] = (double)(((S_complex*) pRhs)->re);
				break;
				case S_MODE_CHAR:
				{
					char* pszRhs = *(char**) pRhs;
					double dValue = atof( pszRhs );
					NUMERIC_POINTER(ps_object)[nIndex] =  dValue;
				}
				break;
			}
			break;
		case S_MODE_COMPLEX:
			COMPLEX_POINTER(ps_object)[nIndex].im = 0.0;
			switch( lRhsMode )
			{
				case S_MODE_LGL:
				case S_MODE_INT:
					COMPLEX_POINTER(ps_object)[nIndex].re = (double)(*((long*) pRhs));
				break;
				case S_MODE_REAL:
					COMPLEX_POINTER(ps_object)[nIndex].re = (double)(*((float*) pRhs));
				break;
				case S_MODE_DOUBLE:
					COMPLEX_POINTER(ps_object)[nIndex].re = (double)(*((double*) pRhs));
				break;
				case S_MODE_COMPLEX:
					COMPLEX_POINTER(ps_object)[nIndex] = *((S_complex*) pRhs);
				break;
				case S_MODE_CHAR:
				{
					char* pszRhs = *(char**) pRhs;
					double dValue = atof( pszRhs );
					COMPLEX_POINTER(ps_object)[nIndex].re = dValue;
				}
				break;
			}
			break;
		case S_MODE_CHAR:
		{
			char *pszRhs = NULL;
#ifdef WIN32
			char tmp[_MAX_PATH];
#else
			char tmp[MAX_FILE_NAME_LEN];
#endif
			switch( lRhsMode )
			{
				case S_MODE_LGL:
				case S_MODE_INT:
				{
					sprintf( tmp, "%ld", *((long*) pRhs) );
					pszRhs = tmp;
				}
				break;
				case S_MODE_REAL:
				{
					sprintf( tmp, "%g", *((float*) pRhs) );
					pszRhs = tmp;
				}
				break;
				case S_MODE_DOUBLE:
				{
					sprintf( tmp, "%g", *((double*) pRhs) );
					pszRhs = tmp;
				}
				break;
				case S_MODE_CHAR:
				{
					pszRhs = *(char**) pRhs;
				}
				break;
			}
			if ( !pszRhs )
				SCONNECT_ThrowException("Error in SCONNECT_SetElement() : Invalid char* pszRhs");

			if(S_is_env_removable_char())
			{
				char* psz_old_strings = NULL;
				if(S_get_arena_why(ps_object, S_evaluator) == CHAR_WHY)
					psz_old_strings = CHARACTER_POINTER(ps_object)[0];

				CHARACTER_POINTER(ps_object)[nIndex] = pszRhs; 
				S_copy_strings(ps_object, ps_object, S_evaluator);

				if(psz_old_strings)
					S_remove_strings(psz_old_strings, S_evaluator);
			}
			else
			{
				char *pszLhs = SPL_AllocateStringInFrame( ps_object, pszRhs );
				if ( pszLhs == NULL )
					SCONNECT_ThrowException("Error in SCONNECT_SetElement() : Invalid char* pszRhs");
				CHARACTER_POINTER(ps_object)[nIndex] = pszLhs; //Leak the previous mem.? 
			}
			break;
		} 
		case S_MODE_LIST:
		{
			s_object* ps_rhs = NULL;
			switch( lRhsMode )
			{
			case S_MODE_LGL:
				ps_rhs = NEW_LOGICAL(1L);
				LOGICAL_POINTER(ps_rhs) = (long*) pRhs;
				break;
			case S_MODE_INT:
				ps_rhs = NEW_INTEGER(1L);
				INTEGER_POINTER(ps_rhs) = (long*) pRhs;
				break;
			case S_MODE_REAL:
				ps_rhs = NEW_SINGLE(1L);
				SINGLE_POINTER(ps_rhs) = (float*) pRhs;
				break;
			break;
			case S_MODE_DOUBLE:
				ps_rhs = NEW_NUMERIC(1L);
				NUMERIC_POINTER(ps_rhs) = (double*) pRhs;
				break;
			break;
			case S_MODE_COMPLEX:
				ps_rhs = NEW_COMPLEX(1L);
				COMPLEX_POINTER(ps_rhs) = (s_complex*) pRhs;
				break;
			case S_MODE_CHAR:
				ps_rhs = NEW_CHARACTER(1L);
				CHARACTER_POINTER(ps_rhs)[0] = *(char**)pRhs;
				S_copy_strings(ps_rhs, ps_rhs, S_evaluator);
				break;
			case S_MODE_LIST:
				ps_rhs = *(s_object**)pRhs;
				break;
			default: 
				SCONNECT_ThrowException("Error in SCONNECT_SetElement() : Unsupport lRhsMode=%d", lRhsMode);
			}
			SET_ELEMENT(ps_object, nIndex, ps_rhs);
			break;
		}
		default: 
			SCONNECT_ThrowException("Error in SCONNECT_SetElement() : Unsupport lLhsMode=%d", lLhsMode);
	}
	return S_TRUE;
}


CSPproxy::operator double() const
{
	return SCONNECT_GetElementAs<double>(m_pSPobject->GetVector(), m_lIndex);
}

CSPproxy::operator long() const
{
	return SCONNECT_GetElementAs<long>(m_pSPobject->GetVector(), m_lIndex);
}

CSPproxy::operator int() const
{
	return SCONNECT_GetElementAs<int>(m_pSPobject->GetVector(), m_lIndex);
}

CSPproxy::operator float() const
{
	return SCONNECT_GetElementAs<float>(m_pSPobject->GetVector(), m_lIndex);
}


