package Splus::SplusUtils;
use v5.8;

=head1 NAME

  Splus::SplusUtils - more utilities used by other Splus perl modules
    (Utils.pm was copied/modified from R, SplusUtils.pm was written at Spotfire
    Division, TIBCO Software Inc.)

=head1 SYNOPSIS

  dosify("My Directory/File") returns filename without spaces and with backslashes,
     using the "MYDIRE~number" mapping used by Windows.  If file does
     not exist, but directory does, convert the directory part of it.
     If directory does not exist, die.  On Unix this just returns
     its input.  Also, on Windows if Splus::Vars::S_BUILD_SYS does not
     start with "MS", it just returns its input.  The intent is that
     we can pass this file name to cmd.exe (or nmake.exe) without quoting it.
     On Unix, please don't use spaces in file names.

  globlist(@_) returns list containing all elements of list @_ passed through
     glob, so that, e.g., *.c is expanded to all file names ending with .c.

  rm_empty_dot_data looks for quasi-empty .Data made by Splus -vanilla
     on Windows (a bug) and removes it.

  copy_recursive(from, to, [line_ending_style]), where line_ending_style
     is one of "unix", "windows", "native" (the default), or "asis".
     line_ending_style determines how text files are translated.

=cut

use Exporter;
@ISA = qw(Exporter) ;
@EXPORT = qw(dosify globlist rm_empty_dot_data copy_recursive add_built_stamp_to_description_file
             add_packaged_stamp_to_description_file safe_abs_path find_package get_libraries_list
        get_default_user_library) ;
use strict ;
my $debug = 0 ;

use Splus::Vars;
use Splus::Utils;
use File::Path;
use File::Basename ;
use File::Copy;
use File::Find;
use Cwd qw(getcwd abs_path) ;

my $ms = ${Splus::Vars::S_BUILD_SYS} =~ /^MS/i ;
my $ostype = ${Splus::Vars::OSTYPE} ; # "windows" or "unix"

sub dosify {
    my $filename = shift ;
    if ($filename && $ms) {
        # print "dosify: \"$filename\"->";
        $filename =~ s^/^\\^g;
        # print "\"$filename\"->";
        if ("$filename" =~ / / && -e "$filename") { $filename = Win32::GetShortPathName($filename) } ;
        if ("$filename" =~ / .*\\/) {
            # GetShortPathName only works if file exists.  For files to
            # make we want to fix up directory portion of name.
            my $dir = $filename; $dir =~ s/\\[^\\]*$// ;
            my $shortdir = Win32::GetShortPathName($dir);
            die "Directory \"$dir\" does not exist" if $shortdir eq "" ;
            $filename =~ s/^.*\\/$shortdir\\/;
        }
        # print "\"$filename\"\n";
    }
    $filename;
}

sub safe_abs_path{
    my $filename = shift ;
    my $abs_filename ;
    eval {
       $abs_filename = abs_path($filename) ;
    } ;
    $abs_filename = "<unknown directory>/$filename" if ($@ or $abs_filename eq "") ;
    $abs_filename ;
}

sub globlist {
    # expand *.c and the like in the argument list
    # This is required on Windows, where bat files don't expand *.c.
    # On Unix we do it for consistency.  It means you cannot use
    # quotes to protect files with names including * and [.
    my @retlist = () ;
    foreach $_ (@_) {
        push @retlist, glob($_) ;
    }
    @retlist ;
}

# Work around bug in Windows Splus START -vanilla:
# it creates .\.Data (and standard subdirectories)
# even though it does not use them.  rm_empty_dot_data
# will remove such a thing if it exists.  If it is
# empty, do not remove it.
# "empty" means it has empty subdirectories "__Help"
# and "__Meta" and empty file "___nonfi".  ("__Meta"
# apparently does not get an initial ___nonfi.)
# Return 1 if there is no longer a .Data on return,
# 0 if we could not or chose not to get rid of it.
sub rm_empty_dot_data {
    -d ".Data" or return 1 ;
    opendir my $dirhandle, ".Data" or return 0 ; # closes when out of scope
    while (my $filename = readdir $dirhandle) {
        next if $filename =~ /^(\.|\.\.)$/ ;
        next if $filename eq "___nonfi" && -z $filename ; # -z means 0-length
        if ($filename =~ /^(__Meta|__Hhelp)$/ && -d $filename) {
            # check that directory is empty
            opendir my $mh, ".Data/$filename" or return 0 ;
            while (my $fn = readdir $mh) {
                next if $fn =~ /^(\.|\.\.)$/ ;
                # print "Found file $fn in .Data/$filename, not empty\n";
                closedir $mh ;
                return 0 ;
            }
            closedir $mh ;
        }
        # print "Found file $filename in .Data, not empty\n";
        closedir $dirhandle ;
        return 0 ;
    }
    closedir $dirhandle ;
    # print "Removing empty .Data directory made by Splus -vanilla\n";
    File::Path::rmtree(".Data") ;
    return 1 ;
}

# _copy_file will copy a binary file, but will translate
# a non-binary file to have desired line endings.
sub _copy_file
{
    my $from = shift or die "Missing from argument to _copy_file" ;
    my $to = shift or die "Missing to argument to _copy_file" ;
    my $line_ending_style = shift or die "Missing line_ending_style argument to _copy_file" ;
    -e $from or die "Source file $from does not exist" ;
    -d $from and die "Source \"file\" $from is a directory" ;
    -r $from or die "Cannot read source file $from to copy" ;
    -e $to and die "Destination file $to already exists (will not overwrite)" ;
    my $copy_as_binary = (-B $from) || ($line_ending_style =~ /^(asis)$/) ;
    print "_copy_file from $from to $to (line_ending_style=$line_ending_style, copy_as_binary=$copy_as_binary)\n" if $debug ;
    if ( $copy_as_binary ) {
        # Binary files are just copied.
        File::Copy::copy("$from","$to") or die "Could not copy file $from to $to ($!)" ;
    } else {
        # Text files will get desired line endings at end of all lines, including last.
        open my $handle_from, "< $from" or die "Cannot open file $from for reading ($!)" ;
        # $handle's file will be closed when $handle goes out of scope
        open my $handle_to, "> $to" or die "Cannot create file $to ($!)" ;
        binmode $handle_to ; # so perl on Windows doesn't add \r before \n
        my $ending ;
        if ($line_ending_style eq "unix") { $ending = "\n" ; }
        elsif ($line_ending_style eq "windows") { $ending = "\r\n" ; }
        else { die "logic error: line_ending_style=\"$line_ending_style\", not \"unix\" or \"windows\"\n"; }
        while (<$handle_from>) {
            chomp ; # removes trailing \n
            s/\r$// ; # removes trailing \r
            print $handle_to "$_$ending" ;
        }
        close $handle_to ;
        close $handle_from ;
    }
}

sub copy_recursive
{
    my $from = shift or die "Missing from argument in call to copy_recursiv" ;
    my $to = shift or die "Missing to argument in call to copy_recursive" ;
    # Expect from and to both be directories
    # to must exist and will be parent directory of new copy.
    # The copy will have the same basename as the original.
    my $line_ending_style = "native" ; # optional 3rd argument, "native","unix","windows","asis"
    if (@_) {
        my $tmp = shift ;
        $line_ending_style = $tmp if $tmp ne "" ;
    }
    $line_ending_style =~ /^(native|unix|windows|asis)$/
        or die "line_ending_style should be one of native, unix, windows, or asis, not \"$line_ending_style\"" ;
    if ($line_ending_style eq "native") { $line_ending_style = $ostype ; }
    $line_ending_style =~ /^(unix|windows|asis)$/
        or die "problem with line_ending_style=native: ostype=\"$ostype\"" ;
    # -d $from or warn "Warning: from argument to copy_recursive is not a directory";
    -d $to or die "to argument to copy_recursive, \"$to\", is not a directory";
    print "copy_recursive from $from to $to, line_ending_style=$line_ending_style\n" if $debug ;
    if ( -d $from ) {
       $to = "$to/" . basename("$from") ;
       print "  made directory $to\n" if $debug ;
       mkdir $to or die "Cannot make directory $to ($!)" ;
       opendir my $dir_handle, "$from" or die "Cannot read from directory $from ($!)" ;
       foreach my $f (readdir $dir_handle) {
           next if ($f eq "." || $f eq "..") ;
           $f = "$from/$f" ;
           # print "  copying $f\n" if $debug ;
           if (-l $f) { warn "Will not copy symblic link $f" ; }
           elsif (-f $f) { _copy_file($f, "$to/" . basename("$f"), $line_ending_style) ; }
           elsif (-d $f) { copy_recursive($f, $to, $line_ending_style) ; }
           else { warn "$f is neither a file nor a directory, will not try to copy"; }
       }
       closedir $dir_handle ;
    } elsif (-l $from) {
       warn "Will not copy symblic link $from" ;
    } elsif (-f $from) {
       _copy_file($from, "$to/" . basename("$from"), $line_ending_style) ;
    } else {
       warn "\"$from\" is not a directory, symbolic link, nor ordinary file: not copying it" ;
    }
}

sub copy_source_package {
    # Copy just the source part of a package.   Try to
    # leave out built stuff.
    my $from = shift or die "Missing from argument in call to copy_source_package" ;
    my $to = shift or die "Missing to argument in call to copy_source_package" ;
    -d $to or die "$to is not a directory" ;
    # Expect from and to both be directories
    # to must exist and will be parent directory of new copy.
    # The copy will have the same basename as the original.
    my $line_ending_style = shift or die "Missing line_ending_style to copy_source_package" ;
    if ($line_ending_style eq "native") { $line_ending_style = $ostype ; }
    my @files_to_copy = qw(DESCRIPTION NEWS COPYING THANKS COPYRIGHTS ChangeLog configure cleanup NAMESPACE LICENSE LICENCE README) ;
    my @dirs_to_copy = qw(R data man src java inst swingui demo tests exec po) ;
    $to = "$to/" . basename($from) ;
    mkdir "$to" or die "Cannot make directory $to ($!)";
    foreach my $file (@files_to_copy) {
        _copy_file("$from/$file", "$to/$file", $line_ending_style) if -e "$from/$file" ;
    }
    foreach my $dir (@dirs_to_copy) {
        copy_recursive("$from/$dir", "$to", $line_ending_style) if -d "$from/$dir" ;
    }
    # Make all files and directories writable (by user)
    File::Find::find(
        sub {
                my $fullname = $File::Find::name ; # for error messages only
                my @s = stat($_) or warn "stat failed on $fullname: $!\n" ;
                if (@s) {
                    my $perm = @s[2] | 0200 ; # add user-write permission
                    chmod $perm, $_ or warn "Could not add write permission to $fullname: $!\n" ;
                }
            },
        "$to");
    # The copy_recursive() copies too much, so we remove stuff now
    if (-d "$to/src") {
        # remove object files or subdir's presumably made by C++ compiler
        opendir my $dirhandle, "$to/src" or die "Cannot read directory $to/src ($!)" ;
        while (my $filename = readdir $dirhandle) {
            next if $filename =~ /^(\.|\.\.)$/ ;
            my $path = "$to/src/$filename" ;
            if (-d $path) {
                File::Path::rmtree("$to/src/$filename") ;
            } elsif (-l $path) {
                # don't want to deal with symbolic links
            } elsif ($filename =~ /\.(o|obj)$/i) {
                unlink($path) ;
            }
        }
        closedir $dirhandle ;
    }
    foreach my $dir (qw(R data)) {
        # remove .Data
        if (-d "$to/$dir") {
            File::Path::rmtree("$to/$dir/.Data") if -d "$to/$dir/.Data" ;
        }
    }
    # !!! Get rid of .p4* files and directory CVS, .svn, etc.
    # There should not be any at top level, but subdirectories
    # might contain such stuff.
    File::Find::find(
        sub {unlink("$File::Find::name")
                if -f $_ && /^(\.p4rc|\.p4ignore|\.project)$/ ;
             File::Path::rmtree("$File::Find::name")
                if -d $_ && /^(\.svn|CVS|\.arch-ids|\.externalTooBuilders)$/ ;
            },
        "$to");
}

# following were in build.pl to add to DESCRIPTION lines like
# Packaged: Mon Jan 23 22:00:28 2006; Liawand
# Built: R 2.3.0; i386-pc-mingw32; 2006-04-24 23:07:46; windows

sub add_packaged_stamp_to_description_file {
    my ($dpath) = @_;
    my @lines = &read_lines($dpath);
    @lines = grep(!/^\s*$/, @lines); # Remove blank lines.
    my $user_name;
    if($ostype eq "windows") {
        $user_name = Win32::LoginName();
    }
    else {
        $user_name = (getpwuid($<))[0];
    }
    my $fh ;
    if ( ! ($fh = new IO::File($dpath, "w"))) {
        warn "Cannot open file '$dpath' to add \"Packaged:\" entry" ;
    } else {
        ## Do not keep previous build stamps.
        @lines = grep(!/^Packaged:/, @lines);
        $fh->print(join("\n", @lines), "\n");
        $fh->print("Packaged: ",
                   scalar(localtime()), "; ",
                   $user_name, "\n");
        $fh->close();
    }
}
sub add_built_stamp_to_description_file {
    my ($dpath) = @_;
    my @lines = &read_lines($dpath);
    @lines = grep(!/^\s*$/, @lines); # Remove blank lines.
    my $user_name;
    if($ostype eq "windows") {
        $user_name = Win32::LoginName();
    }
    else {
        $user_name = (getpwuid($<))[0];
    }
    my $fh ;
    if (! ($fh = new IO::File($dpath, "w"))) {
        warn "Cannot open file '$dpath' to add \"Built:\" entry";
    } else {
        ## Do not keep previous build stamps.
        @lines = grep(!/^Built:/, @lines);
        $fh->print(join("\n", @lines), "\n");
        # Built: R 2.3.0; i386-pc-mingw32; 2006-04-24 23:07:46; windows
        $fh->print("Built: ",
                   "Spotfire S+", "; ", # would like to put version number here
                   "?", "; ",  # details of build environment
                   iso_gmtime(), "; ", # R gives no time zone here, is it local or utc?
                   "$^O", "; ", # operating system, as from uname on Unix
                   "\n");
        $fh->close();
    }
}

sub iso_gmtime {
    my ($sec, $min, $hour, $day, $mon, $yr, $help1, $help2) = gmtime() ;
    $yr += 1900 ;
    $mon += 1 ;
    # "$yr-$mon-$day $hour:$min:$sec UTC"; # need for force 2 digits each for h:m:s
    sprintf "$yr-$mon-$day %02d:%02d:%02d UTC", $hour, $min, $sec ;
}


sub get_default_user_library {
    my $S_USER_APPDATA_DIR = Splus_getenv("S_USER_APPDATA_DIR", "") ;
    my $value = file_path($S_USER_APPDATA_DIR, "library") ;
    return $value ;
}

# get_S_LIBS_list(void): return split($S_PATHSEP,#S_LIBS)
#  optional argument global_only means to omit user-specific library from default.
sub get_libraries_list {
    my $global_only = shift ;
    my $S_PATHSEP = ${Splus::Vars::S_PATHSEP} ;
    my $S_LIBS = Splus_getenv("S_LIBS", "") ;
    my $SHOME = ${Splus::Vars::SHOME} ;
    my $S_USER_APPDATA_DIR = Splus_getenv("S_USER_APPDATA_DIR", "") ;
    my @libraries = () ;
    if ($S_LIBS && $S_LIBS =~ m/[^\s]/) {
        push(@libraries, split($S_PATHSEP, $S_LIBS)) ;
    } else {
        if (!$global_only) {
            push(@libraries, get_default_user_library()) ;
        }
        push(@libraries, file_path($SHOME, "local/library")) ;
        push(@libraries, file_path($SHOME, "library")) ;
    }
    return @libraries ;
}

# find_package(pkgname, library, only_first, global_only): given a package name and perhaps a library
# directory in which to look, give path to package.  If directory is
# not given, use env. var. S_LIBS and look through directories
# listed there.  Return list of all places it is found.
# If only_first is given as 1, then look only in first entry of
# S_LIBS or $library.

sub find_package {
    my $pkgname = shift or die "find_package needs a package name argument" ;
    my $library = shift ;
    my $only_first = shift ;
    my $global_only = shift ;
    my $S_PATHSEP = ${Splus::Vars::S_PATHSEP} ;
    my @libraries = () ;
    
    if ($library) {
        push @libraries, split($S_PATHSEP, $library) ;
    } else {
        push @libraries, get_libraries_list($global_only) ;
    }
    if ($only_first) { @libraries=(@libraries[0]) ; }
    my @retval = () ;
    foreach $library (@libraries) {
       my $path=file_path($library,$pkgname);
       push @retval, $path if -f file_path($path,"DESCRIPTION") ;
    }
    # print "retval=", join("   ", @retval), "\n" ;
    return @retval ;
}

1;
