/* @(#) Copyright (c), 1987, 2006 Insightful Corp.  All rights reserved. */
/* @(#) $RCSfile: S_define.h,v $: $Revision: #15 $, $Date: 2006/06/26 $  */
/***
   NAME S_defines.h
   PURPOSE Contains all public macros for users who write C, C++,
     or Fortran functions.
***/
#ifndef S_DEFINES_H
#define S_DEFINES_H 1

#include <S_tokens.h>

/*
 * all calls to calloc, realloc and free should go through the following
 * three macros.  There should be no explicit calls to malloc in the code.
 */ 
#define Calloc(n,t)	(t *)S_ok_calloc((n),sizeof(t), S_evaluator)
#define Realloc(p,n,t)	(t *)S_ok_realloc((char *)(p),(n)*sizeof(t), S_evaluator)
#define Free(p)		S_ok_free((char *)(p), S_evaluator)
#define Malloc(n,t)     (t *)S_ok_malloc((size_t)((n) * sizeof(t)), S_evaluator)

/* keep this here??? needed by SDK */
#define FREE_HEADER(p)  if((p)->Type == S_MODE_ENTRY_TYPE) free_header(p,S_evaluator)

/* calls to allocation in S frames. */
/* because S stores small amounts w/o chaining, realloc and free need */
/* to know the size of the allocation */
#define Salloc(n, t)	(t *)S_alloc(n, sizeof(t),S_evaluator)
#define Smalloc(n, t)   (t *)S_malloc((long)((n)* sizeof(t)),S_evaluator)
#define Srealloc(p,n,old, t)	(t *)S_realloc((char *)(p),n,old,sizeof(t), S_evaluator)
#define Sfree(p, n)     if(((n) * sizeof(*p))>scrap)(void)free_block((void *)p,S_evaluator)


/* the macro versions of copying */
/* COPY, COPY_ALL as defined in "Programming with Data" */
#define COPY(obj) (obj = s_protect_object(obj, S_evaluator))
#define COPY_ALL(obj) copy_data(obj, NULL_ENTRY, S_evaluator)


/* for user C code: make a class pointer and ensure it has rep'n */
#define MAKE_CLASS(what) make_class_and_rep(what, S_evaluator)

#define CLASS_FOUND(cl) ((cl)->meta_name != 0)
#define ENSURE_CLASS(cl) (CLASS_FOUND(cl)?S_TRUE:find_class_rep(cl,S_evaluator))

#define HAS_SLOTS(cl) ((cl)->slots_table && ((cl)->slots_table->tlen > 0))
#define GET_SLOT_OFFSET(obj, name)  get_slot_offset((obj), name, S_evaluator) 
#define SLOT_POSITION(obj, name)  (find_slot_position((obj), name, S_evaluator) -1)

/* the slots in "array" and "named" classes */
/* these can be used to access specific slots in "array" and "named" objects */
/* for example, creating a named object:
   SET_ELEMENT( named_obj, NAMED_DATA_SLOT, data_vector );
   SET_ELEMENT( named_obj, NAMED_NAMES_SLOT, names_vector );
*/
enum arraySlots {ARRAY_DATA_SLOT, ARRAY_DIM_SLOT, ARRAY_DIMNAMES_SLOT};
enum namedSlots {NAMED_DATA_SLOT, NAMED_NAMES_SLOT};

#define S_IS_VOID_VECTOR(p) ((p)==NULL_ENTRY||(p)==S_void||(p)->mode==S_MODE_MISSING)

/* status flags for ent->x.status */
#define PRECIOUS(p)		(p->frame != NULL_ENTRY)
#define CHECK_IT		63
#define IS_NULL_STRING(p)	((p)==NULL_STRING || *(p)=='\0')

#define SHORT_LIST_LENGTH	4

#define PERM_FRAME		-2L
#define FRAME0			0L

/* For the documented-to-users macros that follow these, define macros that will
 * always be the fprintf/fputs functions used by Splus, whether or not newredef.h
 * redefines them (which it won't if NO_NEWIO is #defined above S.h).
 */
#if defined(USE_NEWIO) /*(*/
#define S_fprintf	S_newio_fprintf
#define S_fputs		S_newio_fputs
#else /*)(*/
#define S_fprintf	fprintf
#define S_fputs		fputs
#endif /*)*/

#define MESSAGE		S_fprintf(error_file,
#define PROBLEM		S_fprintf(error_file,
#define WARNING(x)	), set_error_buf(S_evaluator), Warning(error_buf, x, S_evaluator)
#define RECOVER(x)	), set_error_buf(S_evaluator), Recover(error_buf, x, S_evaluator)
#ifdef ERROR
  #undef ERROR
#endif
#define ERROR		), set_error_buf(S_evaluator), Recover(error_buf, NULL_ENTRY, S_evaluator)
#define WARN		), set_error_buf(S_evaluator), Warning(error_buf, NULL_ENTRY, S_evaluator)
#define TERMINATE	), set_error_buf(S_evaluator), S_terminate(error_buf)
#define PRINT_IT	), set_error_buf(S_evaluator), print_message(error_buf,0L,S_stderr, S_evaluator),S_fputs("\n",S_stderr),FLUSH_STDERR

#define AUDIT		), audit_print_buf()
#define END_MESSAGE	)
#define ERROR_BUF_LENGTH 256
#define GET_MESSAGE s_get_message(S_evaluator)

#define FLUSH_STDERR ok_flush(S_stderr, S_evaluator)
#define FLUSH_STDOUT ok_flush(S_stdout, S_evaluator)

/* how to put a C routine on the end-of-session action list */
#define ON_QUIT(f) add_exit(f, PERM_FRAME, S_evaluator)

/* macros for accessing, setting attributes from C */
#define GET_ATTR(x,what) (x->mode == S_MODE_STRUCTURE ? xact_comp(x, what, S_evaluator) : 0)
#define GET_DATA(x) (x->mode == S_MODE_STRUCTURE ? xact_comp(x, (char*)".Data", S_evaluator) : x)
#define GET_DIM(x) (x->mode == S_MODE_STRUCTURE ? xact_comp(x, (char*)".Dim", S_evaluator) : 0)
#define SET_DIM(x, d) set_dim(x, d)
#define SET_DIMNAMES(x, d) set_dimnames(x, d)
#define GET_DIMNAMES(x) (x->mode == S_MODE_STRUCTURE ? xact_comp(x, (char*)".Dimnames", S_evaluator) : 0)
#define GET_COLNAMES(x) get_dimnames_el(x,1, S_evaluator)
#define GET_ROWNAMES(x) get_dimnames_el(x,0, S_evaluator)
#define GET_LEVELS(x) (x->mode == S_MODE_STRUCTURE ? xact_comp(x, (char*)".Label", S_evaluator) : 0)
#define SET_LEVELS(x, l) set_levels(x, l)
#define GET_TSP(x) (x->mode == S_MODE_STRUCTURE ? xact_comp(x, (char*)".Tsp", S_evaluator) : 0)
#define GET_NAMES(x) get_names(x)
#define SET_NAMES(x, n) set_names(x, n)

/* is the class an SV3 class? */
/* Version 3 classes have no rep. and no prototype, but by context are
 the class of an actual object */
#define VERSION3_CLASS(cl) (ENSURE_CLASS(cl), (cl)->Virtual && !(cl)->prototype)

/* structure of S objects for C programming */
#define GET_CLASS_NAME(p) (p)->Class->name
#define GET_CLASS(p) (p)->Class
#define SET_CLASS(p, clobj) s_set_class_to_class((p), (clobj), S_evaluator)
#define GET_LENGTH(p) (p)->length
#define SET_LENGTH(p, n) set_length2((p), (long)(n), S_evaluator)

/* some coercion to builtin vector types */
#define AS_VECTOR(x) coevec(x, S_MODE_ANY, S_FALSE, CHECK_IT, S_evaluator)
#define AS_LIST(x) ASlist(x)
#define AS_CHARACTER(x) coevec(x, S_MODE_CHAR, S_FALSE, CHECK_IT, S_evaluator)
#define AS_NUMERIC(x) coevec(x, S_MODE_DOUBLE, S_FALSE, CHECK_IT, S_evaluator)
#define AS_SINGLE(x) coevec(x, S_MODE_REAL, S_FALSE, CHECK_IT, S_evaluator)
#define AS_INTEGER(x) coevec(x, S_MODE_INT, S_FALSE, CHECK_IT, S_evaluator)
#define AS_LOGICAL(x) coevec(x, S_MODE_LGL, S_FALSE, CHECK_IT, S_evaluator)
#define AS_RAW(x) coevec(x, S_MODE_RAW, S_FALSE, CHECK_IT, S_evaluator)
#define AS_COMPLEX(x) coevec(x, S_MODE_COMPLEX, S_FALSE, CHECK_IT, S_evaluator)

/* create built-in vector objects */
#define NEW_LIST(n) alcvec(S_MODE_LIST, n, S_evaluator)
#define NEW_NUMERIC(n) alcvec(S_MODE_DOUBLE, n, S_evaluator)
#define NEW_INTEGER(n) alcvec(S_MODE_INT,n, S_evaluator)
#define NEW_LOGICAL(n) alcvec(S_MODE_LGL, n, S_evaluator)
#define NEW_SINGLE(n) alcvec(S_MODE_REAL, n, S_evaluator)
#define NEW_COMPLEX(n) alcvec(S_MODE_COMPLEX, n, S_evaluator)
#define NEW_CHARACTER(n) alcvec(S_MODE_CHAR, n, S_evaluator)
#define NEW_RAW(n) alcvec(S_MODE_RAW, n, S_evaluator)
/* a new object of a given class */
#define NEW(cl) (copy_data(new_S_object(cl, S_evaluator), NULL_ENTRY, S_evaluator))
/* test belonging to or extending the basic classes */
#define IS_NUMERIC(x) (IS_OBJ(x, numeric))
#define IS_INTEGER(x) (IS_OBJ(x, integer))
#define IS_LIST(x) (IS_OBJ(x, list))
#define IS_LOGICAL(x) (IS_OBJ(x, logical))
#define IS_SINGLE(x) (IS_OBJ(x, single))
#define IS_COMPLEX(x) (IS_OBJ(x, complex))
#define IS_CHARACTER(x) (IS_OBJ(x, character))
#define IS_RAW(x) (IS_OBJ(x, raw))

/* scalar values */
#define CHARACTER_VALUE(x) string_value(x, S_evaluator)
#define LOGICAL_VALUE(x) logical_value(x, NULL_ENTRY, S_evaluator)
#define INTEGER_VALUE(x) long_value(x, NULL_ENTRY, S_evaluator)
#define NUMERIC_VALUE(x) double_value(x, S_evaluator)
#define SINGLE_VALUE(x) real_value(x, S_evaluator)
#define LIST_VALUE(x) (PROBLEM "the `value' of a list object is not defined" ERROR)
#define RAW_VALUE(x) (PROBLEM "the `value' of a RAW object is not defined" ERROR)

/* pointers to data */
#define NUMERIC_POINTER(x) (x)->value.Double
#define CHARACTER_POINTER(x) (x)->value.Char
#define SINGLE_POINTER(x) (x)->value.Float
#define LOGICAL_POINTER(x) (x)->value.Long
#define INTEGER_POINTER(x) (x)->value.Long
#define RAW_POINTER(x) (x)->value.name
#define COMPLEX_POINTER(x) (x)->value.Complex
#define LIST_POINTER(x) (x)->value.tree

/* the .Data slot is required to be the first slot */
/* in Version 4  */
#define DATA_SLOT(p) ((p)->value.tree[0])
/* look up data slot for class structure and virtual classes (i.e., */
/* undefined classes from the Version 4 perspective) */
#define X_DATA_SLOT(p) ((p)->mode == S_MODE_STRUCTURE ? ((p)->Class->Virtual ? \
	x_data_slot(p, NULL, S_evaluator) : (p)->value.tree[0]) : p)

#define LENGTH(p) (p->Class->nslots > 0 ? get_length2(p, S_evaluator) : p->length)
#define SET_ELEMENT(p, where, value) (s_set_element(p, where, value, S_evaluator))

#define EMPTY_STRING (char *)(Empty_name->text)
#define NON_EMPTY_NAME(p) (p != EMPTY_STRING && p && *p)

/* some defines for the data pointers.  Hides the use of union for pointers*/
#define LOGICAL_DATA(p) ((p)->value.Long)
#define INTEGER_DATA(p) ((p)->value.Long)
#define REAL_DATA(p) ((p)->value.Float)
#define SINGLE_DATA(p) ((p)->value.Float)
#define DOUBLE_DATA(p) ((p)->value.Double)
#define NUMERIC_DATA(p) ((p)->value.Double)
#define CHARACTER_DATA(p) ((p)->value.Char)
#define COMPLEX_DATA(p) ((p)->value.Complex)
#define RECURSIVE_DATA(p) ((p)->value.tree)

enum s_mode_types {S_ATOMIC_DATA, S_RECURSIVE_DATA, S_NUMERIC_COMPATIBLE_DATA};

#define IS_NUMERIC_COMPATIBLE(x) (s_check_data(x, S_NUMERIC_COMPATIBLE_DATA, S_evaluator))
#define IS_ATOMIC(x) (s_check_data(x, S_ATOMIC_DATA, S_evaluator))
#define IS_RECURSIVE(x) (s_check_data(x, S_RECURSIVE_DATA, S_evaluator))


#define EVAL(ent) Eval_in_frame(ent, Nframe, S_evaluator)
#define EVAL_IN_FRAME(ent, f) Eval_in_frame(ent, f, S_evaluator)
#define ASSIGN(name, obj) set_local(obj, make_name(name, S_evaluator))
#define ASSIGN_IN_FRAME(name, obj, n) set_in_frame(n, obj, make_name(name, S_evaluator), S_evaluator)
#define GET(name) get_data(make_name(name, S_evaluator), S_evaluator)
#define GET_FROM_FRAME(name, n) find_in_frame(make_name(name, S_evaluator), n, S_evaluator)

/* macro versions of is, as: first, for user C code */
#define IS(obj, cl) isObjectIndex(obj, cl?cl->index: -1, obj?obj->Class:0, S_evaluator)
#define AS(obj, cl) as_class(obj, cl, S_FALSE)
/* and for internal S code using global class pointers */
#define IS_OBJ(x, what)  ((x)->Class == s_##what##_class || isObjectIndex(x, s_##what##_class->index, (x)->Class, S_evaluator))
#define AS_OBJ(x, what) (x->Class == s_##what##_class ? x : \
			 as_class(x, s_##what##_class, S_FALSE))

/* I know this file should not have includes in it, but this saves
 * having to put it in both S_engine.h and S.h
 * The orig_locale can be clobbered by the next setlocale() so we
 * need to save it.  It can be arbitrarily long.  I believe the
 * output of setlocale(..., NULL) cannot be NULL -- it must point
 * to a string.
 */
#include <locale.h>
#define IN_C_NUMERIC_LOCALE(expr)                                 \
	{ char *orig_locale = setlocale(LC_NUMERIC, (char*)NULL); \
          char orig_locale_static_buffer[512];                    \
          int locale_changed ; size_t locale_length ;             \
          if (locale_changed = (0 != strcmp(orig_locale, "C"))) { \
	    locale_length = strlen(orig_locale) ;                 \
            orig_locale = (locale_length < 512) ?                 \
	      strcpy(orig_locale_static_buffer, orig_locale):     \
	      S_strdup(orig_locale,S_evaluator);                  \
            (void)setlocale(LC_NUMERIC, "C");                     \
          }                                                       \
          expr;                                                   \
	  if (locale_changed) {                                   \
            (void)setlocale(LC_NUMERIC, orig_locale);             \
            if (orig_locale != orig_locale_static_buffer)         \
	      free(orig_locale);                                  \
	  }                                                       \
        }

#endif /* S_DEFINES_H */
