/*
 imkmeans.c implementation of interfaces for IMinerPrincipalComponentsModel objects.
*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <memory.h>

#include "IMPrincipalComponents.h"


#define PRINC_COMP_LOAD(pIMinerMatrix,i,j) IMINER_DOUBLE_VALUE(IMINER_MATRIX_DATA_PTR(pIMinerMatrix), (i)+(j)*IMINER_MATRIX_NCOLUMNS(pIMinerMatrix));

/* alloc memory for type IMinerPrincipalComponentsModel. */
long IMCSRC_STDCALL IMinerPrincipalComponentsModel_create(
 IMinerObject* pModel,		   /* out: data object */
 IMinerObject* md,			    /* in:  meta data object */
 long nComponents,             /* in:  number of components */
 long nColumns,                /* in:  number of cols */ 
 const double* pdCenters,      /* in:  center factors of length nColumns */
 const double* pdScale,        /* in:  scaling factors of length nColumns */
 const double* pdLoadings      /* in:  loading factors of length (nComponents*nColumns) */
)
{
	long nStatus = IMINER_SUCCESS;
	char* ppszMemberNames[] = {"metaData", "dvCenter", "dvScale", "dmLoading"};

	if(pModel == NULL || nComponents < 1L  || nColumns < 1L)
		return IMINER_BAD_INPUT_ARGS;

	/* a PrincipalComponentsModel always have IMINER_PRINCIPAL_COMPONENTS_LOAD_MEMBER+1 elements as list */
	nStatus = IMinerList_create(pModel, IMINER_PRINCIPAL_COMPONENTS_LOAD_MEMBER+1, IMINER_MODE_MODEL); 
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* member: meta data */
	nStatus = IMinerList_clone(IMINER_PRINCIPAL_COMPONENTS_META_DATA(pModel), md);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* member 0: dvCenter */
	nStatus = IMinerVector_create(IMINER_PRINCIPAL_COMPONENTS_CENTER_PTR(pModel), nColumns, IMINER_MODE_DOUBLE, pdCenters);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* member 1: dvScale */
	nStatus = IMinerVector_create(IMINER_PRINCIPAL_COMPONENTS_SCALE_PTR(pModel), nColumns, IMINER_MODE_DOUBLE, pdScale);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* member 2: dmLoading */
	nStatus = IMinerDoubleMatrix_create(IMINER_PRINCIPAL_COMPONENTS_LOAD_PTR(pModel), nComponents, nColumns, pdLoadings, NULL, NULL);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/*member 4 (special) : member names */
	nStatus = IMinerList_setNamesFromStrings(pModel, (const char**)ppszMemberNames);

	/* set the class ID */
	pModel->m_nMode = IMINER_MODE_MODEL;
	return nStatus;
}

long IMCSRC_STDCALL IMinerPrincipalComponentsModel_isValid(const IMinerObject* pModel)
{
	long nStatus = IMINER_SUCCESS;
	long nRows, nColumns;
	IMinerObject* pElement;

	if(pModel == NULL || !IMinerList_isValid(pModel)  || 
	   IMINER_OBJECT_LENGTH(pModel) < IMINER_PRINCIPAL_COMPONENTS_LOAD_MEMBER+1)
		return 0L;

	if (!IMinerList_isValid(IMINER_PRINCIPAL_COMPONENTS_META_DATA(pModel))) return 0L;

	/* member 0: data */
	pElement = IMINER_PRINCIPAL_COMPONENTS_CENTER_PTR(pModel);
	if(!IMinerVector_isDouble(pElement) )
		return 0L;
	/* member 1: dim */
	pElement = IMINER_PRINCIPAL_COMPONENTS_SCALE_PTR(pModel);
	if(!IMinerVector_isDouble(pElement) )
		return 0L;
	/* member 2: row name */
	pElement = IMINER_PRINCIPAL_COMPONENTS_LOAD_PTR(pModel);
	if(!IMinerDoubleMatrix_isValid(pElement))
		return 0L;
	/* special member : member names mut also be valid and has the same length as this object*/
	pElement = IMINER_LIST_NAMES_PTR(pModel);
	if(!IMinerVector_isString(pElement) && pElement->m_nLen == pModel->m_nLen)
		return 0L;

	/* number of rows and columns must match with length of scale and size respectively*/
	nColumns = IMINER_MATRIX_NCOLUMNS(IMINER_PRINCIPAL_COMPONENTS_LOAD_PTR(pModel));
	nRows    = IMINER_MATRIX_NROWS(IMINER_PRINCIPAL_COMPONENTS_LOAD_PTR(pModel));

	if(IMINER_OBJECT_LENGTH(IMINER_PRINCIPAL_COMPONENTS_SCALE_PTR(pModel)) != nColumns)
		return 0L;
	return 1L;
}

/* free memory */
long IMCSRC_STDCALL IMinerPrincipalComponentsModel_destroy(IMinerObject* pModel )
{
	return IMinerObject_destroy(pModel);
}


/* write to stdout */
long IMCSRC_STDCALL IMinerPrincipalComponentsModel_print(const IMinerObject* pModel)
{
	long nStatus;

	/* printf("Begin IMinerPrincipalComponentsModel ...\n"); */
	if(!IMinerPrincipalComponentsModel_isValid(pModel))
	{
		IMiner_error("%s(%d) : ", __FILE__, __LINE__);
		IMiner_error("Invalid pModel\n");
		return IMINER_BAD_INPUT_ARGS;
	}
	printf("Meta Data:\n");
	if (IMinerMetaData_print(IMINER_PRINCIPAL_COMPONENTS_META_DATA(pModel))!=IMINER_SUCCESS) {
		return IMINER_BAD_INPUT_ARGS;
	}

	printf("Column Centers:\n");
	nStatus = IMinerVector_print(IMINER_PRINCIPAL_COMPONENTS_CENTER_PTR(pModel));
	if(nStatus != IMINER_SUCCESS)
		return IMINER_BAD_INPUT_ARGS;

	printf("Column Scaling Factor:\n");
	nStatus = IMinerVector_print(IMINER_PRINCIPAL_COMPONENTS_SCALE_PTR(pModel));
	if(nStatus != IMINER_SUCCESS)
		return IMINER_BAD_INPUT_ARGS;

	printf("Component Loading Factor:\n");
	nStatus = IMinerDoubleMatrix_print(IMINER_PRINCIPAL_COMPONENTS_LOAD_PTR(pModel));
	if(nStatus != IMINER_SUCCESS)
		return IMINER_BAD_INPUT_ARGS;

	return IMINER_SUCCESS;
}

/* predict cluster memberships */
long IMCSRC_STDCALL IMinerPrincipalComponentsModel_predict(
 IMinerObject* pOutput,     /* out: output rectangular data */
 const IMinerObject* input, /* in: input rectangular data */													 
 IMinerObject* pDescr,		/* in: input description (if NULL, description will be 
                                   created from input data) */
 const IMinerObject* pModel /* in: the model */
)
{
	long nStatus=IMINER_SUCCESS, i,j, k, nRows, nInputColumns, nOutputColumns;
	long nComponents, nMode, *pnColumnsModes;
	double dScore, dTemp, center, scale, load;
	IMinerObject pInput, *pdvCenter, *pdvScale;
	IMinerObject *pdmLoading, *pvCol=NULL;
	IMinerObject *pvCompLoad=NULL, A;

	char **columnNames;

	if(pOutput==NULL || !IMinerDataSet_isValid(input) || !IMinerPrincipalComponentsModel_isValid(pModel))	
		return IMINER_BAD_INPUT_ARGS;

	/* convert input if needed*/
	nStatus = IMinerMetaData_InputConvert(&pInput, IMINER_PRINCIPAL_COMPONENTS_META_DATA(pModel), input, pDescr);
	if(nStatus != IMINER_SUCCESS) return IMINER_FAIL;

	pdmLoading = IMINER_PRINCIPAL_COMPONENTS_LOAD_PTR(pModel);
	pdvCenter =  IMINER_PRINCIPAL_COMPONENTS_CENTER_PTR(pModel);
	pdvScale  =  IMINER_PRINCIPAL_COMPONENTS_SCALE_PTR(pModel);

	nInputColumns = IMINER_DATASET_NCOLUMNS(&pInput);
	nRows         = IMINER_DATASET_NROWS(&pInput);
	nComponents   = IMINER_MATRIX_NROWS(pdmLoading);

	/* create a matrix double to store the input as double */
	nStatus = IMinerDoubleMatrix_create(&A, nRows, nInputColumns, NULL, NULL, NULL);
	if(nStatus != IMINER_SUCCESS)
		return IMINER_FAIL;

	/* Set the input data to the matrix
	TODO: apply coding factor expansion of categorical data */
	for(j=0L; j< nInputColumns; ++j) {
		nStatus = IMinerDataSet_getColumnMode(&nMode, &pInput, j);
		if(nStatus != IMINER_SUCCESS) return IMINER_BAD_INPUT_ARGS;
		/*if(nMode != IMINER_MODE_DOUBLE) return IMINER_BAD_INPUT_ARGS;*/

		for(i=0L; i< nRows; ++i) {
			IMINER_Aij(&A, i, j) = IMinerDataSet_getNonStringValue(&pInput, j, i);
		}
	}



	/* create the output object of the same size as input + one column per component */
	nOutputColumns = /*nInputColumns + */nComponents;
	pnColumnsModes = (long*) malloc(nOutputColumns*sizeof(long));

	for(j=0L; j<nOutputColumns; ++j)pnColumnsModes[j] = IMINER_MODE_DOUBLE;
	nStatus = IMinerDataSet_create(pOutput, nRows, nOutputColumns, pnColumnsModes);
	if(nStatus != IMINER_SUCCESS)return nStatus;
	columnNames = (char**)malloc(nComponents*sizeof(char*));
	for (i=0; i<nComponents; i++) {
		columnNames[i] = (char*)malloc(32*sizeof(char));
		sprintf(columnNames[i], "Component%d", i);
	}
	nStatus = IMinerDataSet_setColumnNamesFromStrings(pOutput, (const char**)columnNames);
	if(nStatus != IMINER_SUCCESS) return nStatus;

	for (i=0; i<nComponents; i++) free(columnNames[i]);
	free(columnNames);

	/* copy content from the input object */
	for(j=nComponents, i=0; j<nOutputColumns; ++j, ++i)
	{
		nStatus = IMinerDataSet_getColumnMode(&nMode, &pInput, i);
		if(nStatus != IMINER_SUCCESS) return IMINER_BAD_INPUT_ARGS;
		/* copy content of pInput into jth column of pOutput */
		nStatus = IMinerDataSet_setColumnAt(pOutput, j, IMINER_DATASET_COLUMN_PTR(&pInput, i)); 
		if(nStatus != IMINER_SUCCESS) return nStatus;
	}
				
	/* for each row (object), find the component score */
	for(k=0L; k<nComponents; ++k)
	{
		pvCol = IMINER_DATASET_VECTOR_PTR(pOutput, k);
		pvCompLoad = IMINER_MATRIX_DATA_PTR(pdmLoading);
		for(i=0L; i<nRows; ++i)
		{		
			dScore = 0.0;
			for(j=0L; j<nInputColumns; ++j)
			{
				dTemp = IMinerDataSet_getNonStringValue(&pInput, j, i);
				center = IMINER_DOUBLE_VALUE(pdvCenter, j);
				scale  = IMINER_DOUBLE_VALUE(pdvScale, j);

				load = PRINC_COMP_LOAD(pdmLoading, j, k);
				dTemp = load*(dTemp-center)/scale;
				
				dScore += dTemp;

			}

			IMINER_DOUBLE_VALUE(pvCol, i) = dScore;
		}
	}

	free(pnColumnsModes);

	IMinerObject_destroy(&pInput);
	IMinerObject_destroy(&A);

	return IMINER_SUCCESS;
}

