////////////////////////////////////////////////////////////////////////////
// euler.cxx: Implementation of euler
////////////////////////////////////////////////////////////////////////////

#include "sconnect.h"
#include "euler.h"

s_object* euler(s_object* ps_fun, s_object* ps_yStart, 
 s_object* ps_tStart, s_object* ps_tStop, s_object* ps_nSteps)
/**************************************************************
euler() - First order method for solving an ODE
 This function illustrates how to use CSPcall to evaluate
 arbitrary S function with actual arguments being C++ objects. 

REQUIRED ARGUMENTS: 
 ps_fun function object
 ps_yStart inital vector value for y
 ps_tStart inital scalar value for t
 ps_tStop  stop scalar value for t
 ps_nSteps numeber of steps
 
VALUE: 
 a matrix, solution of y'=f(t, y) 
 
S FUNCTION EQUIVALENT:  
 euler() defined in euler.ssc

Usage:
 # y'=f(t,y): a spiral
 f<-function(t, y)
 {
	r<-sqrt(sum(y^2))+t/1000
	r*c(cos(t), -sin(t))
 }
 source('euler.ssc')                   # S version of euler
 # S version of euler
 sys.time({y<-euler(f, c(0,1), 0.0, 30*pi, 3000)})
 # C++ version of euler
 sys.time({z<-.Call("euler", f, c(0,1), 0.0, 30*pi, 3000)})
 all.equal(y, z)                      # Should return T
Or:
 sys.time({y<-euler("f", c(0,1), 0.0, 30*pi, 3000)})
 sys.time({z<-.Call("euler", "f", c(0,1), 0.0, 30*pi, 3000)})
 all.equal(y, z)                      # Should return T
Ploting:
 plot(y[1,], y[2, ]

*************************************************************/
{
	try
	{
		//Throw expection when error occurs in the engine.
		CSPevaluator sEvaluator; 

		//Wrap C++ objects around the input arguments and validating them.
		CSPcall f(ps_fun);  //construct a function call.    
		CSPnumeric y(ps_yStart); 
		CSPnumeric t(ps_tStart); 
		CSPnumeric tStop(ps_tStop); 
		CSPinteger nSteps(ps_nSteps);

		//Create the return object
		CSPnumericMatrix ans(y.length(), nSteps[0]); //output matrix.
		ans.SetJthColumnDirect(0, &y); //set the first column. 

		//Step size
		double h = (tStop[0] - t[0]) /(double) nSteps[0];
		//Euler iterations
		for( long j=1; j < nSteps[0]; ++j)
		{
			y += h * f(t, y); //Call the S function.
			t += h;
			ans.SetJthColumnDirect(j, &y);
		}
		return ans.Detach();//Detach the s object and return it.
	}
	catch(...)
	{
	}
	return blt_in_NULL; //return the build-in NULL object.
}







