/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
*/

// spengcon.cxx: implementation of the CSPengineConnect class.
//
//////////////////////////////////////////////////////////////////////

#include <string>
#include <map>

#ifdef __hppa
#include <rw/exception.h>
#endif

#include "s_newio.h"
#include "spexcept.h"
#include "spengcon.h"
#include "spchar.h"
#include "spcall.h"
#include "spnum.h"
#include "spconnec.h"
#include "notifyc.h"
#include "S_y_tab.h"
#include "S_io.h"
#include "sqpe.h"

#ifndef WIN32
// Needed for UNIX specific code in CSPengineConnect::Create
// dynlink includes dlfcn.h or hpdlfcn.h, as needed.
#include "dynlink.h"
#endif

//************************************************************
// STL
//
#ifndef __hppa
#pragma warning(disable:4786)
#endif

#ifdef __hppa
namespace std{};
#endif

using namespace std;
//************************************************************

//////////////////////////////////////////////////////////////////////
// Initialize
//////////////////////////////////////////////////////////////////////

//Warning: manually maintainant is required below when a new CSPclass is created
//to support class runtime support.  

#define S_CLASS_CALL 1   
#define S_CLASS_CHARACTER 2
#define S_CLASS_NUMERIC 3

/////////////////////////////////////////////////////////////////////////////////////////
// Object map support for use in S_notification to call virtual fn on notification (CRD)

class CSPclassIndexMap : public map<string, int , less<string> >
{
public:
	CSPclassIndexMap() { clear(); };
	virtual ~CSPclassIndexMap() {};

	void Insert(string str, int lIndex)
	{
		insert(value_type(str, lIndex));
	}

	int Find(const char* pszClass)
	{
		int index = 0;
//		string strClass = pszClass;
		iterator pItr = find(pszClass);
		if(pItr != end())
			index = (int) (*pItr).second;
		return index;
	}

};

// Global map of class names to #define IDs
CSPclassIndexMap gSPclassIndexMap;

BOOL CSPengineConnect::IsValid(void) const
{
	BOOL bValid = ( CSPengineConnect::GetEngineConnection() != NULL ) ? TRUE:FALSE;
	return bValid;
}

int CSPengineConnect::Create(int argc, char *argv[], char *dlllist[], unsigned int uSConnectFlags, BOOL bUseNewIo)
{
	int bSuccess = 1;

	try
	{
		// Dont reconnect if we have a connection already (CRD)
		if ( IsValid() )
			return bSuccess;

#ifndef WIN32
		// Make sure Connect/C++ is loaded with the flags that we expect on UNIX
		dlopen("libsconnect.so", S_RTLD_MODE);
#endif
		//Initialize the new IO.
		S_newio_initialize();

		{
			BOOL bReadStdFile[2];
			BOOL bSetConsole = FALSE;

			if (uSConnectFlags & S_CONNECT_DEFAULTAUTO) 
			{
				int iStdConsoleFile[3];

				SpGetStdioConsoleStatus(iStdConsoleFile);

				if( bUseNewIo == TRUE &&
					iStdConsoleFile[0] == 1 &&
					iStdConsoleFile[1] == 1 &&
					iStdConsoleFile[2] == 1 )
				{
					iStdConsoleFile[0] = 0;
					iStdConsoleFile[1] = 0;
					iStdConsoleFile[2] = 0;
					
					bSetConsole = TRUE;
				}

				if (iStdConsoleFile[0] == 0)
					uSConnectFlags |= S_CONNECT_SELFDOFIRST;
					
				bReadStdFile[0] = (BOOL)(iStdConsoleFile[1] == 0);
				bReadStdFile[1] = (BOOL)(iStdConsoleFile[2] == 0);
			}
			else 
			{
				bReadStdFile[0] = (BOOL)(uSConnectFlags & S_CONNECT_READSTDOUT);
				bReadStdFile[1] = (BOOL)(uSConnectFlags & S_CONNECT_READSTDERR);
			}
			
			SpSetupTextOutputRouting((bReadStdFile[0]) ? &m_lOsHdStdout:NULL, 
						 (bReadStdFile[1]) ? &m_lOsHdStderr:NULL);
		}

		CSPobject::SetEngineConnection( this );
		SpSetEventLoopFlag((uSConnectFlags & S_CONNECT_EVENTLOOP) ? TRUE:FALSE);

		char *env_argv[S_CONNECT_MAXARGS];

		SpInitArgs(argc, argv, S_CONNECT_MAXARGS, env_argv);
		if (uSConnectFlags & S_CONNECT_SELFDOFIRST)
			SpAddArg(&argc, env_argv, (char*)"S_CONNECT_SELFDOFIRST=yes");
		if(argc<1)
		{
			argc = 1;
			env_argv[0] = (char*)"ConnectC++Application";
		}
		//Make the connection
		if(!Sqpe(argc, env_argv, dlllist))
			return (0); //Failed

		// Turn on notification
		SNotifyState(1, S_evaluator);

		//Intialize class indexe for used by CSPengineConnect::CreateObject() to create a run-time class

		gSPclassIndexMap.Insert("call", S_CLASS_CALL);
		gSPclassIndexMap.Insert("character", S_CLASS_CHARACTER);
		gSPclassIndexMap.Insert("numeric", S_CLASS_NUMERIC);

	}
	catch(...)
	{	
		bSuccess = 0;
	}

	return bSuccess;
}

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CSPengineConnect::CSPengineConnect():
m_uiRefTopLevelEval(0), m_lOsHdStdout(0), m_lOsHdStderr(0) //standard output and error from the engine
{
}

CSPengineConnect::CSPengineConnect(int argc, char *argv[], char *dlllist[]):
m_uiRefTopLevelEval(0), m_lOsHdStdout(0), m_lOsHdStderr(0) //standard output and error from the engine
{
	Create(argc, argv, dlllist);
}

CSPengineConnect::~CSPengineConnect()
{
	if( IsValid() && Sqpe_GetCurrentEvaluator() != NULL )
		::quit_session(S_QUIT_WITHOUT_EXIT, S_evaluator);

	CSPobject::SetEngineConnection( NULL );
}

//////////////////////////////////////////////////////////////////////
// DLL search list functionality

void CSPengineConnect::DllSearchList_Set( char *dlllist[] )
{
	Sqpe_DllSearchList_Set(dlllist);
}
void CSPengineConnect::DllSearchList_Add( char *pszDllName )
{
	Sqpe_DllSearchList_Add(pszDllName);
}
void CSPengineConnect::DllSearchList_Remove( char *pszDllName )
{
	Sqpe_DllSearchList_Remove(pszDllName);
}
void CSPengineConnect::DllSearchList_RemoveAll()
{
	Sqpe_DllSearchList_RemoveAll();
}

BOOL CSPengineConnect::CloseTopLevelEval(void) const
{
	BOOL bSuccess = TRUE;

	// Invalidate member m_ps_object for all local unnamed objects (CRD)
	CSPobject::UnnamedObjectMap_InvalidateLocalObjects(TRUE);

	try
	{
		if ( S_evaluator->_eval_open )
		{
			eval_close(0, S_evaluator);
		}
	}
	catch(...)
	{
		bSuccess = FALSE;
	}
	return bSuccess;
}

//////////////////////////////////////////////////////////////////////
// SyncParseEval() synchronized parse and eval the given expression

int CSPengineConnect::SyncParseEval(const char *pszExpression, s_object **warningList)
{
	int bSuccess = 1;

	try
	{
		CSPobject spObject = S_SyncParseEval_GetWarnings(pszExpression, warningList);
		bSuccess = spObject.GetPtr() != NULL;

//		if ( bSuccess )
//			spObject.Print();
	}
	catch(...)
	{	
		bSuccess = 0;
	}
	return bSuccess;

}

s_object * CSPengineConnect::SyncParseEvalResult(const char *pszExpression, s_object **warningList)
{
	s_object *resultObj = NULL;
	try
	{
		resultObj = S_SyncParseEval_GetWarnings(pszExpression, warningList);
		if ( !resultObj )
			return resultObj;

		if ( (resultObj->mode != S_MODE_QUIT) && (S_evaluator->_quit_flag != FALSE) )
		{
			try
			{
				try_to_free(resultObj, S_FALSE, Nframe, S_evaluator);
			}
			catch(...)
			{
			}
			resultObj = alcvec(S_MODE_QUIT,0L, S_evaluator);
		}
	}
	catch(...)
	{	
		resultObj = NULL;
	}
	return resultObj;
}


//////////////////////////////////////////////////////////////////////
// Notification handling

int CSPengineConnect::OnAttach( s_object* ps_attached )
{
	SqpeTrace0( "WARNING: in CSPengineConnect::OnAttach()\n" );
	return 1;
}

int CSPengineConnect::OnDetach( s_object* ps_attached )
{
	SqpeTrace0( "WARNING: in CSPengineConnect::OnDetach()\n" );
	return 1;
}

int CSPengineConnect::OnOutput( long iOutputFlags )
{
	SqpeTrace0( "WARNING: in CSPengineConnect::OnOutput()\n" );
	return 1;
}

BOOL CSPengineConnect::ReadStdout(
		char* pszBuffer,          // output data buffer 
		long  nBufferSize         // size of data buffer 
  )
{
	return SqpeReadFromPipe( (void*) GetOSHdStdout(), pszBuffer, nBufferSize);
}

BOOL CSPengineConnect::ReadStderr(
		char* pszBuffer,          // output data buffer 
		long  nBufferSize         // size of data buffer 
  )
{
	return SqpeReadFromPipe( (void*)GetOSHdStderr(), pszBuffer, nBufferSize);
}

s_evaluator* CSPengineConnect::GetCurrentEvaluator(void) const
{
	return S_evaluator;
}

long CSPengineConnect::OnAppNotify(long lMsg, long lArgs, void **ppArgs)
{
	long lResult = 0;
	switch (lMsg)
	{
		case S_MSG_EXCEPTION:
		{
			/* Engine is about to do a long jump.  Set s_object pointers
			to NULL in all unnamed CSpObjects
				*/	
			CSPobject::UnnamedObjectMap_InvalidateLocalObjects(TRUE);
			lResult = 1;
		}	
		break;
				
		case S_MSG_MODIFY:
		{
			if ( lArgs != 4 )
				break;
					
			// Notify the SPL object attached to the s_object
			CSPobject *psCSPobject = CSPobject::ObjectMap_Get( (const char*)ppArgs[0] );
			if ( psCSPobject )
				lResult = psCSPobject->OnModify( (s_object*)ppArgs[1], (s_object*)ppArgs[2], (s_object*)ppArgs[3]);
		}
		break;
				
		case S_MSG_PREMODIFY:
		{
			if ( lArgs != 2 )
				break;
					
			// Notify the SPL object attached to the s_object
			CSPobject *psCSPobject = CSPobject::ObjectMap_Get( (const char*)ppArgs[0] );
			if ( psCSPobject )
				lResult = psCSPobject->OnPreModify( (s_object**)ppArgs[1] );
		}
		break;
				
		case S_MSG_REMOVE:
		{
			if ( lArgs != 3 )
				break;
					
			// Notify the SPL object attached to the s_object
			CSPobject *psCSPobject = CSPobject::ObjectMap_Get( (const char*)ppArgs[0] );
			if ( psCSPobject )
				lResult = psCSPobject->OnRemove( (s_object*)ppArgs[1], (s_object*)ppArgs[2]);				
		}
		break;
				
		case S_MSG_ATTACH:
		{
			if ( lArgs != 1 )
				break;
					
			lResult = OnAttach((s_object*) ppArgs[0] );
		}
		break;
				
		case S_MSG_DETACH:
		{
			if ( lArgs != 1 )
				break;
					
			lResult = OnDetach( (s_object*) ppArgs[0] );
		}
		break;
				
		case S_MSG_TEXTOUT:
		{
			lResult = OnOutput( (long)ppArgs[0] );
		}
		break;

		default:
			lResult = 0;
	}
	return lResult;
}

long CSPengineConnect::OnAppQuery(long lMsg, long lArgs, void **ppArgs)
{
	SqpeTrace0( "WARNING: in CSPengineConnect::OnAppQuery()\n" );
	return 1;
}

long CSPengineConnect::OnEvalInput(long lMsg, long lArgs, void **ppArgs)
{
	SqpeTrace0( "WARNING: in CSPengineConnect::OnEvalInput()\n" );
	return 1;
}

long CSPengineConnect::OnCommandsHistory(long lMsg, long lArgs, void **ppArgs)
{
	SqpeTrace0( "WARNING: in CSPengineConnect::OnCommandsHistory()\n" );
	return 1;
}

long CSPengineConnect::OnPseudoProcess(long lMsg, long lArgs, void **ppArgs)
{
	SqpeTrace0( "WARNING: in CSPengineConnect::OnPseudoProcess()\n" );
	return 1;
}

long CSPengineConnect::OnAppHelp(long lMsg, long lArgs, void **ppArgs)
{
	SqpeTrace0( "WARNING: in CSPengineConnect::OnAppHelp()\n" );
	return 1;
}

long CSPengineConnect::OnSGraphics(long lMsg, long lArgs, void **ppArgs)
{
	SqpeTrace0( "WARNING: in CSPengineConnect::OnSGraphics()\n" );
	return 1;
}

long CSPengineConnect::OnAppSpecial(long lMsg, long lArgs, void **ppArgs)
{
	SqpeTrace0( "WARNING: in CSPengineConnect::OnAppSpecial()\n" );
	return 1;
}

long CSPengineConnect::OnNotifyOrQuery(long lMsg, long lArgs, void** ppArgs)
{
	long lResult = 0;

	try
	{
		if ( lMsg <= S_MSG_SETOPT )
		{
			lResult = OnAppNotify(lMsg, lArgs, ppArgs);
		}
		else	if (S_MSG_QUERYBASE <= lMsg && lMsg <= S_MSG_QUERYLAST) 
		{
			lResult = OnAppQuery(lMsg, lArgs, ppArgs);
		}
		else if (S_MSG_EVALINPUTBASE <= lMsg && lMsg <= S_MSG_EVALINPUTLAST) 
		{
			lResult = OnEvalInput(lMsg, lArgs, ppArgs);
		}
		else if (S_MSG_CMDHISTBASE <= lMsg && lMsg <= S_MSG_CMDHISTLAST) 
		{
			lResult = OnCommandsHistory(lMsg, lArgs, ppArgs);
		}
		else if (S_MSG_PSEUDOPROCBASE <= lMsg && lMsg <= S_MSG_PSEUDOPROCLAST) 
		{
			lResult = OnPseudoProcess(lMsg, lArgs, ppArgs);
		}
		else if (S_MSG_HELPBASE <= lMsg && lMsg <= S_MSG_HELPLAST)
		{
			lResult = OnAppHelp(lMsg, lArgs, ppArgs);
		}
		else if (S_MSG_GRBASE <= lMsg && lMsg <= S_MSG_GRLAST) 
		{
			lResult = OnSGraphics(lMsg, lArgs, ppArgs);
		}
		else if (S_MSG_APPSPECIALBASE <= lMsg && lMsg <= S_MSG_APPSPECIALLAST) 
		{
			lResult = OnAppSpecial(lMsg, lArgs, ppArgs);
		}
		else 
		{
			SqpeTrace0( "WARNING: in default message type case in CSPengineConnect::OnNotifyOrQuery()\n" );
		}
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}

	return lResult;
}

//////////////////////////////////////////////////////////////////////
//Assign an S object to a perm. frame assoc. with a name.
// 1. the given object name is assigned  
// 2. the object is commited to disk
// 3. it is cached in the permanent frame assoc with lDataBase

void CSPengineConnect::Assign(const char* pszName, const CSPobject& spObject, long lDataBase)
{
	try
	{
		if(lDataBase <=  0)
			SCONNECT_ThrowException("Invalid database %d", lDataBase);

		CSPevaluator sEvaluator;
		::db_put_data(pszName, S_evaluator->_search_data_index[lDataBase-1], spObject, S_evaluator);
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}

}

//////////////////////////////////////////////////////////////////////
//Create an S object in a perm. frame and return an CSPobject*
// 1. the default constructor for an SV4 class is called to construct the object
// 2. if pszName is not NULL, assign it via CSPengineConnect::Assign()

s_object* CSPengineConnect::CreateObject(
	const char* pszClass,      //class name 
	const char* pszName,       //object name
	const char* pszExpression, //==NULL, uses the default constructor, else use the expression to instantiate it.
	long lDataBase)            //database
{
	s_object* ps_object  = NULL;

	if(lDataBase <=  0)
		SCONNECT_ThrowException("Invalid database %d", lDataBase);
	try
	{
		CSPevaluator sEvaluator;
		s_class* psClass = make_s_class(pszClass, 0, S_evaluator);			
		if(pszExpression == NULL)
		{
			//Construct a new object with the default constructor of the SV4 class
			ps_object = new_S_object(psClass, S_evaluator);
		}
		else 
		{
			//Eval the expression to S object
			ps_object = S_SyncParseEval(pszExpression);
			//Make sure we coarse it to the desired class
			if(ps_object == NULL)
				SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
			ps_object = sEvaluator.CloneIfNeeded(AS(ps_object, psClass));
		}

		ps_object = sEvaluator.CloneIfNeeded(ps_object);
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
		ps_object = NULL;
	}
	return ps_object;
}

//////////////////////////////////////////////////////////////////////
//Find S object in the permanant frames: the first one in the search list will be returned.

s_object* CSPengineConnect::Find(const char* pszName, BOOL bLocal, BOOL bGetData)
{
	CSPobject spObject;
	try
	{
		CSPevaluator sEvaluator;
		s_object* ps_object = sEvaluator.get(pszName, bLocal, bGetData);
		spObject = sEvaluator.CloneIfNeeded(ps_object);
	}
	catch(...)
	{
		spObject.Attach(NULL);
	}

	return spObject.Detach();
}

//ParseTest() parses to test the string. It returns status result.
//The arg. pnCharParsed is length of char. sucessfully parsed.
int CSPengineConnect::ParseTest(const char* pszExpression, long* plCharParsed)
{
	// unused: int iFrame=-1;
	int nAns = -1; //init to Fail
	try
	{
		CSPevaluator sEvaluator;
		s_object* ps_char = NEW_CHARACTER(1);
		char* pszCopiedExpression = SPL_AllocateStringInFrame(ps_char, pszExpression);
		
		CHARACTER_POINTER(ps_char)[0] = pszCopiedExpression;
		CSPtextConnection stextConnection(ps_char);
		nAns = stextConnection.ParseTest(plCharParsed);
	}
	catch(CSPexception& except)
	{
		nAns = -1;
		except.Print();
	}
	catch(...)
	{
		nAns = -1;
	}

	return nAns;
}

