/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001

	spmatrix.h: interface for the CSPmatrix class.
*/

#if !defined(__SCONNECT_SPMATRIX_H_INCLUDED__)
#define __SCONNECT_SPMATRIX_H_INCLUDED__

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include "speval.h"
#include "spint.h"
#include "spnum.h"
#include "spfunc.h"
#include "spcall.h"
#include "sparray.h"
#include "sconnect_gen.h"

class SCONNECT_LIB_EXTERN CSPmatrix : public CSParray  
{
//////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////
public:
	//Default constructor
	CSPmatrix();
	//Copy constructor: copy appropriate data members
	CSPmatrix(const CSPmatrix& sObject);
	//Construct from an object of the base class
	CSPmatrix(const CSPobject& sObject);
	//Construct from a valid S-expression
	explicit CSPmatrix(const char* pszExpression);
	//Construct from s_object* 
	explicit CSPmatrix(s_object* ps_object, BOOL bTryToFreeOnDetach=FALSE);
	//Assign from the same class
	CSPmatrix& operator=(const CSPmatrix& sObject);
	//Assign from the base class
	CSPmatrix& operator=(const CSPobject& sObject);
	//Assign from an S object
	CSPmatrix& operator=(s_object* ps_object);
	//The destructor
	virtual ~CSPmatrix();

	//Construct an nRow X nCol matrix of mode
	CSPmatrix(long nRow, long nCol, int mode=S_MODE_DOUBLE);
	CSPmatrix(const CSPvector& sVector, long nRow, long nCol);
	CSPmatrix(double* pdValues, long nRow, long nCol);
	//Create a persistent object when pszName!=NULL
	virtual BOOL Create(
		const char* pszExpression = NULL, //==NULL, uses the default constructor, else use the expression to instantiate it.
		const char* pszName = NULL,       //object name
		long lDataBase=1 );               //database, only used if name given
	virtual BOOL Create(long nRow, long nCol, int mode=S_MODE_DOUBLE, const char* pszName=NULL,
								long lDataBase=1);
	virtual BOOL Create(const CSPvector& sVector, long nRow, long nCol, const char* pszName=NULL,
								long lDataBase=1);
	virtual BOOL Create(double* pdValues, long nRow, long nCol, const char* pszName=NULL,
								long lDataBase=1);

//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////
public:
  //Return TRUE if the object is a valid matrix class
	virtual BOOL IsValid(void) const;
	//If not valid throw an exception.
	void Validate(void) const;
	//Attach an S object and coerce to appropriate class that override this method
	virtual void Attach(s_object *ps_object, BOOL bTryToFreeOnDetach=FALSE);
  //Return TRUE if lMode is one of valid data mode
	BOOL IsValidDataMode(long lMode) const;
	//Get number of rows
	virtual long GetNRow(BOOL bValidate=TRUE) const;
	virtual long nrow(BOOL bValidate=TRUE) const {return GetNRow(bValidate);}
	//Get number of columns
	virtual long GetNCol(BOOL bValidate=TRUE) const;
	virtual long ncol(BOOL bValidate=TRUE) const {return GetNCol(bValidate);}
	//Set number of row/col
	virtual BOOL SetNRow(long lNRow, BOOL bValidate=TRUE); 
	virtual BOOL SetNCol(long lNCol, BOOL bValidate=TRUE);
	//Insert a vector, data.frame, or matrix at index: zero-base
	virtual BOOL InsertAt(s_object* ps_object, long lCol, const char* pszName=NULL, BOOL bValidate=TRUE);
	//Insert an object created by an expression at index: zero-base
	virtual BOOL InsertAt(const char* pszExpression, long lCol, const char* pszName=NULL, BOOL bValidate=TRUE);
	virtual BOOL InsertRows(long lRow, const char* pszName=NULL, long nRows=1, BOOL bValidate=TRUE);
	//Remove some columns : zero-base index
	virtual BOOL RemoveAt(long lCol, long nCols=1, BOOL bValidate=TRUE);
	virtual BOOL RemoveAt(const char* pszName, BOOL bValidate=TRUE);
	virtual BOOL RemoveAllCols(BOOL bValidate = TRUE);
	virtual BOOL RemoveRows(long lRow, long nRows=1, BOOL bValidate=TRUE);
	//Get the address to the content of an element, zero-based index.
	void* GetElementPtr(long lRowZeroIndex, long lColZeroIndex, BOOL bValidate=TRUE) const;
	//Return the memory for the column corresponding to lCol: zero-based index.
	void* GetRawColPtrAt(long lCol, BOOL bValidate=TRUE) const; 
	void* GetRawColPtrAt(const char* pszName, BOOL bValidate=TRUE) const;
  //Set the values corrensponding to the column lCol: Zero-based index.
	virtual BOOL SetAt(s_object* ps_object, long lCol, const char* pszName=NULL, BOOL bValidate=TRUE);
	
	//Get a copy of the vector at column lCol: Zero-based index
	//Warning: GetAt() for CSPmatrix differs from many other class. It returns a copy of vector column
	virtual CSPobject GetAt(long lCol) const;
	//Get a copy of the vector at column with name pazName
	//Warning: GetAt() for CSPmatrix differs from many other class. It returns a copy of vector column
	virtual CSPobject GetAt(const char* pszName) const;

	CSPcharacter GetRowNames(BOOL bValidate=TRUE) const;
	CSPcharacter GetColNames(BOOL bValidate=TRUE) const;
	virtual BOOL SetRowNames(s_object* ps_rownames, BOOL bValidate=TRUE);
	virtual BOOL SetColNames(s_object* ps_colnames, BOOL bValidate=TRUE);
	//Almost pure virtual: see the derived template class TSPmatrix
	virtual BOOL SetIthRowDirect(long ithRow /*zero-based*/, const s_object* ps_vector /* in: */) 
	{ return FALSE; }
	virtual BOOL PutIthRowInto(s_object* ps_vResult /*out: must pre-alloc */, long ithRow /*in: zero-based*/) const
	{ return FALSE; }
	//Apply the function to I th row of the matrix and put result in matrix psAns.
	virtual BOOL IthRowApply(CSPmatrix& sAns /*in/out: */, long i /*in: */, const CSPcall& sfunCall /* in:*/)const;
	//Apply an S function to the I th row. pAns will change mode if needed according to the new column
	virtual BOOL IthRowApply(CSPmatrix*& pAns /*in/out: */, long i /*in: */, const CSPcall& sfunCall /* in:*/)const;
	//Apply an S function to every row. 
	virtual CSPobject RowApply(const CSPfunction& sfun, /* in: an S function object */
                             s_object* ps_list3Dots=NULL) const;
	//Apply a C function to every row. Return
	virtual CSPobject RowApply(const char* pszCFun, /* C function name */
                             s_object* ps_list3Dots=NULL) const;
	//Copy content of the Jth column into a vector.
	virtual BOOL PutJthColumnInto(s_object* ps_vResult /*out: must pre-alloc */, long jthCol /*in: zero-based*/) const
	{ return FALSE; }
	//Set content of ps_vector into the .Data directly 
	virtual BOOL SetJthColumnDirect(long jthCol /*in: zero-based*/, const s_object* ps_vector /* in: */) 
	{ return FALSE; }
	//Apply an S function to the Jth column. 
	virtual BOOL JthColumnApply(CSPmatrix& sAns /*in/out: */, long j /*in: */, const CSPcall& sfunCall /* in:*/)const;
	//Apply an S function to the Jth column. pAns will change mode if needed according to the new column
	virtual BOOL JthColumnApply(CSPmatrix*& pAns /*in/out: */, long j /*in: */, const CSPcall& sfunCall /* in:*/)const;

	//Apply an S function to every column. 
	virtual CSPobject ColumnApply(const CSPfunction& sfun, /* in: an S function object */
		                            s_object* ps_list3Dots=NULL /* in: list object such as list(...) */ ) const;
	//Apply a C function to every column. 
	virtual CSPobject ColumnApply(const char* pszCFun, /* C function name */
                                s_object* ps_list3Dots=NULL /* in: list object such as list(...) */) const;
	//Apply a C or S function to every row or column depending of lMargin. 
	virtual CSPobject Apply(long lMargin, /* in: 1 for row-wise, 2 for column wise */
		                      s_object* ps_FUN, /* in: S function obj or char name of S or C function */
													s_object* ps_list3Dots=NULL /* in: list object such as list(...) */) const;
	//Apply a C or S function to every row or column depending of lMargin.
	virtual CSPobject Apply(s_object* ps_Margin, /* in: 1 for row-wsie, 2 for column wise */ 
		                      s_object* ps_FUN, /* in: S function obj or char name of S or C function */
													s_object* ps_list3Dots=NULL /* in: list object such as list(...) */) const;
//////////////////////////////////////////////////////
// Operations
//////////////////////////////////////////////////////
public:
	// transpose
	virtual BOOL Transpose(BOOL bValidate=TRUE);
	//matrix-matrix multiply: %*% 
	virtual const CSPmatrix Multiply(const CSPmatrix& sRhs) const;
	virtual const CSPobject Multiply(const CSPnumeric& sRhs) const;
	const CSPmatrix  operator* (const CSPmatrix& sRhs) const
	{ return Multiply( sRhs); }
	//matrix-vector multiply: %*% : return 'matrix' if sRhs is scalar else  return 'vector' 
	const CSPobject  operator* (const CSPnumeric& sRhs) const
	{ return Multiply( sRhs); }
	//The condition number is computed as norm(A) * norm(inv(A)).
	double ConditionNumber(BOOL bValidate=TRUE) const; 
	//QR decomposition
	virtual CSPmatrix QRLS(s_object* ps_y, double dtol=1.e-7, BOOL bValidate=TRUE) const;
	//cross product: A'* B. if ps_B is NULL, return cross product of this matrix.
	virtual CSPmatrix crossprod(s_object* ps_B=NULL);
	// matrix backsolve : A \ B. If ps_B=NULL, returns inverse of A
	virtual  CSPobject backsolve(s_object* ps_B=NULL, long k=0);

#ifdef WIN32
	#include "spmatrix_com.h"
#endif

}; //End class CSPmatrix

////////////////////////// template class TSPmatrix ////////////////////////////
// TEMPLATE CLASS "matrix" with Required pair of parameters:
// (long, S_MODE_LGL), (long,S_MODE_INT), (float, S_MODE_REAL)
// (double, S_MODE_DOUBLE), (char*, S_MODE_CHAR), (CSPobject, S_MODE_LIST) 

template<class TSP_TYPE, long TSP_MODE>
class SCONNECT_LIB_EXTERN TSPmatrix : public CSPmatrix, protected TSPmodeVector<TSP_TYPE, TSP_MODE>
{
public:
	//Default constructor
	TSPmatrix(){};
	//Copy constructors 
	TSPmatrix(const TSPmatrix<TSP_TYPE, TSP_MODE>& sObject) : CSPmatrix()
	{	
		Attach(sObject, sObject.GetTryToFreeOnDetach()); 
	}
	//Construct from a valid S-object
	TSPmatrix(const CSPobject& sObject)
	{	
		Attach(&sObject, sObject.GetTryToFreeOnDetach());
	}
	//Construct from a valid S-expression
	explicit TSPmatrix(const char* pszExpression) 
	{
		CSPmatrix A(pszExpression);
		Attach(&A, TRUE);
	}
	//Construct from a valid S-object
	explicit TSPmatrix(s_object* ps_object, BOOL bTryToFreeOnDetach=FALSE)
	{	
		Attach(ps_object, bTryToFreeOnDetach);
	}
	//Construct an nRow X nCol matrix of mode
	TSPmatrix(long nRow, long nCol): CSPmatrix(nRow, nCol, TSP_MODE)
	{}
	//Assignment from s_object*
	TSPmatrix& operator=(s_object* ps_object)
	{
		Attach(ps_object, FALSE);
		return *this;
	}
	//The assignment operators
	TSPmatrix& operator=(const CSPmatrix& sObject)
	{
		Attach(&sObject, sObject.GetTryToFreeOnDetach());  
		return *this;
	}
	virtual ~TSPmatrix(){};
//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////
public:
  //Return TRUE if the object is a valid numeric class
	BOOL IsValid(void)const{return(CSPmatrix::IsValid()&&IsMyMode(CSPmatrix::GetData(FALSE)));}
	void Validate(void) const{if (!IsValid()) throw SCONNECT_INVALID_SOBJECT;}
	//Attach an S object and coerce to appropriate class that override this method
	virtual void Attach(s_object *ps_object, BOOL bTryToFreeOnDetach=FALSE)
	{
		if(SPL_NotThere(ps_object))
			CSPmatrix::Attach(NULL, FALSE); 
		CSPmatrix sValue(ps_object);
		try
		{
			CSPevaluator sEvaluator;
			if(!this->IsMyMode(sValue.GetData()))
				sValue.SetDataMode(TSP_MODE);
		}
		catch(...)
		{
			sValue.Attach(NULL);
		}
		CSPmatrix::Attach(sValue.Detach(), bTryToFreeOnDetach); 
	}

	//Get a copy of the vector at column lCol: Zero-based index
	//Warning: GetAt() for CSPmatrix differs from many other class. It returns a copy of vector column
	virtual CSPobject GetAt(long lCol) const
	{
		return CSPmatrix::GetAt( lCol);
	}
	//Get a copy of the vector at column with name pazName
	//Warning: GetAt() for CSPmatrix differs from many other class. It returns a copy of vector column
	virtual CSPobject GetAt(const char* pszName) const
	{
		return CSPmatrix::GetAt(pszName);
	}

	//Return the address to the content of a cell (element), zero-based index.
	const TSP_TYPE GetAt(long lRowZeroIndex, long lColZeroIndex) const
	{
		long nRow = INTEGER_POINTER((*this)->value.tree[ARRAY_DIM_SLOT])[0];
		return MyTypeValue((*this)->value.tree[ARRAY_DATA_SLOT], nRow*lColZeroIndex + lRowZeroIndex);		
	}
	TSP_TYPE& GetAt(long lRowZeroIndex, long lColZeroIndex)
	{
		long nRow = INTEGER_POINTER((*this)->value.tree[ARRAY_DIM_SLOT])[0];
		return MyTypeValue((*this)->value.tree[ARRAY_DATA_SLOT], nRow*lColZeroIndex + lRowZeroIndex);		
	}
	//Return the address to the content of a cell (element), zero-based index.
	const TSP_TYPE* GetElementPtr(long lRowZeroIndex, long lColZeroIndex) const
	{
		long nRow = INTEGER_POINTER((*this)->value.tree[ARRAY_DIM_SLOT])[0];
		return MyTypePointer((*this)->value.tree[ARRAY_DATA_SLOT], nRow*lColZeroIndex + lRowZeroIndex);		
	}
	TSP_TYPE* GetElementPtr(long lRowZeroIndex, long lColZeroIndex)
	{
		long nRow = INTEGER_POINTER((*this)->value.tree[ARRAY_DIM_SLOT])[0];
		return MyTypePointer((*this)->value.tree[ARRAY_DATA_SLOT], nRow*lColZeroIndex + lRowZeroIndex);		
	}
	//Copy content of one row to a vector
	virtual BOOL PutIthRowInto(TSP_TYPE* pResult /*out: must pre-alloc */, long ithRow /*in: zero-based*/) const
	{	
		long nRow=GetNRow(FALSE);
		if( ithRow<0L || nRow<=ithRow )
			return FALSE;

		long jCell=0L;
		long nCol = GetNCol(FALSE);
		const TSP_TYPE* pCell = GetElementPtr(ithRow, 0L);			
		for(long j=0L; j<nCol; ++j)
		{
			pResult[j] = pCell[jCell];
			jCell += nRow;
		}
		return TRUE;
	}
	//Copy content of one row to a vector
	virtual BOOL PutIthRowInto(s_object* ps_vResult /*out: alloc if needed*/, long ithRow /*in: zero-based*/) const
	{
		CSPevaluator sEvaluator;
		long nCol = GetNCol(FALSE);
		if(SPL_NotThere(ps_vResult))
			ps_vResult= sEvaluator.alcvec(TSP_MODE, nCol); 
		else if( GET_LENGTH(ps_vResult)< nCol) 
			SET_LENGTH(ps_vResult, nCol);
		
		if(ps_vResult->mode != TSP_MODE)
			ps_vResult = sEvaluator.coevec(ps_vResult, TSP_MODE);

		PutIthRowInto(const_cast<TSP_TYPE*>(this->MyTypePointer(ps_vResult)), ithRow);

		return TRUE;
	}

	//Set content of pValue into the .Data directly.
	virtual BOOL SetIthRowDirect(long ithRow /*in: zero-based*/, TSP_TYPE* pValue /* in: must be valid*/)
	{
		CSPevaluator sEvaluator;
		long nRow = GetNRow(FALSE);
		long nCol = GetNCol(FALSE);
		
		if( ithRow<0L || nRow<=ithRow )
			return FALSE;

		TSP_TYPE* pCell = GetElementPtr(ithRow, 0L);			
		long jCell=0L;
		for(long j=0L; j<nCol; ++j)
		{
			pCell[jCell] = pValue[j];
			jCell +=nRow;
		}
		return TRUE;
	}

	//Set content of ps_vector into the .Data directly 
	//Warning: pValue must have valid length
	virtual	BOOL SetIthRowDirect(long ithRow /*in: zero-based*/, const s_object* ps_vector /* in: */)
	{
		CSPevaluator sEvaluator;
		if(this->IsMyMode(ps_vector))
		{
			if(GetNCol(FALSE) != GET_LENGTH(ps_vector)) 
				return FALSE;
			SetIthRowDirect(ithRow, reinterpret_cast<TSP_TYPE*>(RAW_POINTER(ps_vector)));
			return TRUE;
		}

		CSPvector sVector(const_cast<s_object*>(ps_vector));
		s_object* ps_new_vector= NULL;
		if(CSPmatrix::IsValidDataMode(sVector.GetMode()))
			ps_new_vector = this->AsMyMode(&sVector);
		else if(sVector.GetMode(FALSE)==S_MODE_STRUCTURE)
			ps_new_vector = DATA_SLOT(&sVector);
		else
		{
			ps_new_vector = AS_VECTOR(&sVector);
			if(!this->IsMyMode(ps_new_vector))
			{
				s_object* ps_temp = ps_new_vector;
				ps_new_vector = this->AsMyMode(ps_temp);
				sEvaluator.TryToFree(ps_temp, S_TRUE);
			}
		}

		if(GetNCol(FALSE) != GET_LENGTH(ps_new_vector)) 
			return FALSE;

		SetIthRowDirect(ithRow, reinterpret_cast<TSP_TYPE*>(RAW_POINTER(ps_new_vector)));
		sEvaluator.TryToFree(ps_new_vector, S_TRUE);
		return TRUE;
	}	
	//Copy content of one column to a vector
	virtual BOOL PutJthColumnInto( TSP_TYPE* pResult /*out: must pre-alloc */, long jthCol /*in: zero-based*/) const
	{		
		if( jthCol<0L || GetNCol(FALSE) <= jthCol )
			return FALSE;

		const TSP_TYPE* pCell = GetElementPtr(0L, jthCol);			
		long i=GetNRow(FALSE);
		while(--i>= 0L)
			pResult[i] = pCell[i]; //Maybe faster with memcpy, but this is safer!
		return TRUE;
	}

	//Copy content of one row to a vector
	virtual BOOL PutJthColumnInto(s_object* ps_vResult /*out: alloc if needed*/, long jthCol /*in: zero-based*/) const
	{
		CSPevaluator sEvaluator;
		long nRow = GetNRow(FALSE);
		if(SPL_NotThere(ps_vResult))
			ps_vResult= sEvaluator.alcvec(TSP_MODE, nRow); 
		else if( GET_LENGTH(ps_vResult)< nRow) 
			SET_LENGTH(ps_vResult, nRow);		

		if(!this->IsMyMode(ps_vResult))
			ps_vResult = this->AsMyMode(ps_vResult);
		PutJthColumnInto(const_cast<TSP_TYPE*>(this->MyTypePointer(ps_vResult)), jthCol);
		return TRUE;
	}

	//Set content of ps_vector into the .Data directly .
	virtual BOOL SetJthColumnDirect(long jthCol /*in: zero-based*/,  TSP_TYPE* pValue /* in: must be valid*/)
	{		
		if( jthCol<0L || GetNCol(FALSE)<=jthCol )
			return FALSE;

		TSP_TYPE* pCell = GetElementPtr(0L, jthCol);			
		long i=GetNRow(FALSE);
		while(--i>= 0L)
			pCell[i] = pValue[i]; //Maybe faster with memcpy, but this is safer!
		return TRUE;
	}
	//Set content of ps_vector into the .Data directly 
	//Warning: pValue must have valid length
	virtual BOOL SetJthColumnDirect(long jthCol /*in: zero-based*/, const s_object* ps_vector /* in: */)
	{
		CSPevaluator sEvaluator;

		if(this->IsMyMode(ps_vector))
		{
			if(GetNRow(FALSE) != GET_LENGTH(ps_vector)) 
				return FALSE;
			SetJthColumnDirect(jthCol, reinterpret_cast<TSP_TYPE*>(RAW_POINTER(ps_vector)));
			return TRUE;
		}
		CSPvector sVector(const_cast<s_object*>(ps_vector));
		s_object* ps_new_vector= NULL;
		if(CSPmatrix::IsValidDataMode(sVector.GetMode()))
			ps_new_vector = this->AsMyMode(&sVector);
		else if(sVector.GetMode(FALSE)==S_MODE_STRUCTURE)
			ps_new_vector = DATA_SLOT(&sVector);
		else
		{
			ps_new_vector = AS_VECTOR(&sVector);
			if(!this->IsMyMode(ps_new_vector))
			{
				s_object* ps_temp = ps_new_vector;
				ps_new_vector = this->AsMyMode(ps_temp);
				sEvaluator.TryToFree(ps_temp, S_TRUE);
			}
		}
		if(GetNRow(FALSE) != GET_LENGTH(ps_new_vector)) 
			SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
		SetJthColumnDirect(jthCol, reinterpret_cast<TSP_TYPE*>(RAW_POINTER(ps_new_vector)));
		sEvaluator.TryToFree(ps_new_vector, S_TRUE);
		return TRUE;
	}

	//A vector class to temporary hold the data slot in assisting the (i, j) operator

	class SCONNECT_LIB_EXTERN CSPvectorData : public TSPvector<TSP_TYPE, TSP_MODE>
	{
	public:
		//Default counstructor
		CSPvectorData():m_pmParent(NULL)
		{};
		//Copy constructors 
		CSPvectorData(TSPmatrix<TSP_TYPE, TSP_MODE>* pmParent): 
		m_pmParent(pmParent)
		{};
		virtual ~CSPvectorData()
		{ 
			this->Detach();
			this->m_pmParent = NULL;
		}
		//Only these friends can access private members
		friend class TSPmatrix<TSP_TYPE, TSP_MODE>;
	private:
		void SetParent(TSPmatrix<TSP_TYPE, TSP_MODE>* pmParent)
		{
			m_pmParent = pmParent;
			Attach( DATA_SLOT(&(*pmParent)));  //no checking assume always valid
		}
		virtual void SetAt(long lZeroIndex, const TSP_TYPE rhs /* TSP_TYPE& rhs?*/, BOOL bValidate=TRUE)
		{
			CSPevaluator sEvaluator;
			s_object* ps_data = &(*this);
			if(bValidate)
			{
			        this->Validate();
				if(lZeroIndex < 0L )
					SCONNECT_ThrowException("Index is out of range: %d", lZeroIndex);
				else if(this->GetLength(FALSE) <= lZeroIndex)
					SET_LENGTH(ps_data, lZeroIndex+1L);
			}			

			if(sEvaluator.IsSharing(ps_data))
				ps_data = sEvaluator.CopyForWrite(ps_data);
			if(TSP_MODE==S_MODE_CHAR)
				S_set_arena_why(ps_data, ATOMIC_WHY, sEvaluator.GetPtr());
			this->MyTypeValue(ps_data, lZeroIndex)= rhs;

			if(&(*this) != ps_data) //new data?
			{
				s_object* ps_new_parent = sEvaluator.CopyForWrite(&(*m_pmParent));
				SET_ELEMENT(ps_new_parent, 0L, ps_data);
				//Inefficient but necessary for permanent and persistent (with name) objects
				m_pmParent->ReAttachAndAssign(ps_new_parent);
			}
		}
		//matrix parent of this vector
		TSPmatrix<TSP_TYPE, TSP_MODE>* m_pmParent; 
	};
	//...continue TSPmatrix

	//Indexing: Fortran style, one-based (i, j)

	const TSP_TYPE operator()(int nRowOneBasedIndex, int nColOneBasedIndex) const
	{
		return GetAt(nRowOneBasedIndex-1, nColOneBasedIndex-1);
	}
	const TSP_TYPE operator()(long lRowOneBasedIndex, long lColOneBasedIndex) const
	{
		return GetAt(lRowOneBasedIndex-1, lColOneBasedIndex-1);
	}

	//Warning: l-value can be modified.  For example, A(i,j)= rhs would change the content of A
	//without copy first if needed.

	TSP_TYPE& operator()(int nRowOneBasedIndex, int nColOneBasedIndex) 
	{
		return GetAt(nRowOneBasedIndex-1, nColOneBasedIndex-1);
	}
	TSP_TYPE& operator()(long lRowOneBasedIndex, long lColOneBasedIndex) 
	{
		return GetAt(lRowOneBasedIndex-1, lColOneBasedIndex-1);
	}

	friend class CSPvectorData;
private:
	CSPvectorData m_ps_vData; //temp variable used for indexing via operator() above
}; //End TSPmatrix

//typedef for convenient
typedef TSPmatrix<long,      S_MODE_LGL    > CSPlogicalMatrix;
typedef TSPmatrix<long,      S_MODE_INT    > CSPintegerMatrix;
typedef TSPmatrix<float,     S_MODE_REAL   > CSPsingleMatrix;
typedef TSPmatrix<s_complex, S_MODE_COMPLEX> CSPcomplexMatrix;
typedef TSPmatrix<char,      S_MODE_RAW    > CSPrawMatrix;

///////////////////////////////////////////////////////////////////
////////////////// Matrix of character (double) /////////////////////

class SCONNECT_LIB_EXTERN CSPcharacterMatrix : public TSPmatrix<char*, S_MODE_CHAR>
{
//////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////
public:
	//Default constructor
	CSPcharacterMatrix();
	//Copy constructor: copy appropriate data members
	CSPcharacterMatrix(const CSPcharacterMatrix& sObject);
	//Construct from object of a base class
	CSPcharacterMatrix(const CSPobject& sObject);
	//Construct from a valid S-expression
	explicit CSPcharacterMatrix(const char* pszExpression);
	//Construct from a valid S-object
	explicit CSPcharacterMatrix(s_object* ps_object, BOOL bTryToFreeOnDetach=FALSE);
	//Assign from the same class
	CSPcharacterMatrix& operator=(const CSPcharacterMatrix& sObject);
	//Assign from the base class
	CSPcharacterMatrix& operator=(const CSPobject& sObject);
	//Assign from an S object
	CSPcharacterMatrix& operator=(s_object* ps_object);
	//The destructor
	virtual ~CSPcharacterMatrix();

	//Construct an nRow X nCol matrix of mode
	CSPcharacterMatrix(long nRow, long nCol);
	CSPcharacterMatrix(const CSPvector& sVector, long nRow, long nCol);
	CSPcharacterMatrix(double* pdValues, long nRow, long nCol);

//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////
public:
	//Override so char strings can be properly handled.  Note: must override all overloaded member functions.
	virtual BOOL PutIthRowInto(char** pResult /*out: must pre-alloc */, long ithRow /*in: zero-based*/) const;
	virtual BOOL PutIthRowInto(s_object* ps_vResult /*out: alloc if needed*/, long ithRow /*in: zero-based*/) const;
	virtual BOOL PutJthColumnInto( char** pResult /*out: must pre-alloc */, long jthCol /*in: zero-based*/) const;
	virtual BOOL PutJthColumnInto(s_object* ps_vResult /*out: alloc if needed*/, long jthCol /*in: zero-based*/) const;
	virtual BOOL SetIthRowDirect(long ithRow /*in: zero-based*/, char** pValue /* in: assume valid*/);
	virtual	BOOL SetIthRowDirect(long ithRow /*in: zero-based*/, const s_object* ps_vector /* in: */);
	virtual BOOL SetJthColumnDirect(long jthCol /*in: zero-based*/,  char** pValue /* in: assume valid*/);
	virtual BOOL SetJthColumnDirect(long jthCol /*in: zero-based*/, const s_object* ps_vector /* in: */);

}; //End class CSPcharacterMatrix

///////////////////////////////////////////////////////////////////
////////////////// Matrix of numeric (double) /////////////////////

class SCONNECT_LIB_EXTERN CSPnumericMatrix : public TSPmatrix<double, S_MODE_DOUBLE>
{

//////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////
public:
	//Default constructor
	CSPnumericMatrix();
	//Copy constructor: copy appropriate data members
	CSPnumericMatrix(const CSPnumericMatrix& sObject);
	//Construct from object of a base class
	CSPnumericMatrix(const CSPobject& sObject);
	//Construct from a valid S-expression
	explicit CSPnumericMatrix(const char* pszExpression);
	//Construct from a valid S-object
	explicit CSPnumericMatrix(s_object* ps_object, BOOL bTryToFreeOnDetach=FALSE);
	//Assign from the same class
	CSPnumericMatrix& operator=(const CSPnumericMatrix& sObject);
	//Assign from the base class
	CSPnumericMatrix& operator=(const CSPobject& sObject);
	//Assign from an S object
	CSPnumericMatrix& operator=(s_object* ps_object);
	//The destructor
	virtual ~CSPnumericMatrix();

	//Construct an nRow X nCol matrix of mode
	CSPnumericMatrix(long nRow, long nCol);
	CSPnumericMatrix(const CSPvector& sVector, long nRow, long nCol);
	CSPnumericMatrix(double* pdValues, long nRow, long nCol);

//////////////////////////////////////////////////////
// Operations
//////////////////////////////////////////////////////
public:
	//matrix-matrix multiply: %*% 
	virtual const CSPmatrix Multiply(const CSPmatrix& sRhs) const;
	//matrix-vector multiply: %*% : return 'matrix' if sRhs is scalar else  return 'vector' 
	virtual const CSPobject Multiply(const CSPnumeric& sRhs) const;
	//Call DGECON(LAPACK) to estimate the reciprocal of the condition number of a general
	// real matrix A, in either the 1-norm or the infinity-norm, using
	// the LU factorization computed by DGETRF.
	// The condition number is computed as norm(A) * norm(inv(A)).
	double ConditionNumber(BOOL bValidate=TRUE) const; 
	//QR decomposition
	virtual CSPmatrix QRLS(s_object* ps_y, double dtol=1.e-7, BOOL bValidate=TRUE) const;
	//cross product: A'* B. if ps_B is NULL, return cross product of this matrix.
	virtual CSPmatrix crossprod(s_object* ps_B=NULL);
	// matrix backsolve : A \ B. If ps_B=NULL, returns inverse of A
	virtual CSPobject backsolve(s_object* ps_B=NULL, long k=0);

#if (_MSC_VER > 1300)
	const double operator()(int nRowOneBasedIndex, int nColOneBasedIndex) const
	{
		return GetAt(nRowOneBasedIndex-1, nColOneBasedIndex-1);
	}
	const double operator()(long lRowOneBasedIndex, long lColOneBasedIndex) const
	{
		return GetAt(lRowOneBasedIndex-1, lColOneBasedIndex-1);
	}
	double& operator()(int nRowOneBasedIndex, int nColOneBasedIndex) 
	{
		return GetAt(nRowOneBasedIndex-1, nColOneBasedIndex-1);
	}
	double& operator()(long lRowOneBasedIndex, long lColOneBasedIndex) 
	{
		return GetAt(lRowOneBasedIndex-1, lColOneBasedIndex-1);
	}
#endif // (_MSC_VER > 1300)

}; //End class CSPnumericMatrix

///////////////////////////////////////////////////////////////
// Global "double" matrix operations
//matrix multipy : *psC = sA sB, where sA and sB are matrices
SCONNECT_DLLAPI(CSPobject) SPL_Multiply(const CSPmatrix& sA, const CSPmatrix& sB);
//matrix-vector multiply : *psy = sA sx, where sA is a matrix and sx is a vector
//If length of sx is one, matrix-numeric multiply. 
SCONNECT_DLLAPI(CSPobject) SPL_Multiply(const CSPmatrix& sA, const CSPnumeric& sx);
////////////////////////////Export as extern "C" /////////////////////////////////
extern "C"
{
	//Turn on/off usage of S_matrix_apply().
	SCONNECT_DLLAPI(void) s_set_matrix_apply(long *lFlag);
	//Compute sum of a numeric vector (double or integer)
	SCONNECT_DLLAPI(s_object*) S_numeric_sum(s_object* ps_v);
	//Apply a function (S or C) to every row or col of a matrix which is similar to q_apply().
	SCONNECT_DLLAPI(s_object*) S_matrix_apply(s_object* , s_object* , s_object*, s_object* );
}
#endif // !defined(__SCONNECT_SPMATRIX_H_INCLUDED__)
