/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001

	spvector.h: interface for the CSPvector class.
*/
#if !defined(__SCONNECT_SPVECTOR_H_INCLUDED__)
#define __SCONNECT_SPVECTOR_H_INCLUDED__

#if _MSC_VER > 1000
#pragma once
//supress warning C4275: non dll-interface class 'TSPvector<double,4>' 
//used as base for dll-interface class 'CSPnumeric': TSPvector<...> and its base are in header files.
#pragma warning(disable: 4275)
#endif // _MSC_VER > 1000
#include "spobject.h"
#include "spalcfrm.h" 
class SCONNECT_LIB_EXTERN CSPvector : public CSPobject  
{
public:
	//Default constructor
	CSPvector();
	//Construct from object of a base class
	CSPvector(const CSPobject& sObject);
	//Create a new object of some length=length
	CSPvector(int mode, long lLength); 
	virtual ~CSPvector();
//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////
public:
	//Set/Get length
	virtual void SetLength(long lLength, BOOL bValidate=TRUE);
	virtual long GetLength(BOOL bValidate=TRUE) const;
	long length(BOOL bValidate=TRUE) const {return GetLength(bValidate);}	
	//Return the address to the content of an element, Fotran style and one-index base
	virtual void* GetRawPtr(BOOL bValidate=TRUE) const;
	//Warning: void* pRhs is casted to the type of this object.
	void  SetElementPtr(long lZeroIndex, long lRhsMode, const void* pRhs, BOOL bValidate=TRUE);
	void* GetElementPtr(long lZeroIndex, BOOL bValidate=TRUE) const;
	int GetElementSize();
	// Non-destructive way to change the length of a vector.
	virtual BOOL ChangeLength( long lNewLength, BOOL bValidate=TRUE );
	// TODO: debricate InsertRows after GUI calls are changed to InsertElements
	virtual BOOL InsertRows(long lRow, long nRows=1, BOOL bValidate=TRUE);
	virtual BOOL RemoveRows(long lRow, long nRows=1, BOOL bValidate=TRUE);
	//Get number of rows
	virtual long GetNRow(BOOL bValidate=TRUE) const;
	virtual long nrow(BOOL bValidate=TRUE) const {return GetNRow(bValidate);}
	//Get number of columns
	virtual long GetNCol(BOOL bValidate=TRUE) const;
	virtual long ncol(BOOL bValidate=TRUE) const {return GetNCol(bValidate);}

#ifdef WIN32
	#include "spvector_com.h"
#endif
};

////////////////////////// template class TSPmatrix ////////////////////////////
// TEMPLATE CLASS with Required pair of parameters:
// (long, S_MODE_LGL), (long,S_MODE_INT), (float, S_MODE_REAL)
// (double, S_MODE_DOUBLE), (char*, S_MODE_CHAR), (CSPobject, S_MODE_LIST) 
template<class TSP_TYPE, long TSP_MODE>
class TSPmodeVector 
{
public:
	//Default constructor
	TSPmodeVector(){};
	virtual ~TSPmodeVector(){};
//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////
public:
	//Unify the macros NEW_TYPE (e.g. NEW_NUMERIC) to one method where TYPE is NUMERIC, INTEGER, etc...
	s_object* NewMyType(long lLength=1L) const
	{
		CSPallocFrame sAllocFrame; //Set alloc frame to perm if needed.	
		return sAllocFrame.alcvec(TSP_MODE, lLength);
	}
	//Unify the macros TYPE_VALUE (e.g. NUMERIC_VALUE) to one method where TYPE is NUMERIC, INTEGER, etc...
	const TSP_TYPE MyTypeValue(const s_object* ps_vector, long lZeroIndex=0L) const
	{
		if(ps_vector==NULL || lZeroIndex < 0L || ps_vector->length <= lZeroIndex)
			SCONNECT_ThrowException("Index out of range");
		const TSP_TYPE* pTypePtr = reinterpret_cast<TSP_TYPE*>(ps_vector->value.name);
		return pTypePtr[lZeroIndex];
	}
	TSP_TYPE& MyTypeValue(s_object* ps_vector, long lZeroIndex=0L) //Good for lvalue
	{
		if(ps_vector==NULL || lZeroIndex < 0L || ps_vector->length <= lZeroIndex)
			SCONNECT_ThrowException("Index out of range");
		TSP_TYPE* pTypePtr = reinterpret_cast<TSP_TYPE*>(ps_vector->value.name);
		return pTypePtr[lZeroIndex];
	}
	//Unify the macros TYPE_POINTER (e.g. NUMERIC_POINTER) to one method where TYPE is NUMERIC, INTEGER, etc...
	const TSP_TYPE* MyTypePointer(const s_object* ps_vector, long lZeroIndex=0L) const 
	{
		if(ps_vector==NULL || lZeroIndex < 0L || ps_vector->length <= lZeroIndex)
			SCONNECT_ThrowException("Index out of range");
		TSP_TYPE* pTypePtr = reinterpret_cast<TSP_TYPE*>(ps_vector->value.name);
		return &pTypePtr[lZeroIndex];
	}
	TSP_TYPE* MyTypePointer(s_object* ps_vector, long lZeroIndex=0L) //Good for lvalue
	{
		assert(ps_vector!=NULL);
		if(lZeroIndex < 0L || ps_vector->length <= lZeroIndex)
			SCONNECT_ThrowException("Index out of range");
		TSP_TYPE* pTypePtr = reinterpret_cast<TSP_TYPE*>(ps_vector->value.name);
		return &pTypePtr[lZeroIndex];
	}
	//Unify the macros IS_TYPE (e.g. IS_NUMERIC) to one method where TYPE is NUMERIC, INTEGER, etc...
	BOOL IsMyType(const s_object* ps_vector) const
	{
		if(!ps_vector)
			return FALSE;
		return IS(const_cast<s_object*>(ps_vector), MyClass());
	}
	//A stricker test than IsMyType
	BOOL IsMyMode(const s_object* ps_vector) const 
	{
		if(!ps_vector)
			return FALSE;
		return (ps_vector->mode == TSP_MODE);
	}
	//Unify the macros AS_TYPE (e.g. AS_NUMERIC) to one method where TYPE is NUMERIC, INTEGER, etc...
	s_object* AsMyType(s_object* ps_vector) const
	{
		s_object* ps_value = NULL;
		CSPevaluator sEvaluator;
		if(SPL_NotThere(ps_vector))
			ps_value = NULL;
		else if(IsMyMode(ps_vector))
			ps_value = ps_vector;
		else if(sEvaluator.GetRefCount(ps_vector) == 0 && !PRECIOUS(ps_vector))
			ps_value = sEvaluator.coevec(ps_vector, TSP_MODE);
		else
			ps_value = sEvaluator.coevec(sEvaluator.CopyForWrite(ps_vector), TSP_MODE);
		return sEvaluator.CloneIfNeeded(ps_value);
	}
	//Same as AsMyType
	s_object* AsMyMode(s_object* ps_vector) const
	{
		return AsMyType(ps_vector);
	}
	//Return S class associate with S mode.  This is strickly based on mode asuming 1-to-1 mapping
	s_class* MyClass(void) const
	{
		switch(TSP_MODE)
		{
		case S_MODE_LGL:				
			return s_logical_class;
		case S_MODE_INT:
			return s_integer_class;
		case S_MODE_REAL:
			return s_single_class;
		case S_MODE_DOUBLE:
			return s_numeric_class;
		case S_MODE_CHAR:
			return s_character_class;
		case S_MODE_LIST:
			return s_list_class;
		case S_MODE_COMPLEX:
			return s_complex_class;
		case S_MODE_RAW:
			return s_raw_class;
		default:
			SCONNECT_ThrowException("Unknown class for mode %d", TSP_MODE);
		}
		return s_NULL_class;
	}
}; //TSPabstractVector 

template<class TSP_TYPE, long TSP_MODE>
class TSPvector : public CSPvector, protected TSPmodeVector<TSP_TYPE, TSP_MODE>
{
public:
	//Default constructor
	TSPvector(){};
	//Copy constructors 
	TSPvector(const TSPvector<TSP_TYPE, TSP_MODE>& sObject) : CSPvector()
	{	
		Attach(sObject, sObject.GetTryToFreeOnDetach()); 
	}
	//Construct from a valid S-object
	TSPvector(const CSPobject& sObject)
	{	
		Attach(&sObject, sObject.GetTryToFreeOnDetach());
	}
	//Construct from a valid S-expression
	explicit TSPvector(const char* pszExpression) 
	{
		CSPevaluator sEvaluator;
		s_object* ps_value = sEvaluator.Eval(pszExpression);
		Attach(sEvaluator.CloneIfNeeded(ps_value), TRUE);
	}
	//Construct from a valid S-object
	explicit TSPvector(s_object* ps_object, BOOL bTryToFreeOnDetach=0)
	{	
		Attach(ps_object, bTryToFreeOnDetach);
	}
	//The assignment operators
	TSPvector& operator=(const CSPobject& sObject)
	{
		Attach(&sObject, TRUE);
		return *this;
	}
	TSPvector& operator=(const TSPvector<TSP_TYPE, TSP_MODE>& sObject)
	{
		Attach(&sObject, TRUE);
		return *this;
	}
	TSPvector& operator=(s_object* ps_object)
	{
		Attach(ps_object, FALSE);
		return *this;
	}
	virtual ~TSPvector()
	{
	}
	//Construct an nRow X nCol vector of mode
	explicit TSPvector(long lLength): CSPvector(TSP_MODE, lLength)
	{
	}
//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////
public:
	//Proxy to assist indexing operator for lvalue/rvalue, type conversions and ref count.
	class CSPvectorProxy 
	{
	public:
		CSPvectorProxy(TSPvector<TSP_TYPE, TSP_MODE>& sVector, long lZeroBasedIndex)
		:m_vector(sVector), m_lIndex(lZeroBasedIndex)
		{
		}		
		CSPvectorProxy(const CSPvectorProxy& proxyRhs)
		:m_vector(proxyRhs.m_vector), m_lIndex(proxyRhs.m_lIndex)
		{
		}; 
		operator TSP_TYPE() const //Casting operator: good for rvalue
		{
			return m_vector.GetAt(m_lIndex, FALSE);
		}		
		TSP_TYPE* operator&() //Address of
		{
			return m_vector.GetElementPtr(m_lIndex);
		}
		long GetMode(void) const
		{
			return m_vector.GetMode();
		}
		// Assignment operators
		CSPvectorProxy& operator =(const CSPvectorProxy& proxyRhs)
		{
			void* pRhs = (void*)proxyRhs.m_vector.GetElementPtr(proxyRhs.m_lIndex);
			m_vector.SetElementPtr(m_lIndex, proxyRhs.GetMode(), pRhs, FALSE);
			return *this;
		}
		CSPvectorProxy& operator =(const TSP_TYPE& rhs)
		{
			m_vector.SetAt(m_lIndex, rhs, FALSE);
			return *this;
		}

		TSPvector<TSP_TYPE, TSP_MODE>& m_vector; //The real object this proxy refers to.
		long m_lIndex;                           //Zero based index to the real object.
	};
	//TSPvector continue ... 
	TSPvector& operator=(const CSPvectorProxy& proxyRhs)
	{
		TSP_TYPE element = (TSP_TYPE) proxyRhs.m_vector.GetAt(proxyRhs.m_lIndex);
		s_object* ps_value = this->NewMyType(1L);
		this->MyTypeValue(ps_value, 0) = element;
		ReAttachAndAssign(ps_value);
		return *this;

	}
	//Subscripting, C style zero-based and Fortran one-based-indexing: a derived class must over-load all required [] and ().
	//Not virtual over-loaded operators so the derived classes can return different proxies
	const TSP_TYPE operator[](int nZeroBasedIndex) const //rvalue only
	{ return MyTypeValue(&(*this), nZeroBasedIndex);}
	CSPvectorProxy operator[](int nZeroBasedIndex)//both lvalue and rvalue     
	{ return CSPvectorProxy(*this, nZeroBasedIndex);}
	//Subscripting, Fortran style and one-based-indexing
	const TSP_TYPE operator()(int nOneBasedIndex) const 
	{ return MyTypeValue(&(*this), nOneBasedIndex-1L);}
	CSPvectorProxy operator()(int nOneBasedIndex)            
	{ return CSPvectorProxy(*this, nOneBasedIndex-1);}
	//long arguments are needed to avoid ambiguity between int and operator s_object*()
	const TSP_TYPE operator[](long lZeroBasedIndex) const 
	{	return MyTypeValue(&(*this), lZeroBasedIndex);}
	CSPvectorProxy operator[](long lZeroBasedIndex)            
	{	return CSPvectorProxy(*this, lZeroBasedIndex);}
	const TSP_TYPE operator()(long lOneBasedIndex) const 
	{ return MyTypeValue(&(*this), lOneBasedIndex-1L);}
	CSPvectorProxy operator()(long lOneBasedIndex)            
	{ return CSPvectorProxy(*this, lOneBasedIndex-1);}
  //Return TRUE if the object is a valid numeric class
	BOOL IsValid(void) const
	{
		return (CSPvector::IsValid() && IsMyMode(&(*this)));
	}
	void Validate(void) const
	{
		if (!IsValid())
			throw SCONNECT_INVALID_SOBJECT;
	}
	//Attach an S object and coerce to appropriate class that override this method
	virtual void Attach(s_object *ps_object, BOOL bTryToFreeOnDetach=FALSE)
	{
		CSPvector::Attach(this->AsMyType(ps_object), bTryToFreeOnDetach); 
	}
	// Set to desired length
	virtual void SetLength(long lLength, BOOL bValidate=FALSE)
	{
		if(bValidate)
			Validate();  
		if( GetPtr() == NULL ) 
			ReAttachAndAssign(this->NewMyType(lLength));
		else
			CSPvector::SetLength(lLength, bValidate);
	}
	//Return the address to the content of a cell (element)
	const TSP_TYPE* GetElementPtr(long lZeroIndex) const
	{
		return MyTypePointer(&(*this), lZeroIndex);
	}
	TSP_TYPE* GetElementPtr(long lZeroIndex) 
	{
		return MyTypePointer(&(*this), lZeroIndex);
	}
	//Access to elements of this vector (ZERO-BASED INDEX). 
	const TSP_TYPE GetAt(long lZeroIndex, BOOL bValidate=TRUE) const 
	{
		if(bValidate)
		{
			Validate();
			if(lZeroIndex < 0L || GetLength(FALSE)  <= lZeroIndex)
				SCONNECT_ThrowException("Index is out of range: %d", lZeroIndex);
		}
		return MyTypeValue(&(*this), lZeroIndex);
	}
	TSP_TYPE& GetAt(long lZeroIndex, BOOL bValidate=TRUE) //Good for lvalue
	{
		if(bValidate)
		{
			Validate();
			if(lZeroIndex < 0L || GetLength(FALSE)  <= lZeroIndex)
				SCONNECT_ThrowException("Index is out of range: %d", lZeroIndex);
		}
		return MyTypeValue(&(*this), lZeroIndex);
	}

	void SetAt(long lZeroIndex, const TSP_TYPE rhs /* TSP_TYPE& rhs?*/, BOOL bValidate=TRUE)
	{
		CSPvector::SetElementPtr(lZeroIndex, TSP_MODE, (const void*)&rhs, bValidate);
	}
//////////////////////////////////////////////////////
// Operations
//////////////////////////////////////////////////////

	friend class CSPvectorProxy;
}; //End of TSPvector

#ifdef __cplusplus
extern "C" {
#endif
SCONNECT_DLLAPI(s_object *) SPL_GenerateVector( int nMode, long nLength );
SCONNECT_DLLAPI(s_object *) SPL_GenerateVectorFromDataArray( int nMode, long nLength, void *vpDataArray );

#ifdef __cplusplus
}
#endif

#endif // !defined(__SCONNECT_SPVECTOR_H_INCLUDED__)
