package Splus::HELPINSTALL;
use v5.8;
use strict ;

=head1 NAME

  Splus::HELPINSTALL - install S code files to a given package/.Data directory

=head1 SYNOPSIS

  The following is the plan.  Only the ones before the bar are implemented.

  use Splus::HELPINSTALL;
  $sh = Splus::HELPINSTALL->new(@ARGV) ; # args: DESTDIR file1.q file2.R ...
  $sh->make() ; # do the installation
  --- bar ---
  # XXX $sh->make_virgin() ; # remove all generated files, including destdir
  # XXX $sh->make_from_scratch(); # make_virgin then make
  $sh->add_source_files("foo.Rd"); # add source file to list
  $sh->delete_source_files("foo.Rd"); # remove source file from list
  $sh->get_source_files(); # return list of *.q *.R files
  $sh->tidy_after_build() ; # do nothing.  Supplied to be like SHLIB.

=cut

use Splus::Vars;
use Splus::Utils;
use Splus::SplusUtils;
use File::Basename;
use File::Copy;
use File::Path;
use Cwd qw(getcwd abs_path) ;
use R::Rdconv;

my $ostype = ${Splus::Vars::OSTYPE} ; # "windows" or "unix" (use ^O to distinguish unixen)

sub _Syntax {
    print "-h,--help          # print this help file\n";
    print "--destdir dirName  # install to directory dirName (don't include .Data)\n";
    print "--clean-first      # remove all installed help files before reinstalling\n";
    print "--no-examples      # do not extract example code into R-ex directory\n";
    print "The current working directory must be the package's man directory.\n";
    print "All .Rd files in current directory will be installed.\n";
    print "On Unix you can use file1.Rd file2.Rd to install just a subset of them.\n";
    print "On Windows the --destdir argument is ignored; the .chm file is put in the directory one level above the working directory.\n";
    @_ && die join(", ", @_) ;
}

sub new {
    # Act like old Unix HINSTALL : HINSTALL DESTDIR file1.sgml file2.sgml
    # which installs Splus sgml help files defined to DESTDIR/.Data/__?help.
    # We changed syntax to require '--destdir DESTDIR'.
    # We also have it use Microsoft's hhc.exe to install a chm file on Windows.
    # $tmp = Splus::HELPINSTALL::new qw(--destdir DESTDIR --clean-first file1.q file2.R)
    # sets things up and $tmp->make() does the installation.
    # !!! On Windows, we use perl script and bat file slightly modified
    # from R.  It isn't very flexible and assumes you are in the main package
    # directory, which has a man directory, and it assumes you want to install
    # all the *.Rd's in man.  For consistency with other Splus::*INSTALL.pm modules,
    # I'll accept the usual arguments, but ignore lots of them.
    my $class = shift ;
    my $self = {} ;
    my $ret = bless $self, $class ;
    $self->{srcs} = {};
    $self->{src_id} = 0 ;
    my $src_supplied = 0 ;

    while (@_) {
        $_ = shift @_ ;
        if (/^(-h|--help)/) {
            _Syntax() ;
            exit();
        } elsif (/^(-v|--verbose)/) {
            $self->{verbose} = 1 ;
        } elsif (/^--clean-first/) {
            $self->{clean_first} = 1 ;
        } elsif (/^--no-examples/) {
            $self->{no_examples} = 1 ;
        } elsif (/^--clean-after/) {
            $self->{clean_after} = 1 ;
        } elsif (/^--no-clean-after/) { # for now we always clean after on Windows, no clean after to do on Unix
            $self->{clean_after} = 0 ;
        } elsif (/^--destdir/) {
            $self->{destdir} = shift @_ or _Syntax "No directory name after --destdir" ;
            $self->_canonicalize_destdir() ;
        } else {
            $self->add_source_files($_) ;
            $src_supplied = 1 ;
        }
    }
    $self->{destdir} or _Syntax "No --destdir directory given" ;
    if (!$src_supplied) {
        $self->add_source_files(globlist(qw(*.Rd))) ;
    }
    if ($ostype eq "windows") {
        # we are really assuming the destdir must be ".." and that
        # the current directory is called "man".
        # Let's wait until make() to do this check.
        # die "Current directory is not called 'man'" unless basename(getcwd()) eq "man" ;
    }
    $ret ;
}

# On Unix, help files are installed in .Data/__?help
# so let's make sure .Data and standard directories
# exist.  We will only call these on Unix.
sub _mkdir
{
    # not for public use.
    # make a directory if it does not exist.  Die if problems arise.
    my $dir = shift ;
    if (! -d $dir ) {
       die "A non-directory file $dir exists and we need to make a directory by that name" if -e $dir ;
       mkdir $dir or die "Cannot make directory $dir ($!)" ;
    }
}

sub _prepare_destdir
{
    # make sure destdir has a .Data with right components in it.
    # Don't touch a good one, but fix up incomplete ones.
    # Die if destdir itself does not exist or if there are
    # any problems making .Data and its subdirectories.
    # Should have __Hhelp/  __Meta/  __Shelp/
    # and empty file ___nonfile (unix) or ___nonfi (windows or windows-compatible).
    # For now, we will leave out the nonfile.
    my $self = shift ;
    my $destdir = $self->{destdir} ;
    _Syntax "Destination directory $destdir does not exist or is not a directory" if ! -d $destdir ;
    _mkdir ("$destdir/.Data") ;
    _mkdir ("$destdir/.Data/__Meta") ;
    _mkdir ("$destdir/.Data/__Hhelp") ;
    _mkdir ("$destdir/.Data/__Jhelp") ;
    _mkdir ("$destdir/.Data/__Shelp") ;
    -w "$destdir/.Data/__Hhelp" or die safe_abs_path("$destdir/.Data/__Hhelp") . " is not writable" ;
    -w "$destdir/.Data/__Jhelp" or die safe_abs_path("$destdir/.Data/__Jhelp") . " is not writable" ;
    -w "$destdir/.Data/__Shelp" or die safe_abs_path("$destdir/.Data/__Shelp") . " is not writable" ;
}

sub _clean_destdir_data
{
    # remove help files from .Data/__?help
    my $self = shift ;
    my $destdir = $self->{destdir} ;
    my @helpdirs = ( "$self->{destdir}/.Data/__Hhelp",
                     "$self->{destdir}/.Data/__Shelp",
                     "$self->{destdir}/.Data/__Jhelp") ;
    foreach my $dir (@helpdirs) {
        if (-d "$dir") {
            -w "$dir" or die "Directory " . safe_abs_path("$dir") . " is not writable" ;
        }
    }
    foreach my $dir (@helpdirs) {
        if (-d "$dir") {
            $self->{verbose} and print "$0: removing help files from " . safe_abs_path("$dir") . "\n" ;
            opendir my $dirhandle, $dir or die "Cannot open $dir ($!)";
            while (my $filename = readdir $dirhandle) {
                next if $filename =~ /^(\.|\.\.)$/ ;
                my $fullname = "$dir/$filename" ;
                next if -d $fullname ;
                unlink $fullname or warn "Could not remove file $fullname, $dir may not be clean" ;
            }
            closedir $dirhandle ;
        }
    }
}

sub _canonicalize_destdir
{
    # not for end-user use
    # No arguments (except implicit $self).
    # Look at destdir and remove possible trailing /.Data
    # Also, change any backslashes to slashes.
    # This will only be used by perl and Splus, not by cmd.exe.
    my $self = shift ;
    $self->{destdir} =~ s^\\^/^g ;
    $self->{destdir} =~ s^/\.Data$^^i ;
    -d $self->{destdir} or _Syntax "--destdir $self->{destdir} does not name a directory" ;
}

sub add_source_files
{
    my $self = shift ;
    foreach my $arg (@_) {
        foreach my $source_file (glob($arg)) {
           -e $source_file or _Syntax "help file $source_file does not exist" ;
           -d $source_file and _Syntax "Putative help file $source_file is a directory (use --destdir dir)" ;
           ${$self->{srcs}}{"$source_file"}=$self->{src_id}++ ;
        }
    }
}
sub delete_source_files
{
    my $self = shift ;
    foreach my $source_file (@_) {
        delete ${$self->{srcs}}{"$source_file"} ;
    }
}

sub get_source_files
{
    my $self = shift ;
    # sort to order in which they were given
    my @sorted_srcs = sort { ${$self->{srcs}}{$a} <=> ${$self->{srcs}}{$b} } keys(%{$self->{srcs}}) ;
    @sorted_srcs ;
}

sub make_unix
{
    my $self = shift ;
    my @srcs = get_source_files($self) ;
    if ( scalar @srcs == 0) {
        print "$0 : No help files to install\n" ;
    } else {
        print "Installing help files to directory " . safe_abs_path($self->{destdir}) . " from the ", scalar @srcs, " file(s) ", join(", ", @srcs), "\n" ;
        $self->_prepare_destdir() ;
        $self->_clean_destdir_data() if $self->{clean_first} ;
        
        my $args = join(" ", @srcs) ;
        my $cmd = "Splus RDINSTALL $self->{destdir}/.Data $args" ;
	$self->{verbose} and print "$0: calling cmd=$cmd\n";
        my $status = system($cmd) ;
        print "$0: RDINSTALL status=$status\n" ;
        $cmd = "cd $self->{destdir} && Splus BUILD_JHELP" ;
        my $status = system($cmd) ;
        print "$0: BUILD_JHELP status=$status\n" ;
    }
    1 ; # should return status indicator, or die if make failed
}

sub make_windows
{
    my $self = shift ;
    # ignore all arguments, cd to one above man and run MAKECMM.bat
    # src2bin will put us in man directory, but MAKECHM.bat wants to
    # be in parent of man directory.  We wil try to allow either.
    my $cwd = safe_abs_path(getcwd()) ;
    eval {
        if (basename($cwd) eq "man") {
            chdir ".." or die "cannot cd to .. from $cwd ($!)" ;
        }
        -d "man" or die "There is no subdirectory of " . safe_abs_path(getcwd()) . "called man" ;
        if ( -d "swingui" && -d "swingui/help" ) {
            # The following depends on pkgutils/share/perl/make-help-windows.pl
            # making pkg/chm if it does not exist and otherwise using the
            # existing one.
            mkdir ("chm") or die "Cannot make chm directory ($!)" ;
            opendir my $guihelpdir, "swingui/help" or die "Cannot open swingui/help directory ($!)" ;
            foreach (readdir $guihelpdir) {
                next if /^(\.\.|\.)$/ ;
                if (/\.(html|htm|gif|js|css)$/i) {
                    File::Copy::copy("swingui/help/$_", "chm/$_") or die "Cannot copy swingui/help/$_ to chm/$_ ($!)" ;
                }
            }
            closedir $guihelpdir ;
        }
        $self->{mandir} = safe_abs_path(getcwd()) . "man" ;
        my $status = system("Splus.bat START -exec MAKECHM.bat") ;
        print "$0: MAKECHM status=$status\n";
    } ;
    die $@ if $@ ;
    chdir $cwd or die "Cannot cd back to $cwd after installing help files" ;
}

sub make
{
    my $self = shift ;
    if ($ostype eq "windows") {
        $self->make_windows() ;
    } else {
        $self->make_unix() ;
    }
    $self->extract_examples() if !$self->{no_examples} ;
}

sub extract_examples
{
    my $self = shift ;
    my $Rex = "$self->{destdir}/R-ex" ;
    rmtree($Rex,0,1) if -e $Rex ;
    _mkdir ($Rex) ;
    -w $Rex or die safe_abs_path($Rex) . " is not writable" ;
    my @srcfiles = get_source_files($self) ;
    my $indexfile = "$Rex/AnIndex" ; # R uses help's AnIndex for R-ex.  No help directory in Splus
    open my $index_handle, ">", $indexfile or die "Cannot open example index file, \"", safe_abs_path($indexfile), "\" for appending" ;
    foreach my $srcfile (@srcfiles) {
        print "$srcfile", ": ", safe_abs_path(dirname($srcfile)), "::", basename($srcfile), "\n";
        my $Rexfile = basename($srcfile);
        my $name = $Rexfile ;
        $name =~ s/\.Rd$// ;
        $Rexfile =~ s/\.Rd$/.R/ ;
        $Rexfile = "$Rex/" . $Rexfile ;
        Rdconv($srcfile, "example", "", $Rexfile, "");
        if (!-e $Rexfile) {
            warn "    No \\examples section found in ", safe_abs_path($srcfile) ;
            next ;
        }
        open my $Rex_handle, "<", $Rexfile or die "Cannot open example file $Rexfile to get indexing information ($!)" ;
	my $nametag ; # the internal name, not used much
        while (<$Rex_handle>) {
            chomp ;
            if (/^### Name: */) {
               $nametag = $' ; # chars after matched string, we hope it is file name
            } elsif (/^### Aliases: /) {
               foreach my $alias (split / +/, $') {
                   print $index_handle "$alias\t$name\n";
               }
            }
        }
        close $Rex_handle
    }
    close $index_handle;
}

1;
