package Splus::JAVAINSTALL;
use v5.8;
use strict ;

=head1 NAME

  Splus::JAVAINSTALL - compile and install Java code files in the
    java directory.  The resulting jar's will be placed into
    DESTDIR/jars (which will typically be package/jars).

=head1 SYNOPSIS

  The following is the plan.  Only the ones before the bar are implemented.

  use Splus::JAVAINSTALL;
  $sh = Splus::JAVAINSTALL->new(@ARGV) ; # args: --destdir DESTDIR --name packageName
  $sh->make() ; # do the installation
  --- bar ---
  $sh->make_virgin() ; # remove all generated files, including DESTDIR/jars

=cut

use Cwd qw(getcwd abs_path) ;
use File::Path; # for rmtree(), to remove directory recursively
use File::Find;
no warnings 'File::Find'; # maybe not needed any more
use Splus::Vars;
use Splus::Utils;
use Splus::SplusUtils;
use Splus::Make;
# use Splus::JavaUtils; # determine_target_flag was in JavaUtils, but merged it into here.

Splus::Vars::error("SHOME");
my $SHOME=${Splus::Vars::SHOME} ;
my $ms = ${Splus::Vars::S_BUILD_SYS} =~ /^MS/i ;
if ($ms && $SHOME =~ / /) { $SHOME=dosify($SHOME); }

Splus::Vars::error("MAKE");
Splus::Vars::error("OSTYPE");
my $osname = ${Splus::Vars::OSTYPE} eq "windows" ? "windows" : $^O ; # "windows", "linux", "solaris", "aix", etc.

# copied from SHLIB.pm.
sub find_makevars_file {
    my $dir = shift ;
    $dir = dosify($dir) ;
    my $template="Makevars_" . "$Splus::Vars::OSTYPE" ;
    $template=("$template" . "_" . "$Splus::Vars::S_BUILD_SYS") if "$Splus::Vars::S_BUILD_SYS" ne "" ;
    my $filename="";
    LAB: until ( $template eq ""  ) {
        $filename = dosify("$dir/$template");
        if ( -e $filename) { last LAB ; }
        $template =~ s/_?[^_]*$// ;
    }
    $filename=undef if ($template eq "") ;
    $filename ;
}


sub _Syntax {
    print "-h,--help             # print this help message\n";
    print "--destdir dirName     # put jar files into dirName/jars\n";
    @_ && die join(", ", @_) ;
}

#   determine_target_flag($javac, $classpath_flag) : returns a list like
#      ("-target", "1.5") giving flags you need to give to javac
#      so it will produce class files that the JRE in Splus
#      can load.  $javac is typically "javac", $classpath_flag -classpath.

sub determine_target_flag {
    my ($javac, $classpath_flag) = @_ ;
    $javac = "javac" unless $javac ;
    $classpath_flag = "-classpath" unless $classpath_flag ;
    my $cwd = getcwd() ;
    my $tmpdir = Splus_tempfile("Sjava") ;
    mkdir $tmpdir or die "Cannot make temporary directory $tmpdir: $!" ;
    my @good_flag = () ;
    my $status = 0 ;
    eval {
        chdir $tmpdir or die "Cannot change to temporary directory $tmpdir: $!" ;
        # try to compile and run some really simple java code.
        my $javacode = "public class test { public static void main(String args[]){} }" ;
        open(my $jh, ">", "test.java") or die "Cannot open $tmpdir/test.java: $!" ;
        print $jh $javacode, "\n" ;
        close $jh ;
        my @flags = ( # horrendous nested array notation 
            [qw(-target 1.9)],
            [qw(-target 1.8)],
            [qw(-target 1.7)],
            [qw(-target 1.6)],
            [qw(-target 1.5)], # Splus 8.0 uses JRE 1.5
            [qw(-target 1.4)],
            [qw()],            # if javac matches Splus's JRE this one will work
            [qw(-target 1.3)],
            [qw(-target 1.2)], # this is what javac 1.4 gives by default
            [qw(-target 1.1)]
            );
        foreach my $flag (@flags) {
            my @cmd ;
            push @cmd, $javac ;
            push @cmd, @$flag ; # horrendous array ref notation
            push @cmd, "test.java" ;
            push @cmd, qw(> javacout.txt 2> javacerr.txt) ;
            # must convert to single string to use sh or cmd > and 2>.
            $status = system(join(" ",@cmd)) ;
            # print "javac cmd=", join(" ",@cmd), ": status=$status\n";
            if ($status !=0 && scalar(@$flag) == 0) { # no '-target x.y'
                die "Cannot compile Java code with command \"$javac file.java\".  Please add the directory containing your Java compiler, $javac, to PATH\n";
            }
            unlink("javacout.txt") ; unlink("javacerr.txt") ;
            $status == 0 or next ;
            # need to make sure `Splus SHOME`/java/jre/bin/java
            # can load the class file.
            my $SHOME=${Splus::Vars::SHOME} ;
            @cmd = () ;
            push @cmd , "\"" . Splus::Utils::file_path("$SHOME", qw(java jre bin java) ) . "\"";
            push @cmd, ( $classpath_flag,  "." ) ;
            push @cmd, qw( test ) ;
            push @cmd , qw(> java_out.txt 2> java_err.txt) ;
            $status = system(join(" ",@cmd)) ;
            # print "   java cmd=", join(" ",@cmd), ": status=$status\n";
            if ($status == 0) {
               @good_flag = @$flag ;
               last ;
            }
        }
        if ($status != 0) {
            die "Cannot determine suitable '-target x.y' for java compiler $javac so that Splus's JRE can load the resulting class files" ;
        }
    } ;
    my $eval_err = $@ ; # in case $@ will be overwritten.  Want to clean up before dying.
    chdir $cwd or die "Cannot return to directory $cwd from $tmpdir: $!" ;
    File::Path::rmtree($tmpdir) ;
    die "Problem in determine_target_flag: $eval_err" if $eval_err ;
    # print "Files are in $tmpdir\n";
    # print "target_flag should be: ", join(" ", @good_flag), "\n";
    @good_flag;
}

sub new {
    # Compile java files in ./src to make class files in newly minted
    # ./classes.  Make jar file out of those class files and
    # $tmp = Splus::JAVAINSTALL::new qw(--destdir DESTDIR --clean-first)
    # sets things up and $tmp->make() does the installation.
    my $class = shift ;
    my $self = {} ;
    my $ret = bless $self, $class ;
    $self->{srcdir} = "./src";
    $self->{copydir} = "./copy";
    if (-d "./prebuiltjars") {
        $self->{prebuiltjarsdir} = dosify(getcwd() . "/" . "prebuiltjars");
        $self->{prebuiltjars} = [] ;
        opendir my $dirhandle, $self->{prebuiltjarsdir} or die "Cannot open $self->{prebuiltjarsdir} ($!)";
        while (my $filename = readdir $dirhandle) {
            if ($filename =~ /\.jar$/i) {
                push @{$self->{prebuiltjars}}, $filename ;
            }
        }
        closedir $dirhandle ;
    }

    while (@_) {
        $_ = shift @_ ;
        if (/^(-h|--help)/) {
            _Syntax() ;
            exit() ;
        } elsif (/^(-v|--verbose)/) {
            $self->{verbose} = 1 ;
        } elsif (/^--clean-first/) {
            $self->{clean_first} = 1 ;
        } elsif (/^--clean-after/) { # there is no clean_after action, but supply argument for consistency
            $self->{clean_after} = 1 ;
        } elsif (/^--no-clean-after/) {
            $self->{clean_after} = 0 ;
        } elsif (/^--destdir/) {
            $self->{destdir} = shift @_ or _Syntax("No directory name after --destdir");
            $self->_canonicalize_destdir() ;
        } elsif (/^--name/) {
            $self->{packageName} = shift @_ or _Syntax("No package name after --name");
        }
    }
    $self->{destdir} or _Syntax "No --destdir directory given" ;
    $self->{packageName} or $self->{packageName} = "Spackage" ; # fix this up
    $self->{packagejar} = "$self->{packageName}.jar" ;

    # Following paragraph copied from SHLIB.pm.
    # now find the Makevars files in the standard directories
    # and get macro definitions from them.
    my $mk = Splus::Make->new() ;
    my $makevars_dir ;
    foreach $makevars_dir ("$SHOME/cmd", ".") {
       my $makevars_file=&find_makevars_file($makevars_dir);
       if ($makevars_file) {
          $mk->parse_makefile($makevars_file) ;
          push @{$self->{makevars_files}}, $makevars_file;
       }
    }
    $self->{makevars} = $mk ;
    $mk->parse_macrodef("SHOME=$SHOME") ; # so PKG_JAVAC_CLASSPATH can use $(SHOME)
    $self->{JAVAC} = $mk->expand_makevar("JAVAC") ;
    $self->{JAVAC_CLASSPATH_FLAG} = $mk->expand_makevar("JAVAC_CLASSPATH_FLAG") ;
    $self->{JAVAC_CLASSPATH_SEP} = $mk->expand_makevar("JAVAC_CLASSPATH_SEP") ;
    $self->{PKG_JAVACFLAGS} = $mk->expand_makevar("PKG_JAVACFLAGS") ;
    $self->{PKG_JAVAC_CLASSPATH} = $mk->expand_makevar("PKG_JAVAC_CLASSPATH") ;
    @{$self->{target_flag}} = determine_target_flag($self->{JAVAC}, $self->{JAVAC_CLASSPATH_FLAG}) ; # might die here

    $ret ;
}

sub _mkdir
{
    # not for public use.
    # make a directory if it does not exist.  Die if problems arise.
    my $dir = shift ;
    if (! -d $dir ) {
       die "A non-directory file $dir exists and we need to make a directory by that name" if -e $dir ;
       mkdir $dir or die "Cannot make directory $dir ($!)" ;
    }
}

sub _make_empty_jarsdir
{
    # make sure destdir has a writable jars subdirectory
    my $self = shift ;
    _Syntax "Destination directory $self->{destdir} does not exist or is not a directory" if ! -d $self->{destdir} ;
    $self->{jarsdir} = "$self->{destdir}/jars" ;
    $self->_remove_jarsdir ;
    File::Path::rmtree("$self->{jarsdir}") if -e "$self->{jarsdir}" ;
    _mkdir ("$self->{jarsdir}") ;
    -w "$self->{jarsdir}" or die abs_path("$self->{jarsdir}") . " is not writable" ;
}

sub _remove_jarsdir
{
    # remove destdir/jars
    my $self = shift ;
    $self->{jarsdir} or die "Internal corruption: self->jarsdir not set" ;
}

sub _remove_copy
{
    # remove ./copy
    my $self = shift ;
    $self->{copydir} or die "Internal corruption: self->copydir not set" ;
    File::Path::rmtree("$self->{copydir}") if -e "$self->{copydir}" ;
}

sub _copy_src_to_copydir
{
    my $self = shift ;
    $self->{copydir} or die "Internal corruption: self->copydir not set" ;
    File::Path::rmtree("$self->{copydir}") if -e "$self->{copydir}" ;
    _mkdir ("$self->{copydir}") ;
    -w "$self->{copydir}" or die abs_path("$self->{copydir}") . " is not writable" ;
    # recursively copy src to copy/src without touching line endings
    Splus::SplusUtils::copy_recursive("$self->{srcdir}", "$self->{copydir}", "asis");
}

sub _canonicalize_destdir
{
    # not for end-user use
    # No arguments (except implicit $self).
    # Look at destdir and change any backslashes to slashes.
    # This will only be used by perl and Splus, not by cmd.exe.
    my $self = shift ;
    $self->{destdir} =~ s^\\^/^g ;
    -d $self->{destdir} or die "--destdir $self->{destdir} does not name a directory" ;
}

sub _copy_all_jars_to_destdir
{
    my $self = shift ;
    foreach my $prebuiltjar (@{$self->{prebuiltjars}}) {
        Splus::SplusUtils::_copy_file("$self->{prebuiltjarsdir}/$prebuiltjar", "$self->{jarsdir}/$prebuiltjar", "asis") ;
    }
    Splus::SplusUtils::_copy_file("copy/$self->{packagejar}", "$self->{jarsdir}/$self->{packagejar}", "asis") if -e "copy/$self->{packagejar}";
}

sub _list_java_files
{
    my $self = shift ;
    my @dirs = $self->{srcdir} ;
    my @list ;
    File::Find::find( sub { /\.java$/i  && ! -d $_  && push(@list, $File::Find::name);}, @dirs);
    @list ;
}

sub make
{
    my $self = shift ;
    @{$self->{java_files}} = $self->_list_java_files() ;
    print("Java files in $self->{srcdir}:\n    ", join("\n    ", @{$self->{java_files}}), "\n") if $self->{verbose} ;
    $self->_copy_src_to_copydir() ;
    my $cwd = getcwd() ;
    eval {
       chdir "$self->{copydir}" or die "Cannot chdir $self->{copydir} ($!)" ;
       # javac -d ./classes `find src -name "*.java"`
       #    Omit -d ./classes because we want class files
       #    where java files were.  We will remove the java
       #    files.  This keeps the *.properties and *.gif
       #    files in place for jar to find.
       my @args = $self->{JAVAC} ;
       push @args, @{$self->{target_flag}} ;
       my @jars = () ;
       if ($self->{prebuiltjars} && @{$self->{prebuiltjars}}) {
           push @jars, map( $self->{prebuiltjarsdir} . "/" . $_ , @{$self->{prebuiltjars}}) ;
           print("Added prebuiltjars:\n\t", join("\n\t\t", @jars), "\n") if $self->{verbose} ;
       }
       if ($self->{PKG_JAVAC_CLASSPATH}) {
           # we split before expanding so folks can avoid DOS C: in CLASSPATH
           # and use semicolon on all platforms.  This needs cleaning up.
           my @els = split(/[:;,]/, $self->{makevars}->get_makevar("PKG_JAVAC_CLASSPATH")) ;
           push @jars, map( $self->{makevars}->expand_macro($_), @els) ;
           print("Added PKG_JAVAC_CLASSPATH:\n\t", join("\n\t\t", @jars), "\n") if $self->{verbose} ;
       }
       if (@jars) {
           @jars = map( dosify($_), @jars) ;
           push @args, $self->{JAVAC_CLASSPATH_FLAG} ;
           push @args, join($self->{JAVAC_CLASSPATH_SEP}, @jars) ;
       }
       if ($self->{PKG_JAVACFLAGS}) {
           # Like CLASSPATH: split by whitespace so arguments get separated,
           # but split before expanding variables, so variables can contain
           # spaces.  E.g., "PKG_JAVACFLAGS=-I/Program Files/foo" will not
           # work but "PROGRAMFILES=/Program Files; PKG_JAVACFLAGS=-I$(PROGRAMFILES)/foo"
           # will.
           my @els = split(/[\s]/, $self->{makevars}->get_makevar("PKG_JAVACFLAGS")) ;
           push @args, map( $self->{makevars}->expand_macro($_), @els) ;
       }
       foreach my $java_file (@{$self->{java_files}}) {
           -e "$java_file" or die "Internal error: Java file $java_file not in copy directory";
           push @args, dosify($java_file) ;
       }
       print("compiling with: ", join("\n          ", @args), "\n") if $self->{verbose} ;
       my $status = system @args ;
       if ($status == -1) {
          die("Could not execute Java compiler \'javac\' (PATH may need extending): $!") ;
       } elsif ($status != 0) {
          die("Could not compile java code with \"" . join(" ", @args), "\"") if $status != 0 ;
       }
       unlink @{$self->{java_files}} ; # we are in copy directory, so this is ok.
       # Following depends on the 'jar' command acting like Sun's.
       # We may have to use make variables to parameterize it.
       @args = qw(jar cf) ;
       push @args, $self->{packagejar} ;
       push @args, qw(-C src .) ;
       print("Now make jar in " . getcwd() . " using command " . join(" ", @args) . "\n" ) if $self->{verbose} ;
       $status = system @args ;
       die "Could not make jar file" if $status != 0 ;
    } ;
    die "Problem in JAVAINSTALL: $@" if $@ ;
    chdir $cwd or die "Cannot return to directory $cwd in JAVAINSTALL ($!)";
    $self->_make_empty_jarsdir() ;
    $self->_copy_all_jars_to_destdir() ;
    1 ; # should return status indicator, or die if make failed
}


1;
