/*  RipOff - Plugin based CD Ripper
 *  Copyright (C) 2006 Bobby Ryan Newberry
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <errno.h>
#include <gdk/gdkx.h>
#include "RipOffExtractor.h"
#include "lib/RipOffTrack.h"

/* TODO: Find out the actual max size of a UNIX path */
const size_t path_size = 256;
const int max_retries = 20;
const int paranoia_mode = PARANOIA_MODE_FULL^PARANOIA_MODE_NEVERSKIP;

/* variable that the encoding threads use to inform the GUI thread that they have finished encoding */
gboolean finished_encoding;

/* variable that informs the extraction loops that the user has cancelled the ripping process */
gboolean process_cancelled;

glong sectors_ripped;

struct thread_params_
{
        RipOffExtractor extractor;
        GtkWidget *progress_bar;
        GtkWidget *status_label;
        GtkWidget *status_window;
        cdrom_drive_t *cd_drive;
	cdrom_paranoia_t *p;
};
typedef struct thread_params_ * thread_params;

/* converts target_string to a new output string using delimiters and the data contained in 
track*/
gchar *convert_file_pattern(gchar *target_string, RipOffTrack track);

/* takes in a given UNIX directory tree path and creates it, if it doesn't already exist
   returning true. If the path cannotbe created returns false. */
gboolean create_path(gchar **pathp);

gchar *copy_string(const gchar *string_to_copy);

gpointer rip_and_encode_thread(thread_params params);

gpointer rip_and_encode_track(  RipOffExtractor extractor,
                                RipOffTrack track,
                                GTimer *total_time_spent_ripping, 
                                glong total_sectors_to_rip,
                                GtkWidget *progress_bar,
                                FILE *output_descriptor,
                                cdrom_drive_t *cd_drive,
                                cdrom_paranoia_t *p);

void flush_commands();

glong calculate_total_sectors_to_rip(cdrom_drive_t *cd_drive, RipOffTrackList track_list);

void cancel_extraction(GtkWidget *button, GdkEvent *event, gpointer data);

RipOffExtractor ripoff_extractor_new(	RipOffPlugin plugin, 
					RipOffTrackList track_list,
					gchar *cd_device_path,
					gchar *output_folder, 
					gchar *file_pattern,
					GtkWidget *main_window)
{
	RipOffExtractor new = g_new(struct RipOffExtractor_, 1);

	new->plugin = plugin;
	new->track_list = track_list;
	new->cd_device_path = copy_string(cd_device_path);
	new->output_folder = copy_string(output_folder);
	new->file_pattern = copy_string(file_pattern);
	new->main_window = main_window;

	return new;
}

gboolean ripoff_extractor_perform_operation(RipOffExtractor extractor)
{
	/* GTK Widgets needed for status window */
	GtkWidget *status_window;
	GtkWidget *progress_bar;
	GtkWidget *status_label;
	GtkWidget *vbox;
        GtkWidget *cancel_button;
        cdrom_drive_t *cd_drive;

        thread_params params;

        struct stat s;

	#ifdef DEBUG
	int verbose = 1;
	#endif

	#ifndef DEBUG
	int verbose = 0;
	#endif

	/* make sure a CD is actually inserted */
        if(!ripoff_track_list_is_empty(extractor->track_list))
	{
                #ifdef DEBUG
		g_print("RipOffExtractor: Track List Isn't Empty. Processing....\n");
		#endif

		if(extractor->plugin == NULL)
		{
			gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 0);
			gtk_statusbar_push(	GTK_STATUSBAR(status_bar), 
						0,
						"No valid RipOff encoding plugin could be found");
			return FALSE;
		}

		#ifdef DEBUG
		g_print("RipOffExtractor: Extract Activated\n");
		#endif

		/* checks to make sure we have a valid cd drive device node
	  	and that we can create the needed paranoia structure with which 
	   	to use it */

		if(lstat(extractor->cd_device_path, &s))
		{
			#ifdef DEBUG
			g_printf("RipOffExtractor: Problem with device node\n");
			#endif

			return FALSE;
		}
		else
		{
			cd_drive = cdio_cddap_identify(extractor->cd_device_path, verbose, NULL);
		}

		if(cd_drive == NULL)
		{
			#ifdef DEBUG
			g_printf("RipOffExtractor: drive is NULL\n");
			#endif

			return FALSE;
		}

                if(cdio_cddap_open(cd_drive) != 0)
		{
			#ifdef DEBUG
			g_printf("RipOffExtractor: Failed to open disc drive\n");
			#endif

			return FALSE;
		}

                if(verbose)
			cdda_verbose_set(cd_drive,CDDA_MESSAGE_LOGIT, CDDA_MESSAGE_LOGIT);
		else
			cdda_verbose_set(cd_drive,CDDA_MESSAGE_FORGETIT, CDDA_MESSAGE_FORGETIT);

                #ifdef DEBUG
                g_printf("RipOffExtractor: Creating status window\n");
                #endif

		/* status window setup */
		status_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_widget_set_size_request(status_window, 300, 110);
		gtk_container_set_border_width(GTK_CONTAINER(status_window), 6);
		gtk_window_set_transient_for(	GTK_WINDOW(status_window), 
						GTK_WINDOW(extractor->main_window));
		gtk_window_set_title(GTK_WINDOW(status_window), NAME" CD Extraction");
		gtk_window_set_modal(GTK_WINDOW(status_window), TRUE);
		gtk_window_set_type_hint(GTK_WINDOW(status_window), GDK_WINDOW_TYPE_HINT_DIALOG);
		gtk_window_set_decorated(GTK_WINDOW(status_window), TRUE);
		gtk_window_set_skip_taskbar_hint(GTK_WINDOW(status_window), TRUE);
		progress_bar = gtk_progress_bar_new();
		status_label = gtk_label_new("Ripping CD. This is a bug if visible");
		cancel_button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);

		vbox = gtk_vbox_new(TRUE, 5);


		gtk_box_pack_start(	GTK_BOX(vbox),
					progress_bar,
					TRUE,
					FALSE,
					0);
		
		gtk_box_pack_start(	GTK_BOX(vbox),
					status_label,
					TRUE,
					FALSE,
					0);

		gtk_box_pack_start(	GTK_BOX(vbox),
					cancel_button,
					TRUE,
					FALSE,
					0);

		g_signal_connect(	cancel_button,
					"clicked",
					G_CALLBACK(cancel_extraction),
					NULL);

                gtk_progress_bar_set_fraction(	GTK_PROGRESS_BAR(progress_bar), 0);
                process_cancelled = FALSE;

		gtk_container_add(GTK_CONTAINER(status_window), vbox);
		gtk_widget_show(cancel_button);
		gtk_widget_show_all(status_window);

                #ifdef DEBUG
                g_printf("RipOffExtractor: Status window created\n");
                #endif

                params = g_new(struct thread_params_, 1);
                params->p = cdio_paranoia_init(cd_drive);
		paranoia_modeset(params->p, paranoia_mode);
                params->cd_drive = cd_drive;
                params->status_window = status_window;
                params->progress_bar = progress_bar;
                params->status_label = status_label;
                params->extractor = extractor;

                #ifdef DEBUG
                g_printf("RipOffExtractor: Creating the rip_and_encode thread\n");
                #endif

                g_thread_create(	(GThreadFunc)      rip_and_encode_thread, 
					                   params, 
							   TRUE, 
							   NULL);

                #ifdef DEBUG
                g_printf("RipOffExtractor: rip_and_encode thread created. Good luck!\n");
                #endif
                
        }
}

gpointer rip_and_encode_thread(thread_params params)
{
	/* Timer used to calculate the extraction speed */
	GTimer *interval_timer;
	GTimer *total_time_spent_ripping;

        /* used for creating the output path for an encoded track */
	gchar *complete_output_path;
	const gchar *extension;
	gchar *current_track_name;
	int current_track_string_length;
	gchar *converted_string;
        glong total_sectors_to_rip;

	FILE *output_descriptor;
	gchar *error_message;
        gchar *progress_status;

        /* keeps track of the number of sectors ripped for the entire process */
        sectors_ripped = 0;
        
        total_sectors_to_rip = calculate_total_sectors_to_rip(params->cd_drive, params->extractor->track_list);
        total_time_spent_ripping = g_timer_new();
        extension = ripoff_plugin_get_extension(params->extractor->plugin);

        RipOffTrack track = ripoff_track_list_start(params->extractor->track_list);

        while(track != NULL && !process_cancelled)
        {
                gdk_threads_enter();
		current_track_name = ripoff_track_get_track_title(track);
		current_track_string_length = strlen(current_track_name) + 1;
		progress_status = g_malloc(current_track_string_length + 21);

		g_snprintf(	progress_status, 
				11 + current_track_string_length , 
				"Extracting %s", 
				current_track_name);

		gtk_label_set_text(GTK_LABEL(params->status_label), progress_status);

		converted_string = convert_file_pattern(params->extractor->file_pattern, track);

		#ifdef DEBUG
		g_print("RipOffExtractor: Converted String: %s\n", converted_string);
		#endif
			
		complete_output_path = g_malloc(strlen(params->extractor->output_folder) + 
					   	strlen(converted_string) + strlen(extension) + 3);
		#ifdef DEBUG
		g_print("RipOffExtractor: File Type Extension %s\n", extension);
		g_print("RipOffExtractor: malloc'd succesfully\n");
		#endif

		strcpy(complete_output_path, params->extractor->output_folder);
		strcat(complete_output_path, "/");
		strcat(complete_output_path, converted_string);
		strcat(complete_output_path, ".");
		strcat(complete_output_path, extension);
			
                #ifdef DEBUG
		g_print("RipOffExtractor: catted successfully\n");
		#endif

		#ifdef DEBUG
		g_printf("RipOffExtractor: Complete output path: %s\n", complete_output_path);
		#endif

		if(!create_path(&complete_output_path))
		{
			gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 0);
			gtk_statusbar_push(	GTK_STATUSBAR(status_bar), 
							0,
							"Specified output path could not be created");
			break;
		}
			
		gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 0);
		gtk_statusbar_push(GTK_STATUSBAR(status_bar), 0, NAME" is busy.....");

		output_descriptor = fopen(complete_output_path, "w+b");

		if(output_descriptor == NULL)
		{
			error_message = g_malloc(strlen(complete_output_path) + 
				                 strlen("Could not open ") + 1
					 );
			strcpy(error_message, "Could not open ");
			strcat(error_message, complete_output_path);
			gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 0);
			gtk_statusbar_push(	GTK_STATUSBAR(status_bar), 
							0,
						error_message);
			g_free(error_message);
			break;
		}

                flush_commands();
                gdk_threads_leave();

                #ifdef DEBUG
                g_printf("RipOffExtractor: Calling rip_and_encode_track\n");
                #endif

                rip_and_encode_track(   params->extractor,
                                        track,
                                        total_time_spent_ripping,
                                        total_sectors_to_rip,
                                        params->progress_bar,
                                        output_descriptor,
                                        params->cd_drive,
                                        params->p);

                #ifdef DEBUG
                g_printf("RipOffExtractor: rip_and_encode_track exited\n");
                #endif

                g_free(progress_status);
                g_free(complete_output_path);
                g_free(converted_string);

                track = ripoff_track_list_next_track(params->extractor->track_list);
        }

        gdk_threads_enter();
        gtk_widget_destroy(params->status_window);
        
        if(process_cancelled)
        {
                gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 0);
                gtk_statusbar_push(GTK_STATUSBAR(status_bar), 0, "Extraction process was cancelled");
       
        }
        else 
        {
                gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 0);
                gtk_statusbar_push(	GTK_STATUSBAR(status_bar),
                                        0,
                                        NAME" is ready");
        }
        flush_commands();
        gdk_threads_leave();
        g_timer_destroy(total_time_spent_ripping);
        ripoff_track_list_destroy(params->extractor->track_list);
        ripoff_extractor_destroy(params->extractor);
	cdio_paranoia_free(params->p);
	cdio_cddap_close(params->cd_drive);
        g_free(params);
}

gpointer rip_and_encode_track(  RipOffExtractor extractor,
                                RipOffTrack track,
                                GTimer *total_time_spent_ripping, 
                                glong total_sectors_to_rip,
                                GtkWidget *progress_bar,
                                FILE *output_descriptor,
                                cdrom_drive_t *cd_drive,
                                cdrom_paranoia_t *p)
{
	glong firstsector;
	glong lastsector;
	glong total_bytes_to_encode;
	glong cursor;
	int16_t *audio_buffer;
        GTimer *interval_timer;
        gchar time_string[50];

	/* ok, we should be good to go. It's time to rip and encode a track! */

	#ifdef DEBUG
	g_printf("RipOffExtractor: Preparing to extract audio!\n");
	#endif

	firstsector = cdda_track_firstsector(cd_drive, ripoff_track_get_track_num(track));
	lastsector = cdda_track_lastsector(cd_drive, ripoff_track_get_track_num(track));
	total_bytes_to_encode = (lastsector-firstsector + 1) * CDIO_CD_FRAMESIZE_RAW;

	cursor = firstsector;
	paranoia_seek(p, firstsector, SEEK_SET);

	#ifdef DEBUG
	g_printf("RipOffExtractor: Performing audio file plugin initialization...\n");
	#endif

	if(!ripoff_plugin_perform_setup(        extractor->plugin,
                                                total_bytes_to_encode,
                                                output_descriptor,
					        track))
	{
		g_fprintf(stderr, "Plugin failed to initialize. Cleaning up and moving to next track...\n");

		finished_encoding = TRUE;
		return NULL;
	}

	#ifdef DEBUG
	g_printf("RipOffExtractor: Audio file plugin initialization successful! Ripping...\n");
	#endif

        interval_timer = g_timer_new();

	while(cursor <= lastsector && !process_cancelled)
	{

                if(g_timer_elapsed(interval_timer, 0) >= 3.0)
		{
                        #ifdef DEBUG
                        g_printf("RipOffExtractor: Three seconds have passed. Updating the status window\n");
                        #endif
                      
                        gdk_threads_enter();
                        float sectors_per_second = sectors_ripped / g_timer_elapsed(total_time_spent_ripping , 0);
                        glong seconds_left = (total_sectors_to_rip - sectors_ripped) / sectors_per_second;

                        g_snprintf(     time_string,
			                50,
				        "Ripping at %.1fx with %i:%i%i remaining",
				        sectors_per_second * CDIO_CD_FRAMESIZE_RAW / 150000.0,
				        seconds_left / 60,
				        seconds_left % 60 / 10,
				        seconds_left % 60 % 10 );



                        gtk_progress_bar_set_text(	GTK_PROGRESS_BAR(progress_bar),
									time_string);

                        gtk_progress_bar_set_fraction(	GTK_PROGRESS_BAR(progress_bar), 
							((gdouble) sectors_ripped) / total_sectors_to_rip);

                        flush_commands();
                        gdk_threads_leave();

                        #ifdef DEBUG
                        g_printf("RipOffExtractor: Status window (hopefully) updated\n");
                        #endif

			g_timer_start(interval_timer);
		}

 

		audio_buffer = paranoia_read_limited(p, NULL, max_retries);
		
		if(audio_buffer != NULL)
		{

			#ifdef DEBUG
			g_printf("RipOffExtractor: Audio Buffer isn't NULL. Encoding...\n");
			#endif

			ripoff_plugin_encode_buffer(	extractor->plugin,
							total_bytes_to_encode,
							audio_buffer,
							output_descriptor,
							track);
		}
		else
		{
			#ifdef DEBUG
			g_printf("RipOffExtractor: Audio Buffer was NULL. Ripping is over!\n");
			#endif 

			cursor = lastsector;
		}

		++cursor;
		++sectors_ripped;
	}

	#ifdef DEBUG
	g_printf("RipOffExtractor: All CD audio data has been ripped. Telling plugin to perform closing operations...\n");
	#endif

	ripoff_plugin_perform_cleanup(	extractor->plugin,
					total_bytes_to_encode,
					output_descriptor,
					track);

	#ifdef DEBUG
	g_printf("RipOffExtractor: Plugin cleanup operations finished. Track extracted\n");
	#endif

        g_timer_destroy(interval_timer);

	finished_encoding = TRUE;
	return NULL;
}

void ripoff_extractor_destroy(RipOffExtractor extractor)
{
	g_free(extractor->cd_device_path);
	g_free(extractor->output_folder);
	g_free(extractor->file_pattern);
	g_free(extractor);
}

gchar *convert_file_pattern(gchar *target_string, RipOffTrack track)
{
	gchar *tmp_buffer; /* used for copying strings to a larger buffer */
	gchar *result; /* the string in which the converted string is stored */
	gchar *substitution; /* stores the values that are to be substituted in */
	gboolean match_found; /* stores whether a match has been found in the current cycle */
	int target_length = strlen(target_string) + 1; /* stores the length, in charachters of
							target string. Is larger by 1 so that 
							null character gets on the end of results
							string */
	int j; /* control variable for the target_string */
	int k = 0; /* control variable for the results_string */
	int l; /* control variable for substitution strings */
	int result_length; /* length of the result string */
	int expanded_length; /* length of target string once substitutions have been made */
	int substitution_length; /* length of strings that will be substituted in */

	expanded_length = target_length; /* expanded length is set equal to target string's length
					    since no substitutions have been made yet */
	result = g_malloc(target_length * 2); /* results string equal to two times that of
						 the target string is allocated */
	result_length = target_length * 2; /* length is set accordingly */
	

	for(j = 0; j < target_length; ++j)
	{
		match_found = FALSE; /* sets that no match has been found yet */
		
		#ifdef DEBUG
		g_printf("RipOffExtractor: Target String position and character %i: %c\n", j, target_string[j]);
		#endif

		/* checks for appropriate delimiters */
		if(target_string[j] == '%' && j+1 < target_length)
		{
			/* checks for delimiters and handles each accordingly */
			switch(target_string[j+1])
			{
				case 'N': match_found = TRUE;
			substitution = ripoff_track_get_track_num_string(track, TRUE);
					  substitution_length = strlen(substitution);
					  break;

				case 'n': match_found = TRUE;
			substitution = ripoff_track_get_track_num_string(track, FALSE);
					  substitution_length = strlen(substitution);
					  break;
				case 't': match_found = TRUE;
					  substitution = ripoff_track_get_track_title(track);
					  substitution_length = strlen(substitution);
					  break;
				case 'a': match_found = TRUE;
					  substitution = ripoff_track_get_artist(track);
					  substitution_length = strlen(substitution);
					  break;
				case 'g': match_found = TRUE;
					  substitution = ripoff_track_get_genre(track);
					  substitution_length = strlen(substitution);
					  break;
				case 'y': match_found = TRUE;
					  substitution = ripoff_track_get_year(track);
					  substitution_length = strlen(substitution);
					  break;
				case 'l': match_found = TRUE;
					  substitution = ripoff_track_get_album_title(track);
					  substitution_length = strlen(substitution);
					  break;
			}

			/* case for if a matching delimter was found */
			if(match_found)
			{		 
				#ifdef DEBUG
				g_print("RipOffExtractor: Substitution %s\n", substitution);
				#endif

				/* calculates an appropriate expanded length */
				expanded_length = expanded_length - 2 + substitution_length;
				
				/* checks if result is large enough to handle the string
				   substitution. If not, doubles its size */
				if(result_length < expanded_length)
				{
					tmp_buffer = g_malloc(expanded_length * 2);
					strncpy(tmp_buffer, result, result_length);
					g_free(result);
					result = tmp_buffer;
					result_length = expanded_length * 2;

					#ifdef DEBUG
					g_print("RipOffExtractor: Result length: %i\n", result_length);
					#endif
				}
				/* copies the substitution string into results */
				for(l = 0; l < substitution_length; ++l)
				{
					if(substitution[l] == '/')
						result[k] = '~';
					else
						result[k] = substitution[l];
					++k;
				}

				--k; /* substracts one from k since k is one too large at the
				  	of above for loop */
				++j; /* increases j by one since the delimter must be skipped */
				g_free(substitution); /* string is no longer needed so it 
						       is freed*/
			}
		}

		else
		result[k] = target_string[j]; /* case for which no delimter match is found. 
						 character is simply copied */

		++k; /* k is increased accordingly */
	}

	return result; /* resultant string is returned */

}

gchar*
get_free_filename(gchar* origfilename)
{
  gchar* rv;
  
  if(g_file_test(origfilename, G_FILE_TEST_EXISTS))
    {
      guint length = strlen(origfilename) + 10;
      gchar* newfilename = g_malloc(length);
      
      guint i;
      for(i = 1; i < 100; i++)
        {
          g_snprintf(newfilename, length, "%s (%i)", origfilename, i);
          if(g_file_test(newfilename, G_FILE_TEST_EXISTS))
            continue;
          else
            {
              rv = newfilename;
              goto _exit_;
            }
        }

      rv = NULL;
    }
  else
    {
      rv = g_strdup(origfilename);
    }

 _exit_:
  return rv;
}

gboolean
create_path(gchar** pathp)
{
  gboolean rv = TRUE;
  gchar* dirname;
  gchar* filename;
  GIOChannel* ioc;  
  
  if(pathp == NULL)
    return FALSE;

  dirname = g_path_get_dirname(*pathp);
  if(!g_file_test(dirname, G_FILE_TEST_EXISTS))
    {
      if(g_mkdir_with_parents(dirname, 0700))
        {
          rv = FALSE;
          goto _exit_;
        }
    }

  filename = get_free_filename(*pathp);
  if(filename != NULL)
    {
      if((ioc = g_io_channel_new_file(filename, "w", NULL)) != NULL)
        {
          g_io_channel_close(ioc);
          g_free(*pathp);
          *pathp = filename;
          rv = TRUE;
          goto _exit_;
        }
      else
        {
          rv = FALSE;
          goto _exit_;
        }
    }
  else
    {
      rv = FALSE;
      goto _exit_;
    }

 _exit_:
  g_free(dirname);
  return rv;
}

gchar *copy_string(const gchar *string_to_copy)
{
	int size = strlen(string_to_copy)+1;
	gchar *string_copy = g_malloc(size);
	strncpy(string_copy, string_to_copy, size);
	return string_copy;
}

glong calculate_total_sectors_to_rip(cdrom_drive_t *cd_drive, RipOffTrackList track_list)
{
	glong total_sectors = 0;
	RipOffTrack track = ripoff_track_list_start(track_list);
	while(track != NULL)
	{
		total_sectors += cdda_track_lastsector(cd_drive, ripoff_track_get_track_num(track)) -
				 cdda_track_firstsector(cd_drive, ripoff_track_get_track_num(track)) + 1;
		track = ripoff_track_list_next_track(track_list);
				
	}

	return total_sectors;
}

void cancel_extraction(GtkWidget *button, GdkEvent *event, gpointer data)
{
	process_cancelled = TRUE;
}

void flush_commands()
{
  GdkDisplay *display = gdk_display_get_default ();
  XFlush (GDK_DISPLAY_XDISPLAY (display));
}
