// @HEADER
// *****************************************************************************
//        MueLu: A package for multigrid based preconditioning
//
// Copyright 2012 NTESS and the MueLu contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

#ifndef MUELU_INVERSEAPPROXIMATIONFACTORY_DECL_HPP_
#define MUELU_INVERSEAPPROXIMATIONFACTORY_DECL_HPP_

#include "MueLu_ConfigDefs.hpp"

#include <Teuchos_ParameterList.hpp>

#include <Xpetra_CrsMatrix_fwd.hpp>
#include <Xpetra_Map_fwd.hpp>

#include "MueLu_FactoryBase_fwd.hpp"
#include "MueLu_SingleLevelFactoryBase.hpp"
#include "MueLu_Utilities_fwd.hpp"

namespace MueLu {

/*!
  @class InverseApproximationFactory class.
  @brief Factory for building the approximate inverse of a matrix.

  ## Context, assumptions, and use cases ##

  This factory is intended to be used for building an approximate inverse of a given matrix \A. This is for now only
  used in the SchurComplementFactory to generate a respective \Ainv matrix.

  For blocked matrices, the InverseApproximationFactory per default generates an approximate inverse of the A_00 term.

  ## Input/output of this factory ##

  ### User parameters of InterfaceAggregationFactory ###
  Parameter | type | default | master.xml | validated | requested | description
  ----------|------|---------|:----------:|:---------:|:---------:|------------
  A                           | Factory | null     |  | * | * | Generating factory of the matrix A
  inverse: approximation type | string  | diagonal |  | * | * | Method used to approximate the inverse
  inverse: fixing             | bool    | false    |  | * | * | Fix diagonal by replacing small entries with 1.0

  The * in the master.xml column denotes that the parameter is defined in the master.xml file.
  The * in the validated column means that the parameter is declared in the list of valid input parameters (see GetValidParameters() ).
  The * in the requested column states that the data is requested as input with all dependencies (see DeclareInput() ).

  ### Variables provided by this factory ###

  After InverseApproximationFactory::Build the following data is available (if requested)

  Parameter | generated by | description
  ----------|--------------|------------
  | Ainv | InverseApproximationFactory | The approximate inverse of a given matrix.
*/

template <class Scalar        = DefaultScalar,
          class LocalOrdinal  = DefaultLocalOrdinal,
          class GlobalOrdinal = DefaultGlobalOrdinal,
          class Node          = DefaultNode>
class InverseApproximationFactory : public SingleLevelFactoryBase {
#undef MUELU_INVERSEAPPROXIMATIONFACTORY_SHORT
#include "MueLu_UseShortNames.hpp"

 public:
  //! @name Constructors/Destructors.
  //@{

  //! Constructor.
  InverseApproximationFactory() = default;

  //! Input
  //@{

  void DeclareInput(Level& currentLevel) const;

  RCP<const ParameterList> GetValidParameterList() const;

  //@}

  //@{
  //! @name Build methods.

  //! Build an object with this factory.
  void Build(Level& currentLevel) const;

  //@}

 private:
  //! Sparse inverse calculation method.
  RCP<Matrix> GetSparseInverse(const RCP<Matrix>& A, const RCP<const CrsGraph>& sparsityPattern) const;

};  // class InverseApproximationFactory

}  // namespace MueLu

#define MUELU_INVERSEAPPROXIMATIONFACTORY_SHORT
#endif /* MUELU_INVERSEAPPROXIMATIONFACTORY_DECL_HPP_ */
