\name{cover.design.F}
\alias{cover.design.F}
\title{
Computes Space-Filling "Coverage" designs using Swapping Algorithm
}
\description{
Attemps to find the set of points on a discrete grid (Candidate Set) which
minimize a geometric space-filling criterion.
}
\usage{
cover.design.F(R, nd, nruns=1, nn=T, num.nn=100, max.loop=30,
fixed=NULL, scale.type="unscaled", R.center, R.scale, P=-32, Q=32,
start=NULL, DIST=NULL, return.grid=T, return.transform=F)
}
\arguments{
\item{R}{
A matrix of candidate points to be considered in the design.
}
\item{nd}{
Number of points to add to the design. If points exist and are to
remain in the design (see "fixed" option), nd is the number of points
to add. If no points are fixed, nd is the design size.
}
\item{nruns}{
The number of random starts to be optimized. Uses random starts unless
"start" is specified. If nruns is great than 1, the final results are 
minimum.
}
\item{nn}{
Logical value specifying whether or not to consider only nearest
neighbors in the swapping algorithm. When nn=F, then the swapping
algorithm will consider all points in the candidate space. When nn=T,
then the swapping algorithm will consider only the num.nn closest
points for possible swapping. The default is to use nearest neighbors
only (nn=T).
}
\item{num.nn}{
Number of nearest-neighbors to search over. The default number is 100.
If nn=F then this argument will be ignore.
}
\item{max.loop}{
Maximum number of outer loops in algorithm. 30 is default.
}
\item{fixed}{
A matrix or vector specifying points to be forced into the
experimental design. If fixed is a matrix, it gives coordinates of the
fixed points in the design. In this case fixed must be a subset of
R. If fixed is a vector, then fixed gives the row numbers from the
candidate matrix R that identify the fixed points. The number of
points to be generated, nd, is in addition to the number of points
specified by fixed.
}
\item{scale.type}{
A character string that tells how to scale the candidate matrix, R,
before calculating distances. The default is "unscaled" in which case no
scaling is done. This is appropriate when all of the columns of R
are commensurate; for example, when R gives x and y in spatial
coordinates. When the columns of R are not commensurate, then it is
generally thought that an appropriate choice of scaling will provide a
better design. This would be the case, for example, for a typical
process optimization. Other choices for scale.type are "range", which scales
all columns of R to the range (0,1), "unit.sd", which scales
all columns of R to have 0 mean and unit standard deviation, and
"user", which allows a user specified scaling. We typically use the option
"range" for physical experiments.
}
\item{R.center}{
A vector giving the centering values if scale.type='user'.
}
\item{R.scale}{
A vector giving the scale values if scale.type='user'.
}
\item{P}{


A scalar value specifying a parameter of the criterion to be optimized. It
affects how the distance from a point x to a set of design points D is
calculated.  P=1 gives average distance. P=-1 gives harmonic mean distance.
P=-Inf would give minimum distance (not available as a value). As P gets
large and negative, points will tend to be more spread out.
}
\item{Q}{
A scalar value specifying a parameter of the criterion to be optimized. It
affects how distances from all points not in the design to points in the
design are averaged. When Q=1, simple averaging of the distances is employed.
Q=Inf (not available as a value) in combination with P=-Inf would give a
classical minimax design.
}
\item{start}{
A matrix or vector giving the initial design from which to start
optimization. If start is a matrix, it gives the coordinates of the
design points. In this case start must be a subset of R. If start is a
vector, then start gives the row numbers of the initial design. The
default is to use a random starting design.
}
\item{DIST}{
A distance metric in the form of an S function. If NULL, rdist is the 
default
which computes the 2-d Euclidean distance.
}
\item{return.grid}{
Logical value that tells whether or not to return the candidate matrix
as an attribute of the computed design. The default is return.grid=T.
The candidate matrix is used by plot.spatial.design if it is available.
}
\item{return.transform}{
Logical value that tells whether or not to return the trasformation 
attributes
of candidate set. The default is return.transform=F.
}}
\value{
Returns a design object of class "spatial.design", which inherits from
"data.frame" and "matrix". The design object has the following attributes:

\item{best.id}{
Row numbers of the final design from the original candidate matrix, R.
}
\item{fixed}{
Row numbers of the fixed points from the original candidate matrix, R.
}
\item{opt.crit}{
Value of the optimality criterion for the final design.
}
\item{start.design}{
Row numbers of the starting design from the original candidate matrix, R.
}
\item{start.crit}{
Value of the optimality criterion for the starting design.
}
\item{history}{
The swapping history and corresponding values of the optimality
criterion for the best design.
}
\item{other.designs}{
The designs other than the best design generated when nruns>1.
}
\item{other.crit}{
The optimality criteria for the other designs when nruns>1.
}
\item{DIST}{
The distance function used in calculating the design criterion.
}
\item{nn}{
Logical value for nearest-neighbor search or not.


}
\item{num.nn}{
The number of nearest neighbor set.
}
\item{grid}{
The matrix R is returned if the argument return.grid=T.
}
\item{transform}{
The type of transformation used in scaling the data and the values
of the centering and scaling constants if the argument return.transform=T.
}
\item{call}{
The calling sequence.
}
\item{P}{
The parameter value for calculating criterion.
}
\item{Q}{
The parameter value for calculating criterion.
}
\item{nhist}{
The number of swaps performed.


}
\item{nloop}{
The number of outer loops required to reach convergence if nloop is less
the max.loop.
}
\item{minimax.crit}{
The minimax design criterion using DIST.
}
\item{max.loop}{
The maximum number of outer loops.
}}
\details{
The default is to use the code that calls FORTRAN this function is also 
listed as cover.design.F. A version using just S code with the same 
features is cover.design.S. To switch from FORTRAN to S code just 
overwrite cover.design  ( cover.design<- cover.design.S). 


For nd design points di in the set D and nc candidate points ci in the 
set C,
the coverage criteria is defined as:


   M(D,C) = \{SUM_\{ci in C\}
            [sum_\{di in D\} (dist(di,ci)^P)]^(Q/P) \}^(1/Q)        [1]


Where P < 0, and Q > 0 are parameters.  The algorithm used in
"cover.design" to find the set of nd points in C that minimize this
criterion is an iterative swapping algorithm which will be described
briefly. The resulting design is referred to as a "coverage design"
from among the class of space-filling designs.


ALGORITHM:




 An initial set of nd points is chosen randomly 
if no starting
configuration is provided. The nc x nd distance matrix between the
points in C and the points in D is computed, and raised to the power P.
The "row sums" of this matrix are computed.  Denote these as rs_\{i\} and
the vector of row sums as rs.  Using rs, M(D,C) is computed as:


  (sum_\{i=1,nc\} (rs_\{i\})^(Q/P))^(1/Q)                               [2]


Note that if point d_\{i\} is "swapped" for point c_\{j\}, one must
only recompute 1 column of the original distance matrix, and 1 row.
The row elements not in the ith column will be the same for all j and
so only need computing when the first swapping occurs for each d_\{i\}.
Denote the sum of these off-i elements as "newrow(i)".  The index is
i here since this is the same for all rows j=1,\dots{}nc.


Thus, for each swap, the row sums vector is updated as


  rs(new) = rs(old) - column(i,old) + column(i,new)


And the jth element of rs(new) is replaced by:


  rs(new)[j] = column(i,new)[k] + newrow(i)


Finally, M(D,C) is computed for this swap of the ith design point
for the jth candidate point using [2].  The point in C that when
swapped produces the minimum value of M(D,C) replaces d_\{i\}.


This is done for all nd points in the design, and is iterated until
M(D,C) does not change.


When the nearest neighbor option is selected, then the points
considered for swapping are limited to the num.nn nearest neighbors
of the current design point.


STABILITY




The algorithm described above is guaranteed to converge.  However,
upon convergence, the solution is sensitive to the initial configuration
of points.  Thus, it is recommended that multiple optimizations be
done (i.e. set nruns > 1 ).  Also, the quality of the solution depends
on the density of the points on the region.  At the same time, for
large regions , optimization can be computationally prohibitive using
"cover.design.F" unless the nearest neighbor option is employed.
Up to now, we can be run upto 20,000 candidate points, 400 design points,
and 20 dimensions.




}
\references{
Johnson, M.E., Moore, L.M., and Ylvisaker, D. (1990).  Minimax and
maximin distance designs. Journal of Statistical Planning and
Inference 26, 131-148.


SAS/QC Software. Volume 2: Usage and Reference. Version 6. First
Edition (1995). "Proc Optex". SAS Institute Inc. SAS Campus Drive,


}
\seealso{
expand.grid,plot.spatial.design
}
\examples{
# first generating candidate set
test.df <- expand.grid(x=seq(0,1,0.05),y=seq(0,1,0.05))
# do not use nearest neighbor search
test1.des <- cover.design(R=test.df,nd=10,nn=F,scale.type="range")
# use nearest neighbor search method
test2.des <- cover.design.F(R=test.df,nd=10,num.nn=25,scale.type="range")
# If this design takes too long to converge, then you can calculate the
# design sequentially using max.loop & start.
test3.des <- cover.design(R=test.df,nd=10,num.nn=50,max.loop=2,
                scale.type="range",return.grid=F)


# this design stopped after 2 outer loops.
# next design starts with test3.des output until for 4 additional loops.
test4.des <- cover.design.F(R=test.df,nd=10,num.nn=50,max.loop=4,
                scale.type="range", return.grid=F,
                start=attr(test3.des,'best.id'))


#
#  evaluate predcition error surface from a coverage design
make.surface.grid( list( seq( 0,5,,20),seq( 0,5,,20)))-> xg
cover.design( xg, 5)-> des
as.matrix( des)-> des




se.cov( des, xg, exp.cov, theta=1.0) -> look
as.surface( xg, look)-> out.p


image.plot( out.p, graphics.reset=F)
points( des)   


 






}
\keyword{FUNFITS}
% Converted by Sd2Rd version 0.3-2.








