// SourceGenerator.cpp: implementation of the IMCSRCGenerator class.
//
//////////////////////////////////////////////////////////////////////
#pragma warning(disable:4786)

#include <string>
#include <fstream>
#include <iostream>
#include <sys/stat.h>         // stat function
#ifdef WIN32
#include <direct.h>           //_mkdir function
#endif

#include "IMCSRCGenerator.h"

using namespace std;

#ifdef WIN32
#define mkdir(dir, ignor) _mkdir(dir)
#define FILE_SEPERATOR "\\"
#else
#define FILE_SEPERATOR "/"
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

IMCSRCGenerator::IMCSRCGenerator(const char* pszOutputDir, const char* pszInputDir ):
m_pszInputDir(NULL), m_pszOutputDir(NULL)
{
	if(pszOutputDir && pszOutputDir[0])
	{
		//create dir if needed.

		struct stat status;
		if(stat(pszOutputDir, &status) != 0) //dir does not exist?
		{
			if(mkdir(pszOutputDir, 0) != 0)
				return;
		}

		if(stat(pszInputDir, &status) != 0) //dir does not exist?
			return;
	
		m_pszOutputDir = new char[strlen(pszOutputDir)+1];
		strcpy(m_pszOutputDir, pszOutputDir);

		m_pszInputDir = new char[strlen(pszInputDir)+1];
		strcpy(m_pszInputDir, pszInputDir);

	}
}

IMCSRCGenerator::~IMCSRCGenerator()
{
	if(m_pszInputDir)
		delete [] m_pszInputDir;
	if(m_pszOutputDir)
		delete [] m_pszOutputDir;
	m_pszInputDir = NULL;
	m_pszOutputDir = NULL;
}

long IMCSRCGenerator::isValid(void)
{
	return m_pszOutputDir && m_pszInputDir? 1L : 0L;
}

long IMCSRCGenerator::copyFile(const char* pszFileTo, const char* pszFileFrom, const char* pszModelName)
{
	if(!isValid())
		return 0L;
	if(!pszFileTo && !pszFileTo[0])
		return 0L;

	string strFileTo(m_pszOutputDir);
	strFileTo += FILE_SEPERATOR;
	strFileTo += pszFileTo;

  string strFileFrom(m_pszInputDir);
	strFileFrom += FILE_SEPERATOR;
	//If pszFileFrom is NULL or empty, fileFrom is the same as fileTo
	if(!pszFileFrom || !pszFileFrom[0])
		strFileFrom += pszFileTo;
	else
		strFileFrom += pszFileFrom;

	struct stat status;

	if(stat(strFileFrom.c_str(), &status) != 0) //file does not exist?
	{
		cout << "bad input file\n";
		return -1;
	}

	ifstream ifsFile(strFileFrom.c_str());
	if(!ifsFile)
	{
		cout << "cannot open input file for reading:";
		cout << strFileFrom;
		cout << ".\n";
		return -1;
	}

	ofstream ofsFile(strFileTo.c_str());
	if(!ofsFile)
	{
		cout << "cannot open output file for writing:";
		cout << strFileTo;
		cout << ".\n";
		return -1;
	}

//	char ch;
//	while(ifsFile.get(ch)) ofsFile.put(ch);
	char szLine[2048];
	string strLine;
	while(ifsFile.getline(szLine, 2040))
	{
		strLine = szLine;
		if(pszModelName && pszModelName[0])
		{
			long n = strLine.find("$$model");
			while(0<=n && n<strLine.length())
			{
				strLine.replace(n, 7, pszModelName);
				n = strLine.find("$$model");
			}
		}
		ofsFile.write(strLine.c_str(), strLine.length());
//#ifdef WIN32
//		ofsFile.put('\r');
//#endif
		ofsFile.put('\n');
	}
	return 1L;
}

long IMCSRCGenerator::copyCommonFiles(const char* pszModelName)
{
	long nStatus;
	nStatus=copyFile("readme.txt", "readme.txt", pszModelName);
	if(!nStatus)
		return nStatus;
	nStatus=copyFile("IMObjects.h");
	if(!nStatus)
		return nStatus;
	return copyFile("IMObjects.c");
}

long IMCSRCGenerator::copyAllFilesForModel(const char* pszModelName)
{
	long nStatus;

	nStatus=copyCommonFiles(pszModelName); 
	if(!nStatus)
		return nStatus;
	//.h for the model
	string strNameH("IM");
	strNameH += pszModelName;
	strNameH += ".h";
	nStatus= copyFile(strNameH.c_str());
	if(!nStatus)
		return nStatus;

	//.c for the model
	string strNameC("IM");
	strNameC += pszModelName;
	strNameC += ".c";
	nStatus= copyFile(strNameC.c_str());
	if(!nStatus)
		return nStatus;

	//main.c
	string strMain(pszModelName);
	strMain += "_main.c";

	return copyFile("main.c", strMain.c_str());

}

long IMCSRCGenerator::writeModelAndC(const IMinerObject& model, const char* modelName)
{
	long nStatus;
	if(!isValid())
		return 0L;

	//write to a directory specified for the constructor.
	nStatus = copyAllFilesForModel(modelName); 
	if(!nStatus)
		return nStatus;

	/* write the model data */
    string strModelFile(m_pszOutputDir);
	strModelFile += FILE_SEPERATOR; 
	strModelFile += "model.txt";
	nStatus = IMinerObject_writeToFile(strModelFile.c_str(), &model);

	return 1L;
}
long IMCSRCGenerator::writeModelInputAndC(const IMinerObject& model, const IMinerDataSet& rectDataInput, const char* modelName)
{
	writeModelAndC(model, modelName);

	/* write an input rect data */
    string strInputFile(m_pszOutputDir);
	strInputFile += FILE_SEPERATOR;
	strInputFile += "input.txt";
	//YB:test nStatus = IMinerDataSet_writeToFile(strInputFile.c_str(), &rectDataInput);	
	IMinerObject_writeToFile(strInputFile.c_str(), &rectDataInput);	

	return 1L;
}
