/**
 * An ObjPool object stores a collection of objects
 *   and provides thread-safe access to them.  A
 *   requesting thread is given a reference to the
 *   next object in the list, or is made to wait if
 *   all objects are currently in use.
 *
 * author Gary Nelson, gnelson@insightful.com
 * version 20 April 2001
 **/

import java.util.LinkedList;


public class ObjPool
{
	/* list of objects in the pool */
	protected LinkedList m_lstObjs;

	/**
	 * When the pool is closed, getObj returns null
	 *   (rather than waiting for an object to return).
	 **/
	protected boolean m_bClosed = false;
	public synchronized void setClosed(boolean bClosed) {
		m_bClosed = bClosed;
		if (bClosed) {
			// Tell everyone waiting in line
			//   that the pool is closed.
			notifyAll();
		}
	}

	public ObjPool() {
		m_lstObjs = new LinkedList();
	}

	public boolean hasObjs() {
		return m_lstObjs.size() > 0;
	}

	public synchronized void addObj(Object obj) {
		m_lstObjs.addLast(obj);
		// Wake up any threads waiting for an object.
		notifyAll();
	}

	public synchronized Object getObj() {
		while (m_lstObjs.size() == 0) {
			if (m_bClosed) { return null; }
			try {
				wait();
			}
			catch (InterruptedException e) {}
		}

		return m_lstObjs.removeFirst();
	}
}
