/**
 * An S-PLUS Connection Manager manages S-PLUS sessions
 *   for a Web site.
 * On initialization, a servlet calls the static function
 *   getManager to get a reference to the manager for its
 *   Web application.  This function creates the manager
 *   if it doesn't exist already; otherwise it simply
 *   returns a reference to the one already created.
 *   Thus several servlets can use the same manager.
 *   However, the getManager will create different
 *   managers for different "servlet contexts", which
 *   means that each Web application will have its own
 *   manager.
 * Each call to getManager should be balanced with a
 *   call to release.  When all servlets using a manager
 *   have released it, the manager causes all its
 *   sessions to disconnect.  In general a servlet should
 *   call getManager from its init method and release
 *   from its destroy method.
 * Version 2.3 of the Servlet API will provide an easier
 *   way to associate a single instance of the manager
 *   with the servlet context, from what I understand.
 *
 * author Gary Nelson, gnelson@insightful.com
 * version 24 April 2001
 */

import java.util.*;
import java.io.*;
import java.rmi.RemoteException;
import javax.servlet.ServletContext;
import com.insightful.splus.*;


public class SpConnectionManager
{
	/* pool of available S-PLUS connections */
	protected ObjPool m_pool = null;

	/* servlet context whose connection this object manages. */
	ServletContext m_ctx;

	/* number of servlets using this manager */
	protected int m_nNumServlets = 0;

	/* number of connections (available and currently in use) */
	protected int m_nNumConnections = 0;

	//===============================================================
	// Initialization
	//===============================================================

	public static final String SPLUS_MANAGER = "SplusManager";

	/**
	 * Create a new S-PLUS connection manager and store it as
	 *   an attribute for this servlet context.
	 **/
	public static synchronized SpConnectionManager getManager(
		ServletContext ctx)
	{
		SpConnectionManager mgr;

		// Check whether a manager is already available.
		// (Another servlet in the same context might have
		//   created it.)
		mgr = (SpConnectionManager)
			ctx.getAttribute(SPLUS_MANAGER);
		if (mgr != null) {
			mgr.m_nNumServlets++;
			return mgr;
		}

		// Start a new S-PLUS connection manager.
		mgr = new SpConnectionManager(ctx);
		mgr.init();

		// Set the reference count.
		mgr.m_nNumServlets = 1;

		return mgr;
	}

	/* constructor */
	protected SpConnectionManager(ServletContext ctx) {
		// Remember the context.
		m_ctx = ctx;

		// Set a context attribute to point to this manager, in case
		//   any additional servlets need a reference.
		ctx.setAttribute(SPLUS_MANAGER, this);
	}

	/**
	 * Initialize this S-PLUS connection manager.
	 * Create its object pool and populate it with connections.
	 **/
	protected void init() {
		m_pool = new ObjPool();

		// Get the XML file with connection information.
		String strConnectionsFilename =
			m_ctx.getInitParameter("connections-file");
		if (strConnectionsFilename == null) {
			System.out.println("Manager:  No connections " +
				"filename provided in the web.xml file");
			m_pool.setClosed(true);
			return;
		}

		File file = new File(m_ctx.getRealPath("/"),
			strConnectionsFilename);
		SpConnectionCreator creator = new SpConnectionCreator(m_pool);
		int nNewConnections = creator.parseFile(file);
		System.out.println("Manager: " + nNewConnections +
			" connections established");
		m_nNumConnections += nNewConnections;
		m_pool.setClosed(m_nNumConnections <= 0);
	}

	/**
	 * Release this manager.  The manager will disconnect all S-PLUS 
	 *   connections when the reference count drops to zero.
	 * Code using the connection manager must call release() once for
	 *   every call to getManager(); otherwise this shutdown code might
	 *   run either too early or not at all.
	 **/
	public void release() {
		if (--m_nNumServlets > 0) {
			return;
		}

		// Disconnect all connections, and remove them from the pool.
		while (m_pool.hasObjs()) {
			SpConnection con = getConnection();
			con.disconnect();
		}

		// Remove the context attribute pointing to this manager.
		m_ctx.removeAttribute(SPLUS_MANAGER);
	}

	//===============================================================
	// Type-safe access to the object pool
	//===============================================================

	public SpConnection getConnection() {
		return (SpConnection) m_pool.getObj();
	}

	public void releaseConnection(SpConnection connection) {
		m_pool.addObj(connection);
	}

	public void recycleConnection(SpConnection connection) {
		System.out.println("Manager: connection \"" +
			connection.getFriendlyName() + "\" recycled:");
		System.out.println("\t" +
			((connection.getLastError() == null) ?
			"(Error not reported)" : connection.getLastError()));

		// For now we just delete bad connections.
		// A better idea might be to store bad connections
		//   and provide some mechanism to fix or replace
		//   them.

		// Decrement the connection count
		if (--m_nNumConnections > 0) {
			return;
		}
		// No connections remain, so close the pool.
		m_pool.setClosed(true);
	}

	public void addNewConnection(SpConnection connection) {
		m_nNumConnections++;
		m_pool.addObj(connection);
	}

	//===============================================================
	// SplusSession functionality
	//===============================================================
	// Perhaps this class should provide a robust version of the
	//   SplusSession interface.  Any function called would be passed
	//   on to a connection; in case of a remote exception it would
	//   be passed to another connection without the caller needing
	//   to know.
	// This feature is nice, and eventually we might write it.
	//   However callers with sequences of function calls (e.g.
	//   create a graphlet, transfer the file) will want them to run
	//   on a single connection.

	/* Evaluate an S-PLUS expression and return the result. */
	public SplusDataResult eval(String strExp,
		boolean bGetOutput, boolean bGetResult,
		boolean bGetErrors, boolean bGetWarnings, boolean bGetExpr)
	{
		while (true) {  // If the first connection fails, try another.
			SpConnection con = getConnection();
			if (con == null) {
				return new SplusDataResult("",
					"No S-PLUS connections available",
					strExp, new String[0]);
			}
			try {
				SplusDataResult spdr = con.eval(strExp, bGetOutput,
					bGetResult, bGetErrors, bGetWarnings, bGetExpr);
				releaseConnection(con);
				return spdr;
			}
			catch (RemoteException ex) {
				recycleConnection(con);
			}
		}
	}
}
