/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
*/

// spchar.cxx: implementation of the CSPcharacter class.
//
//////////////////////////////////////////////////////////////////////

#include <string>

#ifdef __hppa
#include <rw/exception.h>
#endif

#include "spchar.h"
#include "spcall.h"
#include "spconnec.h"
#include "sys_codes.h"

#ifdef __hppa
namespace std{};
#endif

#include <string>
using namespace std;

//Direct access to elements of character vector (ZERO-BASED INDEX). Use it carefully!
static s_boolean SPL_SetCharAtDirect(s_object* ps_object, //in/out 
   char** ppszValues, long lStartIndex, long lEndIndex, BOOL bRemovableStrings)
{
	if((lStartIndex > lEndIndex) 
	|| (ps_object->mode != S_MODE_CHAR) || (ppszValues == NULL)
	|| (lStartIndex >= ps_object->length) || (lEndIndex >= ps_object->length))
		return S_FALSE;  

	if(S_is_env_removable_char())
	{
		long nValue=0;
		for(long n=lStartIndex; n<= lEndIndex; n++)
		{
			if(ppszValues[nValue])
				CHARACTER_POINTER(ps_object)[n] = (char*)ppszValues[nValue]; 
			else
				CHARACTER_POINTER(ps_object)[n] = NULL_STRING; 
			nValue++;
		}

		if(bRemovableStrings) //Make it as removable character so that we can free it when no longer needed. 
			S_copy_strings(ps_object, ps_object, S_evaluator);
		else //Mark it as non-removable
			S_set_arena_why(ps_object, ATOMIC_WHY, S_evaluator);

	}
	else
	{
		long nValue=0;

		for(long n=lStartIndex; n<= lEndIndex; n++)
			CHARACTER_POINTER(ps_object)[n] = SPL_AllocateStringInFrame(ps_object, ppszValues[nValue++]); //pdValues[nValue] better be valid!
	}

	return S_TRUE;
}

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

//Default constructor
CSPcharacter::CSPcharacter()
: TSPvector<char*, S_MODE_CHAR>()
{
}
//Copy constructor 
CSPcharacter::CSPcharacter(const CSPcharacter& sObject)
: TSPvector<char*, S_MODE_CHAR>()
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a base class object
CSPcharacter::CSPcharacter(const CSPobject& sObject)
: TSPvector<char*, S_MODE_CHAR>()
{	
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a valid S-expression
CSPcharacter::CSPcharacter(const char* pszExpression)
: TSPvector<char*, S_MODE_CHAR>(pszExpression)
{
}
//Construct from a valid S object
CSPcharacter::CSPcharacter(s_object* ps_object, BOOL bTryToFreeOnDetach)
: TSPvector<char*, S_MODE_CHAR>()
{
	Attach(ps_object, bTryToFreeOnDetach);
}
//Assigment from the same class
CSPcharacter& CSPcharacter::operator=(const CSPcharacter& sObject)
{
	Attach(&sObject, TRUE);
	return *this;
}
//Assigment from the base class
CSPcharacter& CSPcharacter::operator=(const CSPobject& sObject)
{
	Attach(&sObject, TRUE);
	return *this;
}
//Assigment from the S object
CSPcharacter& CSPcharacter::operator=(s_object* ps_object)
{
	Attach(ps_object, FALSE);
	return *this;
}
//The destructor
CSPcharacter::~CSPcharacter()
{
}
//////////////////// Other constructor/destructor and assignment operators

//Construct a 'character' vector of length nLength
CSPcharacter::CSPcharacter(long lLength)
: TSPvector<char*, S_MODE_CHAR>(lLength)
{
}

CSPcharacter::CSPcharacter(int nLength)
: TSPvector<char*, S_MODE_CHAR>((long) nLength)
{
}

CSPcharacter::CSPcharacter(char** ppszValues, long lLength)
: TSPvector<char*, S_MODE_CHAR>()
{
	s_object* ps_object = NewMyType(lLength);
	if(SPL_SetCharAtDirect(ps_object, ppszValues, 0L, lLength-1, TRUE) )
		Attach(ps_object);
}
//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////

//Return index (ZERO-BASED) for a given char string, return -1 if not found.
long CSPcharacter::Find(const char* pszText, BOOL bValidate) const
{
	if(bValidate && (!IsValid()))
	SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT); 
	
	long lLen = GetLength(FALSE);
	s_object* ps_object = &(*this);
	for (long lIndex=0; lIndex<lLen; lIndex++ )
	{
		char *pszNameFound = CHARACTER_POINTER(ps_object)[lIndex];
		if ( strcmp( pszNameFound, pszText ) == 0 )
			return lIndex;
	}

	//Not found
	return -1L;
	 
}

//////////////////////////////////////////////////////
// Operations
//////////////////////////////////////////////////////

CSPobject CSPcharacter::Eval(void)
{
	return CSPobject(&(*this));
}

CSPobject CSPcharacter::ParseEval(void) 
{
	CSPevaluator sEvaluator;

	CSPtextConnection stextConnection(&(*this));
	s_object* ps_arglist = NEW_LIST(2);
	LIST_POINTER(ps_arglist)[0] = stextConnection.Parse();
	LIST_POINTER(ps_arglist)[1] = blt_in_TRUE;
	S_evaluator->_sys_index = S_EVAL;
	s_object* ps_object = S_dummy(blt_in_NULL, ps_arglist, S_evaluator);

	return CSPobject(sEvaluator.CloneIfNeeded(ps_object));
		
}

//Append a string to the end fo the vector.
long CSPcharacter::Add(const char *pszElement, BOOL bValidate)
{
	if(bValidate && (GetPtr() != NULL && !IsValid()))
	SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);  

	long lCurLength = 0;
	if ( GetPtr() != NULL )
	{
		lCurLength = GetLength( FALSE );
		if ( lCurLength < 0 )
			lCurLength = 0;
	}
	lCurLength++;

	s_object* ps_object = NewMyType(lCurLength);
	//Put back in the old strings
	for(long n=0; n< lCurLength-1; n++)
		CHARACTER_POINTER(ps_object)[n] = CHARACTER_POINTER(GetPtr())[n];

	if(SPL_SetCharAtDirect(ps_object, (char**) &pszElement, lCurLength-1, lCurLength-1, TRUE) )
		ReAttachAndAssign(ps_object);
	return lCurLength-1;
}

long CSPcharacter::Append(char **ppszElements, long lElements, BOOL bValidate)
{
	long elementsAdded = 0;
	if ( !ppszElements || lElements <= 0 )
		return elementsAdded;

	if(bValidate && (GetPtr() != NULL && !IsValid()))
	SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);  
	
	if(S_is_env_removable_char())
	{
		long lCurLength = 0;
		if ( GetPtr() != NULL )
		{
			lCurLength = GetLength( FALSE );
			if ( lCurLength < 0 )
				lCurLength = 0;
		}

		//TODO: some cases we don't need to allocate new object if ps_object->nalloc > ps_object->length && ???
		s_object* ps_object = NewMyType(lCurLength+lElements);

		//Copy current strings to the new longer-length object.
		for(long n=0; n< lCurLength; n++)
			CHARACTER_POINTER(ps_object)[n] = CHARACTER_POINTER(GetPtr())[n];
	
		if(SPL_SetCharAtDirect(ps_object, ppszElements, lCurLength, (long)(lElements-1), TRUE) )
		{
			elementsAdded = lElements;
			ReAttachAndAssign(ps_object);
		}
	}
	else
	{
		long curLength = 0;
		try
		{
			if ( GetPtr() == NULL )
				Attach(NEW_CHARACTER(0));

			curLength = GetLength( bValidate );
			if ( curLength < 0 )
				curLength = 0;
		}
		catch(...)
		{
		}
		SetLength(curLength+lElements);
		long curIndex = curLength;
		for ( int i=0; i<lElements; i++ )
		{
			char *pszElement = ppszElements[i];
			if ( !pszElement )
				SetAt(curIndex, "");
			else
				SetAt(curIndex, pszElement, FALSE);
			curIndex++;
		}
	}

	return elementsAdded;
}
//Safe access to elements of character vector (ZERO-BASED INDEX): copy if needed.
void CSPcharacter::SetAt(long lIndex, const char* pszValue, BOOL bValidate)
{
	SetAt((char**)&pszValue, lIndex, lIndex, bValidate);
}

//Safe access to elements of character vector (ZERO-BASED INDEX): copy if needed.
void CSPcharacter::SetAt(char** ppszValues, long lStartIndex, long lEndIndex, BOOL bValidate)
{
	if(bValidate  && (!IsValid() || (lStartIndex > lEndIndex)	))
	SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);  

	if(ppszValues == NULL)
		return;
	
	s_object* ps_object = GetPtr();
	if(S_is_env_removable_char())
	{
		long n;
		long lCurrLength= ps_object->length;
		ps_object = NewMyType((lCurrLength<=lEndIndex?lEndIndex+1:lCurrLength));

		//Copy current strings to the new longer-length object.
		for(n=0; n< lCurrLength; n++)
			CHARACTER_POINTER(ps_object)[n] = CHARACTER_POINTER(GetPtr())[n];

		//Insert the empty strings
		for(n=lCurrLength; n< lStartIndex; n++)
			CHARACTER_POINTER(ps_object)[n] = NULL_STRING;

		if(SPL_SetCharAtDirect(ps_object, ppszValues, lStartIndex, lEndIndex, TRUE))
			ReAttachAndAssign(ps_object);
	}
	else
	{
		if(GetLength(FALSE) <= lEndIndex)
			SetLength(lEndIndex+1, FALSE);
		for(long n=lStartIndex; n<= lEndIndex; n++)
			CHARACTER_POINTER(ps_object)[n] = SPL_AllocateStringInFrame(ps_object, ppszValues[n]); //pdValues[n] better be valid!
	}
}

//Direct access to elements of character vector (ZERO-BASED INDEX). Use it carefully!
void CSPcharacter::SetAtDirect(long lIndex, const char* pszValue, BOOL bValidate, BOOL bRemovableStrings)
{
	if(bValidate && (!IsValid()) || (lIndex >= GetLength(FALSE)))
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);  

	if(!SPL_SetCharAtDirect(GetPtr(), (char**)&pszValue, lIndex, lIndex, bRemovableStrings) )
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);  
}

//Direct access to elements of character vector (ZERO-BASED INDEX). Use it carefully!
void CSPcharacter::SetAtDirect(char** ppszValues, long lStartIndex, long lEndIndex, BOOL bValidate, BOOL bRemovableStrings)
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);  

	if(!SPL_SetCharAtDirect(GetPtr(), ppszValues, lStartIndex, lEndIndex, bRemovableStrings) )
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);  

}

//Make strings removable if they are not.
void CSPcharacter::MakeStringsRemovable( BOOL bValidate )
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);  

	s_object* ps_object = GetPtr();
	if(S_is_env_removable_char() && !S_is_removable_char(ps_object, S_evaluator))
	{
		char* psz_old_strings = NULL;
		if(S_get_arena_why(ps_object, S_evaluator) == CHAR_WHY) // removable char?
			psz_old_strings = CHARACTER_POINTER(ps_object)[0];

		//Make it as removable character so that we can free it when no longer needed. 
		S_copy_strings(ps_object, ps_object, S_evaluator);
		if(psz_old_strings) // previously removable char?
			S_remove_strings(psz_old_strings, S_evaluator);
	}
}

//////////////////////////////////////////////////////
// Proxy implementation
CSPcharacter::CProxy::CProxy(CSPcharacter& sVector, long lZeroBasedIndex)
:TSPvector<char*, S_MODE_CHAR>::CSPvectorProxy(sVector, lZeroBasedIndex)
{
}		
CSPcharacter::CProxy::CProxy(const CProxy& proxyRhs)
:TSPvector<char*, S_MODE_CHAR>::CSPvectorProxy(proxyRhs.m_vector, proxyRhs.m_lIndex)
{
}		
//Assignment operators:
CSPcharacter::CProxy& CSPcharacter::CProxy::operator=(const char* pszRhs)
{
	static_cast<CSPcharacter&>(m_vector).SetAt(m_lIndex, pszRhs, FALSE);
	return *this;
}

CSPcharacter::CProxy& CSPcharacter::CProxy::operator=(const CSPcharacter::CProxy& proxyRhs)
{
	const char* pszLhs = (const char* ) proxyRhs;
	static_cast<CSPcharacter&>(m_vector).SetAt(m_lIndex, pszLhs, FALSE);
	return *this;
}

BOOL CSPcharacter::CProxy::operator!=(const char* pszRhs) const
{
	const char* pszLhs = (const char*)(*this);
	size_t len = strlen(pszRhs);
	if(strlen(pszLhs) != len)
		return TRUE;
	else
		return strncmp(pszLhs, pszRhs, len);
}
BOOL CSPcharacter::CProxy::operator==(const char* pszRhs) const
{
	return !operator!=(pszRhs);
}

CSPcharacter& CSPcharacter::operator=(const CSPcharacter::CProxy& proxyRhs)
{
	char* pszElement = (char*) proxyRhs.m_vector.GetAt(proxyRhs.m_lIndex);
	s_object* ps_value = NewMyType(1L);
	if(SPL_SetCharAtDirect(ps_value, &pszElement, 0L, 0L, TRUE) )
		ReAttachAndAssign(ps_value);
	return *this;
}

