/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001

	spproxy.h: interface for the CSPproxy class.
*/

#if !defined(__SCONNECT_SPPROXY_H_INCLUDED__)
#define __SCONNECT_SPPROXY_H_INCLUDED__

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include "spobject.h"

// CSPproxy is an unsafe wraper to an s_object* to assist its derive class.
// Common member functions are useful for 'proxy' derive classes to efficiently
// access and change the object.  However, these are unsafe for 
// public usage.
//
// WARNING: All member functions are not safe to be used by public .
//

class SCONNECT_LIB_EXTERN CSPproxy  
{
public:

	CSPproxy();
	CSPproxy(CSPobject* pSPobject, long lZeroBasedIndex);
	CSPproxy(const CSPproxy& rhs); //copy constructor
	virtual ~CSPproxy();

	//this constructor creates CSPobject* and the destructor will delete it
	CSPproxy(s_object* ps_object, long lZeroBasedIndex); 

	CSPproxy& operator=(s_boolean bRhs)
	{
		SetElementAs((void*)&bRhs, S_MODE_LGL);
		return *this;
	}
	CSPproxy& operator=(long lRhs)
	{
		if(m_pSPobject->GetVector()->mode == S_MODE_LGL)
			SetElementAs((void*)&lRhs, S_MODE_LGL);
		else
			SetElementAs((void*)&lRhs, S_MODE_INT);
		return *this;
	}
	CSPproxy& operator=(int nRhs)
	{
		long lRhsPass = (long)nRhs;
		SetElementAs((void*)&lRhsPass, S_MODE_INT);
		return *this;
	}
	CSPproxy& operator=(float fRhs)
	{
		SetElementAs((void*)&fRhs, S_MODE_REAL);
		return *this;
	}
	CSPproxy& operator=(double dRhs)
	{
		SetElementAs((void*)&dRhs, S_MODE_DOUBLE);
		return *this;
	}
	CSPproxy& operator=(char* pszText)
	{
		SetElementAs((void*)pszText, S_MODE_CHAR);
		return *this;
	}
	CSPproxy& operator=(const char* pszText)
	{
		SetElementAs((void*)pszText, S_MODE_CHAR);
		return *this;
	}

	CSPproxy& operator=(const CSPproxy& proxyRhs);

	//Cast-able to double?
	BOOL canbeDouble(void) const
	{
		int nMode= mode();
		if( (nMode == S_MODE_LGL) || (nMode == S_MODE_REAL) || (nMode == S_MODE_INT) || (nMode == S_MODE_DOUBLE))
			return TRUE;
		else
			return FALSE;
	}

	//Numerical comparisons

	BOOL equal(double dRhs) const ;
	BOOL lessThan(double dRhs) const ;
	BOOL lessThanEqual(double dRhs) const ;
	BOOL greaterThan(double dRhs) const ;
	BOOL greaterThanEqual(double dRhs) const ;

	BOOL operator==(long lRhs) const {return equal((double)lRhs);}
	BOOL operator==(int nRhs) const {return equal((double)nRhs);}
	BOOL operator==(float fRhs) const {return equal((double)fRhs);}
	BOOL operator==(double dRhs) const {return equal(dRhs);}

	BOOL operator!(){return !equal(0.0);}

	BOOL operator!=(long lRhs) const {return !equal((double)lRhs);}
	BOOL operator!=(int nRhs) const {return !equal((double)nRhs);}
	BOOL operator!=(float fRhs) const {return !equal((double)fRhs)  ;}
	BOOL operator!=(double dRhs) const {return !equal(dRhs);}

	BOOL operator<(long lRhs) const {return lessThan((double)lRhs);}
	BOOL operator<(int nRhs) const {return lessThan((double)nRhs);}
	BOOL operator<(float fRhs) const {return lessThan((double)fRhs);}
	BOOL operator<(double dRhs) const {return lessThan(dRhs);}

	BOOL operator<=(long lRhs) const {return lessThanEqual((double)lRhs);}
	BOOL operator<=(int nRhs) const {return lessThanEqual((double)nRhs);}
	BOOL operator<=(float fRhs) const {return lessThanEqual((double)fRhs);}
	BOOL operator<=(double dRhs) const {return lessThanEqual(dRhs) ;}

	BOOL operator>(long lRhs) const {return greaterThan((double)lRhs);}
	BOOL operator>(int nRhs) const {return greaterThan((double)nRhs);}
	BOOL operator>(float fRhs) const {return greaterThan((double)fRhs);}
	BOOL operator>(double dRhs) const {return greaterThan(dRhs);}

	BOOL operator>=(long lRhs) const {return greaterThanEqual((double)lRhs);}
	BOOL operator>=(int nRhs) const {return greaterThanEqual((double)nRhs);}
	BOOL operator>=(float fRhs) const {return greaterThanEqual((double)fRhs);}
	BOOL operator>=(double dRhs) const {return greaterThanEqual(dRhs);}

	//Character comparisons
	BOOL operator==(const char* pszRhs) const ;
	BOOL operator==(const CSPproxy&  proxyRhs) const ;

	BOOL operator!=(const char* pszRhs) const ;
	BOOL operator!=(const CSPproxy&  proxyRhs) const ;

	//Helpers:
	virtual long CopyIfNeeded(long lMode);

	//Set the element to the value void* pRhs which is casted as lMode
	virtual void SetElementAs( void* pRhs, long lMode, BOOL bCopyIfNeeded = TRUE);

	//Get pointer to the element and cast it to void*
	virtual void* GetElementPtr(void) const;

	//Get the address of data for the current index
	virtual void* operator&(){return GetElementPtr();}

	void SetIndex(long lIndex){m_lIndex = lIndex;}
	long GetIndex(void) const {return m_lIndex;}

	//Return S_MODE: double, integer, char, etc..
	long mode(BOOL bValidate=TRUE) const
	{
		return (long) m_pSPobject->GetDataMode();
	}

	// Auto type conversions
	operator double() const;
	operator long() const;
	operator int() const; 
	operator float() const;

	// WARNING: the returned pointer points to the internal data.
	// Be very carefull when modifying its content.
	operator char*() const 
	{
		if (m_pSPobject->GetVector()->mode == S_MODE_CHAR)
			return CHARACTER_POINTER(m_pSPobject->GetVector())[m_lIndex];
		else
			return NULL;
	}

	//virtual operator CSPcomplex() {return (CSPcomplex) GetElementAs((CSPcomplex) NULL);}

	//Binary operations involving (CSPproxy) op (long)  
	s_object* BinaryOp(long lRhs, long lBinOp) const;
	//Binary operations involving (CSPproxy) op (float)  
	s_object* BinaryOp(float fRhs, long lBinOp) const;
	//Binary operations involving (CSPproxy) op (double)  
	s_object* BinaryOp(double dRhs, long lBinOp) const;
	//Binary operations involving (CSPproxy) op (CSPproxy)
	s_object* BinaryOp(const CSPproxy& proxyRhs, long lBinOp) const;

	CSPproxy operator+(long lRhs) const {return CSPproxy(BinaryOp(lRhs, S_BINOP_PLUS), 0);}
	CSPproxy operator-(long lRhs) const {return CSPproxy(BinaryOp(lRhs, S_BINOP_MINUS), 0);}
	CSPproxy operator*(long lRhs) const {return CSPproxy(BinaryOp(lRhs, S_BINOP_MULTIPLY), 0 );}
	CSPproxy operator/(long lRhs) const {return CSPproxy(BinaryOp(lRhs, S_BINOP_DIVIDE), 0);}

	CSPproxy operator+(float fRhs) const {return CSPproxy(BinaryOp(fRhs, S_BINOP_PLUS), 0);}
	CSPproxy operator-(float fRhs) const {return CSPproxy(BinaryOp(fRhs, S_BINOP_MINUS), 0);}
	CSPproxy operator*(float fRhs) const {return CSPproxy(BinaryOp(fRhs, S_BINOP_MULTIPLY), 0 );}
	CSPproxy operator/(float fRhs) const {return CSPproxy(BinaryOp(fRhs, S_BINOP_DIVIDE), 0 );}

	CSPproxy operator+(double dRhs) const {return CSPproxy(BinaryOp(dRhs, S_BINOP_PLUS), 0);}
	CSPproxy operator-(double dRhs) const {return CSPproxy(BinaryOp(dRhs, S_BINOP_MINUS), 0);}
	CSPproxy operator*(double dRhs) const {return CSPproxy(BinaryOp(dRhs, S_BINOP_MULTIPLY), 0);}
	CSPproxy operator/(double dRhs) const {return CSPproxy(BinaryOp(dRhs, S_BINOP_DIVIDE), 0);}

	CSPproxy operator+(const CSPproxy& proxyRhs) const {return CSPproxy(BinaryOp(proxyRhs, S_BINOP_PLUS), 0);}
	CSPproxy operator-(const CSPproxy& proxyRhs) const {return CSPproxy(BinaryOp(proxyRhs, S_BINOP_MINUS), 0);}
	CSPproxy operator*(const CSPproxy& proxyRhs) const {return CSPproxy(BinaryOp(proxyRhs, S_BINOP_MULTIPLY), 0);}
	CSPproxy operator/(const CSPproxy& proxyRhs) const {return CSPproxy(BinaryOp(proxyRhs, S_BINOP_DIVIDE), 0);}

	friend SCONNECT_DLLAPI( s_object* ) SPL_BinaryOp( void* , long , void* , long , long , BOOL );			

protected:
	long m_lIndex;
	CSPobject* m_pSPobject; //C++ object that this proxy represents.
	BOOL m_bTempSPobject; //TRUE if m_pSPobject points to a temp obj deletable by the destructor
};

SCONNECT_DLLAPI( s_object* ) SPL_BinaryOp( 
	void* pvLhs, long lLhsMode, //pointer to left-hand-side of op and its mode
	void* pvRhs, long lRhsMode, //pointer to right-hand-side of op and its mode
	long lBinOp,							  //operation: S_BINOP_PLUS, S_BINOP_MINUS, etc...
	BOOL bCopyIfNeeded);			  //TRUE allow copy if needed

//Binary operations involving (double) op (CSPproxy)
SCONNECT_DLLAPI( s_object* ) SPL_BinaryOp(double dLhs, const CSPproxy& proxyRhs,  long lBinOp); 
//Binary operations involving (double) op (CSPproxy)
SCONNECT_DLLAPI( s_object* ) SPL_BinaryOp(long lLhs, const CSPproxy& proxyRhs,  long lBinOp); 
//Binary operations involving (double) op (CSPproxy)
SCONNECT_DLLAPI( s_object* ) SPL_BinaryOp(float fLhs, const CSPproxy& proxyRhs,  long lBinOp) ;

//(double) op= CSPprox
inline CSPproxy operator+=(double dLhs, const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(dLhs, proxyRhs, S_BINOP_PLUS), 0);
}
inline CSPproxy operator-=(double dLhs,  const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(dLhs, proxyRhs, S_BINOP_MINUS), 0);
}
inline CSPproxy operator*=(double dLhs,  const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(dLhs, proxyRhs, S_BINOP_MULTIPLY), 0);
}
inline CSPproxy operator/=(double dLhs,  const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(dLhs, proxyRhs, S_BINOP_DIVIDE), 0);
}

//(long) op= CSPprox
inline CSPproxy operator+=(long lLhs, const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(lLhs, proxyRhs, S_BINOP_PLUS), 0);
}
inline CSPproxy operator-=(long lLhs,  const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(lLhs, proxyRhs, S_BINOP_MINUS), 0);
}
inline CSPproxy operator*=(long lLhs,  const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(lLhs, proxyRhs, S_BINOP_MULTIPLY), 0);
}
inline CSPproxy operator/=(long lLhs,  const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(lLhs, proxyRhs, S_BINOP_DIVIDE), 0);
}

//(double) op CSPprox
inline CSPproxy operator+(double dLhs, const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(dLhs, proxyRhs, S_BINOP_PLUS), 0);
}
inline CSPproxy operator-(double dLhs,  const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(dLhs, proxyRhs, S_BINOP_MINUS), 0);
}
inline CSPproxy operator*(double dLhs,  const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(dLhs, proxyRhs, S_BINOP_MULTIPLY), 0);
}
inline CSPproxy operator/(double dLhs,  const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(dLhs, proxyRhs, S_BINOP_DIVIDE), 0);
}
//(long) op CSPprox
inline CSPproxy operator+(long lLhs, const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(lLhs, proxyRhs, S_BINOP_PLUS), 0);
}
inline CSPproxy operator-(long lLhs,  const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(lLhs, proxyRhs, S_BINOP_MINUS), 0);
}
inline CSPproxy operator*(long lLhs,  const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(lLhs, proxyRhs, S_BINOP_MULTIPLY), 0);
}
inline CSPproxy operator/(long lLhs,  const CSPproxy& proxyRhs)
{
	return CSPproxy(SPL_BinaryOp(lLhs, proxyRhs, S_BINOP_DIVIDE), 0);
}

s_boolean SCONNECT_SetElement(s_object* ps_object, long nIndex, long lRhsMode, void* pRhs);

#endif // !defined(__SCONNECT_SPPROXY_H_INCLUDED__)

