/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
*/

// spvector.cxx: implementation of the CSPvector class.
//
//////////////////////////////////////////////////////////////////////

#include <memory.h>
#include "spvector.h"
#include "spexcept.h"
#include "spalcfrm.h"
#include "spproxy.h" //TODO: move SCONNECT_SetElement to speval.cxx and remove this line

#include "spchar.h"

CSPvector::CSPvector()
:CSPobject()
{
#ifdef WIN32
	m_b2DGet = FALSE;
#endif // WIN32
}

CSPvector::CSPvector(const CSPobject& sObject)
{
#ifdef WIN32
	m_b2DGet = FALSE;
#endif // WIN32

	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}

CSPvector::CSPvector(int mode, long length)
{
#ifdef WIN32
	m_b2DGet = FALSE;
#endif // WIN32

	try
	{
		CSPallocFrame sAllocFrame; //Set alloc frame to perm if needed.	
		s_object* ps_object = sAllocFrame.alcvec(mode, length);
		Attach(ps_object);
	}
	catch(...)
	{
	}
}

CSPvector::~CSPvector()
{
}

//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////

void* CSPvector::GetElementPtr(long lZeroIndex, BOOL bValidate) const
{
	if(bValidate)
		Validate();
	void* pReturn=NULL;
	s_object* ps_object = GetPtr();
	switch(ps_object->mode) 
	{
		case S_MODE_LGL:
		case S_MODE_INT:
			pReturn = &INTEGER_POINTER(ps_object)[lZeroIndex];
			break;
		case S_MODE_REAL:
			pReturn = &SINGLE_POINTER(ps_object)[lZeroIndex];
			break;
		case S_MODE_DOUBLE:
			pReturn = &NUMERIC_POINTER(ps_object)[lZeroIndex];
			break;
		case S_MODE_CHAR:
			pReturn =	CHARACTER_POINTER(ps_object)[lZeroIndex]; 
			break;
		case S_MODE_COMPLEX:
			pReturn =	&COMPLEX_POINTER(ps_object)[lZeroIndex] ;
			break;
		case S_MODE_RAW:
			pReturn =	&RAW_POINTER(ps_object)[lZeroIndex] ;
		default: 
			return NULL;
	}
	return pReturn;
}
void CSPvector::SetElementPtr(long lZeroIndex, long lRhsMode, const void* pRhs, BOOL bValidate)
{
	CSPevaluator sEvaluator;
	s_object* ps_object = &(*this);

	if(ps_object==NULL)
	{
		if(lZeroIndex < 0L )
			SCONNECT_ThrowException("Index is out of range: %d", lZeroIndex);
		ps_object = sEvaluator.alcvec(lRhsMode, lZeroIndex+1L);
		Attach(ps_object);
	}
	if(bValidate)
	{
		Validate();
		if(lZeroIndex < 0L )
			SCONNECT_ThrowException("Index is out of range: %d", lZeroIndex);
		else if(GetLength(FALSE) <= lZeroIndex)
			SET_LENGTH(ps_object, lZeroIndex+1L);
	}
	if(IsSharing(FALSE))
	{
		ps_object = SPL_Clone(ps_object, FALSE);
		SCONNECT_SetElement(ps_object, lZeroIndex, lRhsMode, (void*) pRhs);
		ReAttachAndAssign(ps_object);
	}
	else
		SCONNECT_SetElement(ps_object, lZeroIndex, lRhsMode, (void*) pRhs);
}

long CSPvector::GetNCol(BOOL bValidate) const
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
	return 1;
}

long CSPvector::GetNRow(BOOL bValidate) const
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
	return GetLength(bValidate);
}

long CSPvector::GetLength(BOOL bValidate) const
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	long length = GetPtr()->length;
	if ( IS(GetPtr(), MAKE_CLASS("factor")) )
	{
		length = DATA_SLOT(GetPtr())->length;
	}
	else if ( IS(GetPtr(), MAKE_CLASS("timeDate")) )
	{
		length = LIST_POINTER(DATA_SLOT(GetPtr()))[0]->length;
	}
	return length;
}

void CSPvector::SetLength(long lLength, BOOL bValidate) 
{
	if ( bValidate && !IsValid() )
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
	if ( lLength < 0 )
		CSPexception("Invalid length", lLength);

	if ( GetLength() == lLength )
		return;

	try
	{
		CSPevaluator sEvaluator;

		//Clone it so that SET_LENGTH() does not destroy the old object. 
		//It could alloc new block to replace the old one in the case of extending the vector length.
		s_object* ps_object = SPL_Clone(GetPtr(), FALSE);

		SET_LENGTH(ps_object, lLength);	
		ReAttachAndAssign(ps_object);
	}
	catch(...)
	{
		SCONNECT_ThrowException("Failed to set length for object of class %s", GetClass()->name);
	}
}

void* CSPvector::GetRawPtr(BOOL bValidate) const
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	void *pRawPtr = GetPtr()->value.name;
	if ( IS(GetPtr(), MAKE_CLASS("factor")) ||
		  IS(GetPtr(), MAKE_CLASS("timeDate")) )
	{
		pRawPtr = (void*)(LIST_POINTER(GetPtr())[0]->value.name);
	}
	return pRawPtr;
}

// Non-destructive way to change the length of a vector (CRD)
BOOL CSPvector::ChangeLength( long lNewLength, BOOL bValidate )
{
	if ( bValidate && !IsValid() )
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	BOOL bSuccess = FALSE;

	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator s;

		long lCurrLength = GetLength();
		if ( lCurrLength == lNewLength )
			return bSuccess;
		s_object* ps_new = NULL;
		int nCurrMode = GetMode();
		if ( nCurrMode == S_MODE_LIST && !IS_OBJ(&(*this),list) )
		{
			ps_new =	SPL_ExtendVector(&(*this), lNewLength-lCurrLength, -1);
		}
		else
		{
			long lCopyLength = ( lNewLength < lCurrLength ) ? lNewLength:lCurrLength;
			void *vpCurrData = GetRawPtr(FALSE);
			ps_new = SPL_GenerateVector( nCurrMode, lNewLength );
			if ( !SPL_NotThere(ps_new) && lNewLength > 0)
			{
				int nElemSize = mode_size(nCurrMode);
				if ( vpCurrData != NULL )
				{
					if ( nCurrMode == S_MODE_CHAR )
					{
						copy_strings(ps_new->value.Char, (char **)vpCurrData, lCopyLength, S_evaluator);
					}
					else if ( nCurrMode == S_MODE_LIST )
					{
						s_object** ppCurList = (s_object**)vpCurrData;
						for ( long i=0; i<lCopyLength; i++ )
						{
							LIST_POINTER(ps_new)[i] = ppCurList[i];
						}
					}
					else
					{
						long nBytes = nElemSize * lCopyLength;
						memcpy((void*)(ps_new->value.name), vpCurrData, nBytes);
					}
				}
				if ( lNewLength > lCurrLength )
				{
					long i = lCurrLength;
					for ( char* pcData=(char*)RAW_POINTER(ps_new)+i*nElemSize; i<lNewLength; pcData+=nElemSize, i++)
					{
						if ( nCurrMode == S_MODE_CHAR )
							*(char**)pcData = EMPTY_STRING;				 
						else if ( nCurrMode == S_MODE_LIST )
							*(s_object**)pcData = blt_in_NULL;
						else
							set_na(pcData, nCurrMode, S_evaluator);
					}
				}
			}
		}
		if ( !SPL_NotThere(ps_new) )
		{
			ReAttachAndAssign(ps_new);
			bSuccess = TRUE;
		}
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	return bSuccess;
}

s_object * SPL_GenerateVector( int nMode, long nLength )
{
	s_object *pVector = NULL;
	try
	{
		switch ( nMode )
		{
			case S_MODE_LIST:
			case S_MODE_DOUBLE:
			case S_MODE_INT:
			case S_MODE_LGL:
			case S_MODE_REAL:
			case S_MODE_COMPLEX:
			case S_MODE_CHAR:
			case S_MODE_RAW:
				pVector = alcvec(nMode, nLength, S_evaluator);
			break;
		}
	}
	catch(...)
	{
	}
	return pVector;
}

s_object * SPL_GenerateVectorFromDataArray( int nMode, long nLength, void *vpDataArray )
{
	if ( nMode != S_MODE_LGL && nMode != S_MODE_INT && nMode != S_MODE_REAL && 
			nMode != S_MODE_DOUBLE && nMode != S_MODE_CHAR && nMode != S_MODE_RAW )
		SCONNECT_ThrowException("Invalid mode %s to copy external data to an S object", token_name(nMode, S_evaluator));

	if ( !vpDataArray && nLength != 0 )
		nLength = 0;

	s_object *ps_new = blt_in_NULL;

	int nElemSize = mode_size(nMode);
	long nBytes = nElemSize * nLength;
	if ( nMode == S_MODE_CHAR )
	{
		try
		{
			CSPevaluator openSesame;
			CSPcharacter s_new((char**)vpDataArray, nLength);
			ps_new = openSesame.CloneIfNeeded(s_new.Detach());
		}
		catch(...)
		{
		}
	}
	else
	{
		ps_new = alcvec(nMode, nLength, S_evaluator);
		memcpy((void*)(ps_new->value.name), vpDataArray, nBytes);
	}

	return ps_new;
}

int CSPvector::GetElementSize()
{
	int nMode = S_MODE_NULL;
	int nElementSize = 0;
	try
	{
		nMode = GetMode();
		nElementSize = mode_size(nMode);
	}
	catch(...)
	{
	}
	return nElementSize;
}

BOOL CSPvector::InsertRows(long lRow, long nRows, BOOL bValidate)
{
	if( bValidate && !IsValid() )
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
	if( lRow < 0 )
		SCONNECT_ThrowException("Invalid row index %d", lRow);
	if ( nRows <= 0 )
		SCONNECT_ThrowException("Invalid number of rows to be inserted %d", nRows);

	BOOL bSuccess = FALSE;

	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator s;

		s_object *ps_new = SPL_ExtendVector(Clone(), nRows, lRow);
		ReAttachAndAssign(ps_new);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}

	return bSuccess;
}

BOOL CSPvector::RemoveRows(long lRow, long nRows, BOOL bValidate)
{
	if( bValidate && !IsValid() )
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
	if( lRow < 0 )
		SCONNECT_ThrowException("Invalid row index %d", lRow);
	if ( nRows <= 0 )
		SCONNECT_ThrowException("Invalid number of rows %d to be inserted", nRows);

	BOOL bSuccess = FALSE;

	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator s;

		s_object* ps_new = SPL_ShrinkVector(GetPtr(), nRows, lRow);
		ReAttachAndAssign(ps_new);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}

	return bSuccess;
}
