newPackage(
    "BooleanGB",
    Version => "1.0", 
    Date => "September 27, 2010",
    Authors => {{Name => "Franziska Hinkelmann", 
    Email => "fhinkel@vt.edu", 
    HomePage => "http://www.math.vt.edu/people/fhinkel/"}, 
    {Name => "Mike Stillman"},
    {Name => "Elizabeth Arnold"}},
    Headline => "Groebner bases in a Boolean ring",
    DebuggingMode => true
    )

exportFrom_Core {"gbBoolean"}

-- Code here


beginDocumentation()

doc ///
Key
  BooleanGB
Headline
  Compute Groebner Basis in a Boolean Ring
Description
  Text
    gbBoolean is a fast Groebner Basis computation done bitwise instead of
    symbolically when working over the quotient ring F2/J where J is the ideal
    generated by X^2 - X .
  Example
    n = 3;
    R = ZZ/2[vars(0)..vars(n-1)];
    J = apply( gens R, x -> x^2 + x); 
    QR = R/J;
    I = ideal(a+b,b);
    gbBoolean I
    gens gb I
Caveat
  gbBoolean assumes the quotient ring, regardless of the ring in which the
  ideal was generated. 

  R = ZZ/2[x]

  gbBoolean ideal(x^3)
SeeAlso
///

doc ///
Key
  (gbBoolean, Ideal)
  gbBoolean
Headline
  Compute Groebner Basis 
Usage
  gbBoolean I
Inputs
  I:Ideal
Outputs
  J:Ideal
    the reduced Groebner basis of I
Consequences
Description
  Text
    gbBoolean is a fast Groebner Basis computation done bitwise instead of
    symbolically when working over the quotient ring F2/J where J is the ideal
    generated by X^2 - X .
  Example
    n = 3
    R = ZZ/2[vars(0)..vars(n-1)]
    J = apply( gens R, x -> x^2 + x) 
    QR = R/J
    I = ideal(a+b,b)
    gbBoolean I
    gens gb I
Caveat
  gbBoolean assumes the quotient ring, regardless of the ring in which the
  ideal was generated. 
  R = ZZ/2[x]
  gbBoolean ideal(x^3)
SeeAlso
  gb
///


-- These tests check the generators for equivalence

--TEST ///
--R = ZZ/2[a..t, MonomialOrder=>GRevLex]
--l = apply(gens R, x-> x^2+x);
--QR = R/l;
--I = ideal {a*b*c*d*e,
--  a+b*c+d*e+a+b+c+d , 
--  j*h+i+f +a*b+c*d+e*f+g*h*i+i*j+a*t+s*r, 
--  g+f +m*n+o + o*p + r*s*t+a*l+h*i*q*s + k*c,
--  j+i+d*c, 
--  r+s+t*a*b*c*d*f*g*h+i*o*p*q*r*s*t, 
--  m*n+o*p, 
--  b*s+q+p*n*m+i + i*j*h*a*c*t, 
--  b*s+q*n*m+i+b*l*o*r*s*c, 
--  b*k+q+l*n*m +n,
--  i*q*a*c*e+f*g+o*q*d*m +b+d, 
--  b*s+q*n*m+i+j*s*t+s, 
--  b*k+r*q+l*m+i*j+n, 
--  b*k+d*n*m+i, 
--  b+q+l*n*m+i*d, 
--  a*k+c*l*n*f, 
--  q*r+c+q+l*n*m+i
--};
--C = gens gb I;
--R = ZZ/2[a..t, MonomialOrder=>Lex]
--l = apply(gens R, x-> x^2+x);
--QR = R/l;
--C = sub(C, QR)
--C = gb C;
--I = sub(I,QR)
--B = gbBoolean I;
--assert( sort gens B - sort gens C == 0 )
--///

TEST ///
R = ZZ/2[vars(0..14), MonomialOrder=>Lex]
l = apply(gens R, x-> x^2+x);
QR = R/l;
I = ideal(b*k+a+o+1,a*k+b,a*c*i+c*d*i+a*i*o+c*d+1,h*i*j*l+c*h*j+i*l+d+l,b*c*d*f*n+c*d*f+b*d*n+b*c+b*f+d*f+b*n+b+c+d+e,f,a*b*g*j*n+b*g*n+a*b+b*g+b+g,e*i*m*o+e*h*i+e*i*o,d*f*g+c*f*o+f+i,f*g*j+h*m+h+j,b*d*i+d*f*j+f*i*j+k,e*o+o,d*i*k+d*i+m,d*e*k*o+d*e*k+d*g*o+e*g*o+e*k*o+d*e+e*g+g*o+n+1,a*d*e+a*e*j+a*d*m+a*e*m+d*j*m+a*m+j+o+1)
C = gb I;
B = gbBoolean I;
assert( sort gens B - sort gens C == 0 ) 
///

TEST ///
R = ZZ/2[a..t, MonomialOrder=>Lex]
l = apply(gens R, x-> x^2+x);
QR = R/l;
I = ideal {
  b*c+1,
  a*b*c*d*f*g*h*t + i*o*p*q*r*s*t + r + s, 
  a*c*e*i*q + d*m*o*q + f*g
};
C = gb I;
B = gbBoolean I;
assert( sort gens B - sort gens C == 0 ) 
///

TEST ///
R=ZZ/2[vars(0..31), MonomialOrder=>Lex]
l =  apply( gens R, x -> x^2+x);
RQ = R/l
I = ideal(a);
C = gb I;
B = gbBoolean I;
assert( sort gens B - sort gens C == 0 )
///

TEST ///
R = ZZ/2[a..t, MonomialOrder=>Lex]
l = apply(gens R, x-> x^2+x);
QR = R/l;
I = ideal {
  b*c+1,
  a*b*c*d*f*g*h*t + i*o*p*q*r*s*t + r + s, 
  b*c*l*o*r*s + b*s + i + m*n*q, 
  a*c*e*i*q + d*m*o*q + f*g, 
  i + l*m*n + q*r + q +1
};
C = gb I;
B = gbBoolean I;
assert( sort gens B - sort gens C == 0 ) 
///

TEST ///
R = ZZ/2[a..t, MonomialOrder=>Lex]
l = apply(gens R, x-> x^2+x);
QR = R/l;
I = ideal {
  b*c,
  a*b*c*d*f*g*h*t + i*o*p*q*r*s*t + r + s, 
  b*c*l*o*r*s + b*s + i + m*n*q, 
  a*c*e*i*q + d*m*o*q + f*g, 
  i + l*m*n + q*r + q
};
C = gb I;
B = gbBoolean I;
assert( sort gens B - sort gens C == 0 ) 
///

TEST ///
R=ZZ/2[vars(0..31), MonomialOrder=>Lex]
l =  apply( gens R, x -> x^2+x);
RQ = R/l
I = ideal(a,b, a*c+d);
C = gb I;
B = gbBoolean I;
assert( sort gens B - sort gens C == 0 )
///

TEST ///
  R = ZZ/2[x,y,z, MonomialOrder=>Lex]
  QR = R / ideal apply(gens R, x -> x^2 + x)
  I = ideal(x+y,x)
  correct = sort flatten entries gens gb I
  G = sort flatten entries gens gbBoolean I
  assert(correct === G ) 
///

TEST ///
  R = ZZ/2[x,y,z, MonomialOrder=>Lex]
  QR = R / ideal apply(gens R, x -> x^2 + x)
  I = ideal(x*y+z)
  correct = sort flatten entries gens gb I
  G = sort flatten entries gens gbBoolean I
  assert(correct === G ) 
///

--TEST ///
--R = ZZ/2[vars(0..17), MonomialOrder=>Lex]
--l = apply(gens R, x-> x^2+x);
--QR = R/l;
--I = ideal(0,0,0,0,b*f*i*k+h*i*j+f*j*k+h*j*k+b*k*l+f*i+f*j+a,d*g*m*o+h*m*o+h*o*r+g*h+d*o+i*r+m*r+b,a*d*j*l+c,b*c*k*l*q+b*c*g*l+c*k*l*n+c*l*n*q+k*l*n*q+b*g*l+g*k*l+b*c*q+l*n*q+b*n+d,b*h*j*l+d*h*l*p+b*d*h+b*j*l+j*l*p+e*p+e,i*m*p*r+j*m*p*r+f,c*f*g*i*n+g,c*f*j*l*o*r+j*q*r+h,a*f*h*p+h*m*o*r+a*m*p+a*o*p+h*o*p+f*m*r+h*m*r+o*r+i,0,e*f*g*h*m*q+e*f*g*h+b*f*g+e*m*q+f*m*q+k,a*e*f*r+e*f*g+c*g*r+c*p+g*p+f*r+e+l,d*e*h*i*k*m+d*e*f*h*m+d*e*i*m+e*i*m+d*k*m+m,a*c*e*j*n*q+c*e*m*n+n,a*f*n*o+o,a*j*k*l*n+p,b*c*e*n*q+q,d*e*i+r);
--C = gb(I, Algorithm=>Sugarless);
--B = gbBoolean I;
--assert( sort gens B - sort gens C == 0 ) 
--///


 TEST ///
   R = ZZ/2[ vars(1..20), MonomialOrder=>Lex]
   QR = R / ideal apply(gens R, x -> x^2 + x)
   II3 = ideal (c*k*r + 1, b*d*h*i*n + b*h*i*n + b*d*h + b*d*i + b*i*n + d*n + b, g*h*l*o*r + g*o, j*l*m + d*m*t + l*m*t + l*t, e*k*t*u + g*k*t*u + e*g*k + e*g*u + g*k + u, m*n*q*r + k*n + n*q + m*r + 1, b*e*g*o + e*g*o*s + b*g*o + e*g*o + b*o*s + e*o*s + e*g, e*g*k*q + g*k*q*t + g*k*q + g*t + k*t, j*m*t*u + f*j*t, o*q*t*u + o*t*u, p*s*u + q*r + r*s + q + u, b*s, b*f*n*s + f*n*s + n*s*t + f*n + f, d*p + d*t, g*l*q*t + q*t, c*d*e*p*q, d*q*r*t + o*q*r + d*q + o*r + r*t + o, d*h*m*n*p + h*m*n*p, f*k*o*s*t + f*k*o*s + f*o*s*t + k*o*s*t + f*k*o + f*k*s + f*k*t + f*k + o*t + f, k*q*t + h*q + h + 1)
   correctSolution = sort flatten entries gens gb( II3, Algorithm=>Sugarless)
   G = sort flatten entries gens gbBoolean II3
   assert( G == correctSolution )
 ///


 TEST ///
   R = ZZ/2[ vars(1..20)]
   QR = R / ideal apply(gens R, x -> x^2 + x)
   II5 = ideal(l,c*g*h*k*q+c*h*k*q+c*k*q+h*k*q+g*k+h*q+k+1,b*f*k*q*t+b*f*k*t+b*t+q,c*d*j*k+c*d*j*t+c*d*k*t+d*j*k+j*k*t+d*j+j*k+j,e*j*p*q*u+e*j*q*u+e*j+e*q+p*q+q,c*k*m*s*u+c*k*m*u+c*k*m+k*m*s+k*m*u+k*u+m*u+m,b*f*g*r+e*f*g*r+e*f*r+f*g+e*r,f*k*l*u+i*k*l*u+f*l*u+k*l*u+i*k+i*l+k*l+k,f*l*o+f*n+l*o+n*o+l*u,d*e*g*n+d*n+g,d*j*m*o+d*e*o+d*o+m*o+1,f*g*h*i+f*g*h+f*g*i+f*g*q+h*i+1,d*p*r*s+d*p*s+f*r+f*s+f,b*j*k*q*r+b*j*k*q+j*k*q*r+b*q,d*f*g*n+d*f*p+f*n*p+g*n*p+d*g+g*p+1,k*p*q,h*l*o+h*n*r+h*o*r+l*o*r+n*o*r+l+o,f*p*u+c*f+p*u+1,d*h+d,b*g*h+h)
   correct = sort flatten entries gens gb II5      -- used 0.0001 seconds
   G = sort flatten entries gens gbBoolean II5
   assert( correct == G ) 
 ///

 TEST ///
   R = ZZ/2[ vars(1..20), MonomialOrder=>Lex]
   QR = R / ideal apply(gens R, x -> x^2 + x)
   II6 = ideal(b*c*e*j+b*c*j*n+b*e*j*n+c*e+c*n+c,g*k+g,d*e*f*o+d*f*o*r+d*f*o+e*f*o+d*e*r+e+1,f*s+n*s,d*e*j*o+d*e*j*q+d*j*o*q+e*j*o+d*j*q+e*o+d+o,f*i*n+f*n,f*j*l*p+f*j+j*l,e*k*n*s+e*g*s+e*n*s+g*s+g,c*p*s*t+c*j*t+s,c*k+f,b*e*f+b*e*o+b*o*t+e*o*t+b*o+f*o,b*g+f*q+q,i*m+b*t+k,e*i*l+e*i*m+h*i+h*m+e+1,r*t+1,d*m,d*f*p+e*p*q+f*p*q+d*f+d*p,e*i*m+e*i*p+i*m*p+e*m+f*p+f+i+p,e*g*h*i*u+g*h*i*u+g*h+h*i,c*q+i*q)
   G = sort flatten entries gens gbBoolean II6
   correctSolution = sort flatten entries gens gb (II6, Algorithm=>Sugarless)
   assert(G == correctSolution )
 ///

 TEST ///
   R = ZZ/2[ vars(1..20)]
   QR = R / ideal apply(gens R, x -> x^2 + x)
   II4 = ideal (d*h*j + h*j*o + d*k*o + j*k*o + d*k + d + h, e*f*o*q + e*g*o + f*g*o + e*g + f*q + g*q + g + 1, h*l*n + j*n*r, q*u + f + q + 1, b*j + h*n, l*m*o + l*q, f*h*o*q + f*h*o + f*o*t + h*q*t + h + 1, g*h*p*r + g*h*r + g*p*r + h*p*r + g*r + h*r + p*r + h + m, d*k + d*r + f*r + k, b*j*o*p*s + b*j*p*s + b*j*p + b*o + b + p + 1, g*r, e*j*r*s + o*r*s + o*r + r*s + e + j + s, m*u + n*u, i*j*p*q + h*i + h*p + j + q, e*l*t*u + d*e*l + e*l*t + d*l*u + e*l*u + d, e*l*m*r + e*l*m*s + l*m*r + l*r*s + m*r*s + l*m, j*m*q*r*t + j*m*q + j*m*t, b*d*r*u + d*p*r*u + d*p*u + b*p + d*p + b*r + p, c*e*m*s, d*e*q*u + e*u + q*u + q)
   GG = sort flatten entries gens gbBoolean II4
   assert( GG == {1})
 ///

--------- here is the big end ----------
---------------------------------------
end

TEST ///
R = ZZ/2[ vars(1..36), MonomialOrder=>Lex]
QR = R / ideal apply(gens R, x -> x^2 + x)

correctLexBasis = sort {G*J, G*I, G*H, F*I+F, F*H*K+H*I*K, F*H*J+F*H+H*I*J+H*I, F*G, E*F*H+E*H*I, D*F*H+D*H*I, D*E*J*K, D*E*I, D*E*G, D*E*F, C*I*J+C*I+C*J+C+E*I*J+E*I+E*J+E+I*J+I+J+1, C*G+E*G+G, C*F*J+C*F+C*J+C+E*F*J+E*F+E*J+E+F*J+F+J+1, C*F*H+C*H*I, C*E*J+C*E+E*F*J+E*F, C*D*E*K, B*I*J*K+F*J*K+I*J*K, B*H*J, B*G*K, B*F+B*I*J+B*I+F*J+F+I*J+I, B*E*I*J+E*F*J+E*I*J, B*E*H*I*K+B*E*H*K+B*H*I*K+B*H*K, B*E*G, B*D*I*J+D*F*J+D*I*J, B*D*H*I*K+B*D*H*K+D*E*H*K, B*D*E*K+D*E*K, B*D*E*J+B*D*E+D*E*J+D*E, B*D*E*H+D*E*H*J+D*E*H, B*C*J+B*C+B*E*I+B*E*J+B*E+B*J+B+E*F+E*I, B*C*I+B*I*J+B*I+C*F+C*I+F*J+F+I*J+I, B*C*H+B*E*H*I+B*E*H+B*H, A*I*J*K, A*H*J*K, A*H*I*J+F*H+H*I, A*G, A*F*J, A*F*H+A*H*I+F*H+H*I, A*E*J*K+A*J*K, A*E*I*K+A*F*K, A*E*I*J, A*E*H*K+A*H*K, A*E*F+A*E*I, A*D*J*K, A*D*I*K, A*D*F*K, A*D*E*J+B*D*E+D*E, A*C*J+A*C+A*E*I+A*E*J+A*E+A*J+A, A*C*I*K+A*I*K, A*C*H*K+A*H*I*K, A*C*H*I+A*H*I+F*H+H*I, A*C*F+A*F, A*C*E*K+A*C*K+A*E*K+A*K, A*C*E*I+A*E*I, A*C*D*K+A*D*E*K+A*D*K, A*C*D*E+B*C*D*E+C*D*E, A*B*K+A*F*K+A*K, A*B*I+A*F+A*I+B*I*J+F*J+I*J, A*B*H+A*E*H*I+A*E*H*J+A*E*H, A*B*E*J+A*B*E+A*E*I+A*E*J+A*E, A*B*D*E+A*D*E+B*D*E+D*E, z*I*K+F*K, z*I*J+z*I+F*J+F, z*G+G, z*F+F, z*E*I+E*F, z*D*I+D*F, z*D*E*K, z*D*E*J+z*D*E, z*C*J+z*C+z*E*J+z*E+z*J+z+C*J+C+E*J+E+J+1, z*C*I+C*F, z*B*J*K, z*B*I+B*I*J+B*I+F*J+F+I*J+I, z*B*H+B*E*H*I+B*E*H+B*H, z*B*E*K+B*E*I*K+E*F*K+E*I*K, z*B*E*J+z*B*E+B*E*I+E*F+E*I, z*B*C+z*B, z*A*J*K, z*A*I+z*I+A*F+F, z*A*H*K+A*H*I*K, z*A*E*K+A*F*K, z*A*E*J+z*A*E+A*E*I, z*A*D*E+z*B*D*E+z*D*E, z*A*C*K+z*A*K, z*A*B*J, z*A*B*E, y*I+I, y*F+F, y*C*J+y*C+y*E*J+y*E+y*J+y+C*J+C+E*J+E+J+1, y*B*E*H*K+y*B*H*K+B*E*H*K+B*H*K, y*B*D*H*K+y*D*E*H*K+B*D*H*K+D*E*H*K, x*J*K+E*J*K+J*K, x*H*I*K+A*H*I*K+B*E*H*K+B*H*I*K+B*H*K+E*H*I*J*K+H*I*K, x*H*I*J+x*H*I+A*E*H*I+B*E*H*I+H*I*J+H*I, x*G+G, x*F*J+E*F*J+F*J, x*F*H+A*E*H*I+B*E*H*I+E*H*I*J+F*H, x*E*I*K+x*F*K+E*F*J*K+E*I*J*K+E*I*K+F*K, x*E*I*J+x*E*I+x*F+E*F*J+E*I*J+E*I+F, x*E*H*I+A*E*H*I+B*E*H*I+E*H*I*J+E*H*I, x*E*F+x*F+E*F+F, x*D*I+D*I, x*D*F+D*F, x*C*K+x*E*K+x*F*K+x*K+C*E*K+C*K+E*F*K+E*K+F*K+K, x*C*J+x*C+x*E*J+x*E+x*F+x*J+x+C*J+C+E*F*J+E*J+E+F+J+1, x*C*H*I+A*E*H*I+B*E*H*I+C*E*H*I+C*H*I+E*H*I*J+E*H*I, x*C*F+x*F+C*E*F+C*F+E*F+F, x*B*I*K+x*I*K+A*F*K+B*I*K+E*F*J*K+I*K, x*B*I*J+x*B*I+x*I*J+x*I+A*E*I+B*I*J+B*I+I*J+I, x*B*H*K+x*E*H*K+A*H*I*K+B*H*K+E*H*J*K+E*H*K, x*B*H*I+B*E*H*I+B*H*I, x*B*E*K+x*E*K+A*F*K+B*E*J*K+B*E*K+E*J*K+E*K, x*B*E*J+x*B*E+x*E*J+x*E+A*E*I+B*E*J+B*E+E*J+E, x*B*E*I+x*E*I+A*E*I+B*E*I+E*F*J+E*I, x*B*E*H+x*E*H*J+x*E*H+A*E*H*I+B*E*H+E*H*J+E*H, x*A*K, x*A*J+x*B*J+x*J+A*E*J+A*J+B*E*J+B*J+E*J+J, x*A*H*I+x*H*I+A*H*I+B*E*H*I+E*H*I*J+H*I, x*A*F+A*E*I+A*F, x*A*E*I, x*A*C+x*A*E+x*A+x*B*C+x*B*E+x*B+x*C+x*E+x+A*C*E+A*C+A*E*I+A*E+A+B*C*E+B*C+B*E*I+B*E+B+C*E+C+E*F*J+E*I+E+1, x*A*B*E+x*A*E+x*B*E+x*E+A*E*I+B*E*J+B*E+E*J+E, x*z*E*K+x*F*K+z*E*J*K+z*E*K+E*F*J*K+F*K, x*z*E*J+x*z*E+x*F+z*E*J+z*E+E*F*J+F, x*z*A*E+x*z*B*E+x*z*E+z*E*J+z*E+A*E*I+B*E*I+E*F+E*I, v*F+v*I, v*B*I*J, v*B*H*I, v*B*E*H*K+v*B*H*K, v*B*D*H*K+v*D*E*H*K, v*A*I*K+A*F*K, v*A*I*J, v*A*E*I+A*E*I, v*A*C*I+v*A*I, v*z*I+v*I, v*x*I+v*B*E*I+v*E*I*J+v*I+x*F+B*E*I+E*F*J+E*F+E*I+F, u, t*J*K+J*K, t*H*I*J+F*H+H*I*J+H*I, t*F*J+F*J, t*F*H+t*H*I, t*D*I*J+D*I*J, t*D*E*K+D*E*K, t*D*E*J+t*D*E+D*E*J+D*E, t*C*I*K+t*C*K+t*E*I*K+t*E*K+t*I*K+t*K+C*I*K+C*K+E*I*K+E*K+I*K+K, t*C*H+t*E*H*J+t*E*H+t*H*I+t*H+A*C*H+A*E*H*J+A*E*H+A*H*I+A*H+B*E*H*I+B*H*I+C*H*J+H*I*J+H*J, t*C*F+t*C*J+t*C+t*E*F+t*E*J+t*E+t*F+t*J+t+C*F+C*J+C+E*F+E*J+E+F+J+1, t*C*E*K+t*E*F*K+C*E*K+E*F*K, t*C*D*I+t*C*D*J+t*C*D+t*D*I+t*D*J+t*D+C*D*I+C*D*J+C*D+D*I+D*J+D, t*B*I+t*C*J+t*C+t*E*J+t*E+t*I+t*J+t+x*F+A*F+E*F*J+F*J+F, t*B*G+B*G, t*B*E*H+t*B*H+B*E*H+B*H, t*B*D*H+B*D*H, t*B*C+t*B*E*J+t*B*E+t*B+t*C*J+t*J+x*F+A*B*C+A*B+A*C+A*E*I+A+B*C+B*E*I+B*E*J+B*E+B+C*J+E*F*J+E*F+E*I+F+J, t*A+t*B*J+t*J+A+B*J+J, t*z*I+t*F, t*z*C+t*z*E*J+t*z*E+t*z+t*C*J+t*C+t*E*J+t*E+t*J+t+z*A*C+z*A+z*C+z*E*J+z*E+z+C*J+C+E*J+E+J+1, t*z*B*K+t*C*K+t*E*K+t*F*K+t*K+x*F*K+z*B*K+A*F*K+B*I*K+C*J*K+E*F*J*K+E*J*K+F*J*K+I*K+J*K, t*z*B*J+t*z*B+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*F+z*B*J+z*B+A*F+B*I*J+B*I+E*F*J+I*J+I, t*y*C*K+t*y*E*K+t*y*K+t*C*K+t*E*K+t*K+y*C*K+y*E*K+y*K+C*K+E*K+K, t*x*I*J+t*E*I*J+t*I*J, t*x*H*I+t*H*I+A*E*H*I+B*E*H*I+E*H*I*J, t*x*F+t*F+x*F+F, t*x*E*K+t*E*K+x*E*K+E*K, t*x*E*J+t*x*E+t*E*J+t*E+x*E*J+x*E+E*J+E, t*x*E*I+t*E*I*J+t*E*I+x*F+E*F*J+F, t*x*C*I+t*x*I+t*C*E*I+t*C*I+t*E*F+t*E*I*J+t*I+E*F*J, t*x*C*E+t*x*C+t*x*E+t*x+t*C*E+t*C+t*E+t, t*x*B*H+t*B*H+x*B*H+B*H, t*v*I*J+v*I*J, t*v*C*K+t*v*E*K+t*v*I*K+t*v*K+t*C*K+t*E*K+t*F*K+t*K+v*B*E*I*K+v*B*I*K+v*C*J*K+v*E*J*K+v*I*J*K+v*J*K+B*E*I*K+B*I*K+C*J*K+E*F*K+E*I*K+E*J*K+F*J*K+F*K+I*K+J*K, t*v*C*J+t*v*C+t*v*E*J+t*v*E+t*v*I+t*v*J+t*v+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+v*A*I+v*B*E*I+v*B*I+v*I*J+A*F+B*E*I+B*I*J+B*I+E*F+E*I+F+I*J+I, t*v*C*I+t*v*E*I+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+v*A*I+v*B*E*I+v*B*I+v*C*I+v*C*J+v*C+v*E*I+v*E*J+v*E+v*I*J+v*I+v*J+v+A*F+B*E*I+B*I*J+B*I+E*F+E*I+F+I*J+I, s*I+v*B*I+F+I, s*H*K, s*H*J+s*H, s*G, s*F+v*B*I, s*E*J*K+s*E*K+s*J*K+s*K+v*B*E*I*K+v*B*I*K+B*E*I*K+B*E*J*K+B*E*K+B*I*K+B*J*K+B*K+E*F*K+E*I*K+F*K+I*K, s*D*E*K+s*D*J*K+s*D*K+v*B*D*I*K+B*D*I*K+B*D*J*K+B*D*K+D*E*K+D*F*K+D*I*K, s*C+s*E*J+s*E+s*J+s+v*B*E*I+B*C+B*E*I+B*E*J+B*E+B*J+B+E*F+E*I, s*B*K+s*K, s*B*J+s*B+s*J+s, s*B*H, s*A*H+s*H, s*A*E*K+s*A*K+A*E*K+A*K, s*A*E*J+s*A*E+s*A*J+s*A+A*B*J+A*B+A*E*I+A*E*J+A*E, s*A*D*E+s*A*D*J+s*A*D+s*B*D*E+s*D*E+A*B*D*J+A*B*D+A*D*E+B*D*E+D*E, s*A*B+s*A+s*B+s, s*z*K+v*B*I*K+z*B*K+B*I*K+F*K+I*K, s*z*J+s*z+s*E*J+s*E+s*J+s+v*B*E*I+z*B*J+z*B+B*E*I+B*E*J+B*E+B*J+B+E*F+E*I, s*z*B+s*E*J+s*E+s*J+s+v*B*E*I+z*B+B*E*I+B*E*J+B*E+B*J+B+E*F+E*I, s*z*A+s*z+s*E*J+s*E+s*J+s+v*B*E*I+z*A*B+z*B+B*E*I+B*E*J+B*E+B*J+B+E*F+E*I, s*y*B*D*E+s*y*D*E*J+s*y*D*E+s*B*D*E+s*D*E*J+s*D*E+y*B*C*D*E+B*C*D*E, s*x*E*K+s*x*K+s*E*K+s*K+x*B*K+x*E*K+A*F*K+B*E*J*K+B*K+E*J*K+E*K, s*x*A*E+s*x*A+s*x*B*E+s*x*B+s*x*E+s*x+s*A*E+s*A+s*B*E+s*B+s*E+s+x*A*B+x*A*E+x*B*E+x*E+A*B*E+A*B+A*E*I+B*E*J+B*E+E*J+E, s*x*y*B+s*x*y*J+s*x*y+s*x*B+s*x*J+s*x+s*y*B*E+s*y*B+s*y*E*J+s*y*E+s*y*J+s*y+s*B*E+s*B+s*E*J+s*E+s*J+s+x*y*B*C+x*y*B*J+x*y*B+x*y*E*J+x*y*E+x*B*C+x*B*J+x*B+x*E*J+x*E+y*B*C*E+y*B*C+y*B*J+y*B+y*E*J+y*E+B*C*E+B*C+B*J+B+E*J+E, s*t*H, s*t*E*K+s*t*K+s*E*K+s*K+t*B*E*K+t*B*K+B*E*K+B*K, s*t*E*J+s*t*E+s*t*J+s*t+s*E*J+s*E+s*J+s+t*B*E*J+t*B*E+t*B*J+t*B+B*E*J+B*E+B*J+B, s*t*D*K+s*D*K+t*B*D*K+B*D*K, s*t*D*J+s*t*D+s*D*J+s*D+t*B*D*J+t*B*D+B*D*J+B*D, s*t*D*E+s*D*E*J+s*D*E+t*B*D*E+t*C*D*E+t*D*E, s*t*B+s*t, s*t*z+s*E*J+s*E+s*J+s+t*z*B+v*B*E*I+B*E*I+B*E*J+B*E+B*J+B+E*F+E*I, s*t*y*J+s*t*J+s*y*B+s*y*J+s*y+s*B+s*J+s+t*y*B*J+t*y*C+t*y*E*J+t*y*E+t*y+t*B*J+t*C+t*E*J+t*E+t+y*A*B*C+y*A*B+y*A*C+y*A+y*B*J+y*C+y*E*J+y*E+y+A*B*C+A*B+A*C+A+B*J+C+E*J+E+1, s*t*y*E+s*t*y+s*t*E+s*t+s*y*B*E+s*y*B+s*B*E+s*B+t*y*B*E+t*y*B+t*y*C*E+t*y*C+t*y*E+t*y+t*B*E+t*B+t*C*E+t*C+t*E+t+y*A*B*C*E+y*A*B*C+y*A*B*E+y*A*B+y*A*C*E+y*A*C+y*A*E+y*A+y*B*E+y*B+y*C*E+y*C+y*E+y+A*B*C*E+A*B*C+A*B*E+A*B+A*C*E+A*C+A*E+A+B*E+B+C*E+C+E+1, s*t*y*D+s*t*D+s*y*B*D+s*B*D+t*y*B*D+t*y*C*D+t*y*D+t*B*D+t*C*D+t*D+y*A*B*C*D+y*A*B*D+y*A*C*D+y*A*D+y*B*D+y*C*D+y*D+A*B*C*D+A*B*D+A*C*D+A*D+B*D+C*D+D, s*t*x+s*t*E+s*t*J+s*x*J+s*x+s*E*J+s*E+t*x*B+t*x*C+t*x+t*B*E+t*B*J+t*C*E+t*C+t*E*F+t*E*J+t+x*B*J+x*B+x*E*J+x*E+x*F+B*E*J+B*E+E*J+E+F, r*H*J*K, r*G+G, r*F*J, r*F*H+r*H*I*J+r*H*I, r*E*H*I*J, r*D*H*I*J, r*C*H*J, r*C*H*I+r*C*H+r*E*H*I+r*E*H*J+r*E*H+r*H*I*J+r*H*I+r*H*J+r*H, r*C*F+r*C*J+r*C+r*E*F+r*E*J+r*E+r*F+r*J+r, r*C*E*H+r*E*H*I, r*C*D*E, r*B*J*K+r*J*K, r*B*I*K+r*F*K+r*I*K+v*B*I*K+B*I*K+F*K+I*K, r*B*I*J+r*B*I+r*F+r*I*J+r*I+v*B*I+B*I*J+B*I+F*J+F+I*J+I, r*B*H*I+B*H*I, r*B*E*I+r*E*F+r*E*I+v*B*E*I+B*E*I+E*F+E*I, r*B*E*H+r*B*H+B*E*H+B*H, r*B*D*I+r*D*F+r*D*I+v*B*D*I+B*D*I+D*F+D*I, r*B*D*H+r*D*E*H*J+r*D*E*H+B*D*H+D*E*H*J+D*E*H, r*B*D*E+r*D*E*J+r*D*E+s*B*D*E+s*D*E*J+s*D*E+B*C*D*E, r*B*C+r*B*E*J+r*B*E+r*B*J+r*B+r*C*J+v*B*E*I+B*E*I+E*F+E*I, r*A*H*J+r*H*J, r*A*H*I*K, r*A*F*K, r*A*E*I, r*A*C*H+r*A*E*H+r*A*H+r*E*H*J+r*H*J, r*A*B*J+r*A*J+r*B*J+r*J, r*A*B*E+r*A*E+r*B*E*J+r*E*J, r*z*J*K, r*z*C+r*z*E*J+r*z*E+r*z*J+r*z+r*C*J+r*C+r*E*J+r*E+r*J+r, r*z*B+s*E*J+s*E+s*J+s+v*B*E*I+B*E*I+B*E*J+B*E+B*J+B+E*F+E*I, r*z*A*K, r*z*A*J+r*z*J, r*z*A*E+r*z*E*J, r*y*J+r*J, r*y*C+r*y*E+r*y+r*C+r*E+r, r*y*z*D*E+r*z*D*E, r*x*F+r*F+v*B*E*I+B*E*I+E*F+E*I, r*x*E*I+r*E*I*J+r*E*I+v*B*E*I+x*C*E*I+x*E*I+B*E*I+E*F*J+E*I*J, r*x*C+r*x*E*J+r*x*E+r*x*J+r*x+r*C*E+r*C+r*E*F+r*E*J+r*E+r*J+r+v*B*E*I+B*E*I+E*F+E*I, r*x*B*J+r*B*E*J+r*B*J+s*x*B+s*x*J+s*x+s*B*E+s*B+s*E*J+s*E+s*J+s+x*B*C+x*B*J+x*B+x*E*J+x*E+x*F+B*C*E+B*C+B*E*I+B*J+B+E*F*J+E*F+E*I+E*J+E+F, r*x*B*I+r*x*I*J+r*x*I+r*B*I+r*E*I*J+r*I*J+r*I+x*B*I+x*C*I+A*E*I+B*I+C*E*I+C*I+E*F+E*I*J, r*x*B*H+r*x*E*H*J+r*x*E*H+r*B*H+r*E*H*J+r*E*H+x*B*H+x*E*H*J+x*E*H+A*E*H*I+B*H+E*H*J+E*H, r*x*B*E+r*x*E*J+r*x*E+r*B*E*J+r*B*E+r*E*J+r*E+s*x*B*E+s*x*E*J+s*x*E+x*B*C*E+x*F+A*E*I+B*E*I+E*F*J+E*F+E*I+F, r*x*y*z*E+r*x*z*E+r*y*z*E+r*z*E, r*v*I*J, r*v*C*I+r*v*C*J+r*v*C+r*v*E*I+r*v*E*J+r*v*E+r*v*I+r*v*J+r*v, r*v*B*I, r*v*A*I, r*t*J+r*B*J+s*t*J+s*B+s*J+s+t*B*J+t*C*J+t*J+A*B*C+A*B+A*C+A+B*J+C*J+J, r*t*D*E+r*D*E*J+r*D*E, r*t*C+r*t*E+r*t*F+r*t+r*A*F+r*B*E*J+r*B*J+r*C*J+s*t*E+s*t+s*B*E+s*B+t*B*E+t*B+t*C*E+t*C*J+t*E*F+t*E*J+t*J+v*B*E*I+v*B*I+A*B*C*E+A*B*C+A*B*E+A*B+A*C*E+A*C+A*E+A+B*E*I+B*E+B*I*J+B*I+B+C*E+C*J+E*I+E*J+F*J+F+I*J+I+J, r*t*x*E+r*t*E+r*x*E*J+r*x*E+r*B*E*J+r*E*J+r*E+s*t*E+s*t*J+s*t+s*B*E+s*J+s+t*B*E+t*B*J+t*B+t*C*E+t*E*F+t*E*J+A*B*C*E+A*B*E+A*C*E+A*E+B*E+B*J+B+C*E+E*F+E*J, r*s*J+r*J, r*s*H+r*H*J, r*s*E+r*s+r*B*E*J+r*B*E+r*B*J+r*B+r*E*J+r*J+s*E*J+s*E+s*J+s+B*E*J+B*E+B*J+B, r*s*D+r*B*D*J+r*B*D+r*D*E*J+r*D*E+r*D*J+s*D*E*J+s*D*E+s*D*J+s*D+B*D*J+B*D+D*E*J+D*E, r*s*B+r*s+r*B*J+r*J, r*s*A*K+r*A*E*K+r*A*J*K+r*A*K, r*s*z+r*z*J, r*s*x+r*s+r*x*B+r*x*E*J+r*x*E+r*x*J+r*B*E*J+r*B+r*E*J+r*E+r*J+s*x*B+s*x*E*J+s*x*E+s*B*E+s*B+x*B*C+x*F+A*E*I+B*C*E+B*C+B*E*I+E*F*J+E*F+E*I+F, p*F+p*I+F*J+I*J, p*B*I*J+B*I*J, p*A*I*J+A*I*J, p*A*C*I+p*A*I+A*C*I+A*I, p*z*I+p*I+z*I+F*J+F+I*J, p*y*J+p*J, p*y*C+p*y*E+p*y+p*C+p*E+p, p*y*B*D*E+p*y*D*E+p*B*D*E+p*D*E, p*y*A*B*E+p*y*A*E+p*A*B*E+p*A*E, p*y*z*D*E+p*z*D*E, p*y*z*B*K+p*z*B*K, p*y*z*B*E+p*z*B*E, p*y*z*A*K+p*z*A*K, p*y*z*A*E+p*z*A*E, p*y*z*A*B+p*z*A*B, p*x*I*J+p*E*I*J+p*I*J+x*I*J+E*I*J+I*J, p*x*H*I+p*A*E*H*I+p*B*E*H*I+p*E*H*I*J+p*H*I+x*H*I+A*E*H*I+B*E*H*I+E*H*I*J+H*I, p*x*E*I+p*x*I+p*E*I+p*I+x*E*I+x*F+x*I*J+E*F*J+E*I+F+I*J, p*x*C*I+p*x*I+p*C*E*I+p*C*I+p*E*I+p*I+x*C*I+x*I+C*E*I+C*I+E*F*J+E*F+E*I*J+I, p*x*B*I+p*x*I+p*A*E*I+p*B*I+p*E*I*J+p*I+x*B*I+x*I+A*E*I+B*I+E*I*J+I, p*x*A*I+p*A*E*I+p*A*I+x*B*I+x*I+A*E*I+B*I+E*F*J+I, p*x*y*B*E+p*x*y*E+p*x*B*E+p*x*E+p*y*B*E+p*y*E+p*B*E+p*E, p*x*y*z*E+p*x*z*E+p*y*z*E+p*z*E, p*v*I*J+p*I*J+v*I*J+I*J, p*v*C*I+p*v*C*J+p*v*C+p*v*E*I+p*v*E*J+p*v*E+p*v*I+p*v*J+p*v+p*C*I+p*C*J+p*C+p*E*I+p*E*J+p*E+p*I+p*J+p+v*C*I+v*C*J+v*C+v*E*I+v*E*J+v*E+v*I+v*J+v+C*I+C*J+C+E*I+E*J+E+I+J+1, p*t*I*J+p*I*J+t*I*J+I*J, p*t*C*I+p*t*C*J+p*t*C+p*t*E*I+p*t*E*J+p*t*E+p*t*I+p*t*J+p*t+p*C*I+p*C*J+p*C+p*E*I+p*E*J+p*E+p*I+p*J+p+t*C*I+t*C*J+t*C+t*E*I+t*E*J+t*E+t*I+t*J+t+C*I+C*J+C+E*I+E*J+E+I+J+1, p*t*y*D*E+p*t*D*E+p*y*D*E+p*D*E, p*t*y*z*B+p*t*z*B+p*y*z*B+p*z*B, p*t*x*I+p*t*I+p*x*I+p*I+t*E*I*J+x*I*J+I*J, p*t*x*y*E+p*t*x*E+p*t*y*E+p*t*E+p*x*y*E+p*x*E+p*y*E+p*E, p*s*B*D*E+p*s*D*E*J+p*s*D*E+p*B*C*D*E+s*B*D*E+s*D*E*J+s*D*E+B*C*D*E, p*s*y*H+p*s*H, p*s*y*E+p*s*y+p*s*E+p*s+p*y*z*B+p*y*B*E+p*y*B+p*z*B+p*B*E+p*B, p*s*y*D+p*s*D+p*y*z*B*D+p*y*B*D+p*y*D*E+p*z*B*D+p*B*D+p*D*E, p*s*y*B+p*s*y+p*s*B+p*s, p*s*y*A*K+p*s*A*K+p*y*A*E*K+p*y*A*K+p*A*E*K+p*A*K, p*s*y*z+p*s*z, p*s*x*B+p*s*x*J+p*s*x+p*s*B*E+p*s*B+p*s*E*J+p*s*E+p*s*J+p*s+p*x*B*C+p*x*B*J+p*x*B+p*x*E*J+p*x*E+p*x*I+p*B*C*E+p*B*C+p*B*E*I+p*B*J+p*B+p*E*I*J+p*E*J+p*E+p*I+s*x*B+s*x*J+s*x+s*B*E+s*B+s*E*J+s*E+s*J+s+x*B*C+x*B*J+x*B+x*E*J+x*E+x*F+x*I*J+B*C*E+B*C+B*E*I+B*J+B+E*F+E*I+E*J+E+F+I*J, p*s*x*y+p*s*x+p*s*y+p*s+p*x*y*z*B+p*x*y*B+p*x*y*E+p*x*z*B+p*x*B+p*x*E+p*y*z*B+p*y*B+p*y*E+p*z*B+p*B+p*E, p*s*t*J+p*s*B+p*s*J+p*s+p*t*B*J+p*t*C*J+p*t*J+p*A*B*C+p*A*B+p*A*C+p*A+p*B*J+p*C*J+p*J+s*t*J+s*B+s*J+s+t*B*J+t*C*J+t*J+A*B*C+A*B+A*C+A+B*J+C*J+J, p*s*t*E+p*s*t+p*s*B*E+p*s*B+p*t*B*E+p*t*B+p*t*C*E+p*t*C*J+p*t*E*I+p*t*E*J+p*t*J+p*A*B*C*E+p*A*B*C+p*A*B*E+p*A*B+p*A*C*E+p*A*C+p*A*E+p*A+p*B*E+p*B+p*C*E+p*C*J+p*E*I+p*E*J+p*J+s*t*E+s*t+s*B*E+s*B+t*B*E+t*B+t*C*E+t*C*J+t*E*F+t*E*I*J+t*E*J+t*J+A*B*C*E+A*B*C+A*B*E+A*B+A*C*E+A*C+A*E+A+B*E+B+C*E+C*J+E*F+E*I*J+E*J+J, p*s*t*D+p*s*B*D+p*t*B*D+p*t*C*D*J+p*t*D*J+p*A*B*C*D+p*A*B*D+p*A*C*D+p*A*D+p*B*D+p*C*D*J+p*D*J+s*t*D+s*B*D+t*B*D+t*C*D*J+t*D*J+A*B*C*D+A*B*D+A*C*D+A*D+B*D+C*D*J+D*J, p*r*J+r*J, p*r*C*I+p*r*C+p*r*E*I+p*r*E+p*r*I+p*r+r*C*I+r*C+r*E*I+r*E+r*I+r, p*r*C*E+p*r*E*I+r*C*E+r*E*F+r*E*I*J, p*r*B*I+p*v*B*I+p*B*I+r*B*I+r*F+r*I*J+r*I+v*B*I+B*I+F*J+F+I*J+I, p*r*A*I*K, p*r*A*C+p*r*A*E+p*r*A+r*A*C+r*A*E+r*A, p*r*z*D*E+r*z*D*E, p*r*x*I+p*r*I+p*v*B*E*I+p*B*E*I+r*x*I*J+r*I*J+E*F*J+E*I*J, p*r*x*z*E+p*r*z*E+p*v*B*E*I+p*B*E*I+r*x*z*E+r*z*E+v*B*E*I+B*E*I+E*F*J+E*F+E*I*J+E*I, o, n*J+J, n*I+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*F+E*F*J+F*J+F+I*J, n*G+B*G, n*F+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*F+E*F*J+F, n*D*E+D*E, n*C+n*E+n+x*F+C*J+E*F*J+E*J+F+J, n*B+n*E+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+z*B*J+z*B+A*F+B*I*J+B*I+B*J+E*J+F+I*J+I, n*A*K+z*A*K+A*E*K+A*F*K, n*A*E+n*A+z*A*B+A*B*J+A*B+A*E*I+A*E*J+A, n*A*D+z*A*B*D+A*B*D*J+A*B*D+A*D*E+A*D+B*D*E+D*E, n*z*E+x*F+z*E*J+E*F*J+F, n*z*A+z*A, n*x*E+n*E+x*E+E, n*x*A+n*A+x*z*A*B+x*A*B+x*A*E+x*A+x*B*E+x*E+z*A*B+A*B*E+A*B+A*E*I+A+B*E*J+B*E+E*J+E, n*t+n+t*J+J, n*s*H+s*H, n*s*E+n*s+s*E+s+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+z*B*J+z*B+A*E*I+A*F+B*E*J+B*E+B*I*J+B*I+B*J+B+F+I*J+I, n*s*D+s*D+t*C*D*J+t*C*D+t*D*F+t*D*J+t*D+z*B*D*J+z*B*D+A*D*F+B*D*I+B*D*J+B*D+D*F*J+D*F+D*I, n*s*z+s*z+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*F+A*F+B*I*J+B*I+E*F*J+I*J+I, n*s*x+n*s+s*x+s+x*z*B*J+x*z*B+x*B*J+x*B+x*E*J+x*E+x*F+z*B*J+z*B+B*J+B+E*F*J+E*J+E+F, n*r*s+r*J, n*p*s*y+n*p*s, m*J+J, m*G, m*F+m*I+F+I, m*D*E*H+D*E*H*J, m*C*I+m*C+m*E*I+m*E+m*I+m+C*I+C+E*I+E+I+1, m*C*E+m*E*I+C*E+E*I, m*B*I+v*B*I+B*I*J+F*J+F+I*J+I, m*B*H, m*B*D*E+m*D*E+B*D*E+D*E, m*B*C+m*B*E+m*B+v*B*E*I+B*C+B*E*I+B*E+B+E*F+E*I, m*A*I+v*A*I+A*F+A*I, m*A*H*K+A*H*I*K, m*A*E*H+A*E*H*I+A*E*H*J, m*A*C+m*A*E+m*A+A*C+A*E+A, m*A*B*E+m*A*E+A*B*E+A*E, m*z*I+m*I+z*I+I, m*z*D*E+z*D*E, m*z*C+m*z*E+m*z+m*C+m*E+m+z*C+z*E+z+C+E+1, m*z*B+s*E*J+s*E+s*J+s+v*B*E*I+z*B+B*E*I+B*E*J+B*E+B*J+B+E*F+E*I, m*z*A*K+z*A*K, m*z*A*E+z*A*E, m*y*C+m*y*E+m*y+m*C+m*E+m+y*C+y*E+y+C+E+1, m*x*I+m*I+v*B*E*I+x*I+B*E*I+E*F+E*I+I, m*x*E*H+m*E*H+x*E*H*J+A*E*H*I+E*H*J, m*x*C+m*x*E+m*x+m*C+m*E+m+v*B*E*I+x*C+x*E+x+B*E*I+C+E*F+E*I+E+1, m*x*B*E+m*x*E+m*B*E+m*E+x*B*E+x*E+B*E+E, m*x*z*E+m*z*E+v*B*E*I+x*z*E+z*E+B*E*I+E*F+E*I, m*t*D*E+m*D*E+t*D*E+D*E, m*t*C+m*t*E+m*t*I+m*t+t*C+t*E+t*I+t+v*A*I+v*B*E*I+v*B*I+A*F+B*E*I+B*I*J+B*I+E*F+E*I+F*J+F+I*J+I, m*t*x*E+m*t*E+m*x*E+m*E+t*x*E+t*E+x*E+E, m*s*H+s*H, m*s*E+m*s+m*B*E+m*B+s*E*J+s*J+B*E*J+B*J, m*s*D+m*B*D+m*D*E+s*D*E*J+s*D*E+s*D*J+B*D*J+D*E*J, m*s*B+m*s+s*B+s, m*s*A*K+m*A*E*K+m*A*K+s*A*K+A*E*K+A*K, m*s*z+s*z, m*s*x+m*s+m*x*B+m*x*E+m*B+m*E+s*x*E*J+s*x*E+s*x*J+s*E*J+s*E+s*J+x*B*J+x*E*J+A*E*I+B*J+E*J, m*r+m*s+r*s+s, m*p*y*D*E+m*p*D*E, m*p*y*B*E+m*p*y*B+p*r*y*B*E+p*r*y*B+p*y*B*E+p*y*B+r*B*E*J+r*B*J, m*p*y*B*D+m*p*D*E+p*r*y*B*D+p*r*y*D*E+p*y*B*D+p*y*D*E+r*B*D*J+r*D*E*J, m*p*y*A*K+m*p*A*E*K+p*r*y*A*E*K+p*r*y*A*K+p*y*A*E*K+p*y*A*K, m*p*y*A*H+m*p*A*H, m*p*y*A*E+m*p*y*A+m*p*A*B+m*p*A+p*r*y*A*B+p*r*y*A*E+p*y*A*B+p*y*A*E+r*A*E*J+r*A*J+r*B*J+r*J, m*p*y*A*D+m*p*A*B*D+m*p*A*D*E+m*p*A*D+p*r*y*A*B*D+p*r*y*A*D*E+p*y*A*B*D+p*y*A*D*E+r*A*D*J+r*B*D*J+r*D*E*J+r*D*J+s*B*D*E+s*D*E*J+s*D*E+B*C*D*E, m*p*y*A*B+m*p*y*A+m*p*A*B+m*p*A, m*p*y*z*A+m*p*z*A, m*p*x*y*E+m*p*x*E+m*p*y*E+m*p*E, m*p*x*y*B+m*p*x*E+m*p*y*B+m*p*E+p*r*x*y*B+p*r*x*y*E+p*r*y*B+p*r*y*E+p*x*y*B+p*x*y*E+p*y*B+p*y*E+r*x*E*J+r*B*E*J+r*E*J+s*x*B+s*x*J+s*x+s*B*E+s*B+s*E*J+s*E+s*J+s+x*B*C+x*B*J+x*B+x*E*J+x*E+x*F+B*C*E+B*C+B*E*I+B*J+B+E*F*J+E*F+E*I+E*J+E+F, m*p*x*y*A+m*p*x*A*B+m*p*x*A*E+m*p*x*A+m*p*y*A+m*p*A*B+m*p*A*E+m*p*A+p*r*x*y*A*B+p*r*x*y*A*E+p*r*y*A*B+p*r*y*A*E+p*x*y*A*B+p*x*y*A*E+p*y*A*B+p*y*A*E+r*x*E*J+r*B*E*J+r*E*J+s*x*B*E+s*x*E*J+s*x*E+x*B*C*E+x*F+A*E*I+B*E*I+E*F*J+E*F+E*I+F, m*p*v*I+m*p*I+v*I*J+I*J, m*p*v*C+m*p*v*E+m*p*v+m*p*C+m*p*E+m*p+p*v*C*J+p*v*E*J+p*v*J+p*C*J+p*E*J+p*J, m*p*s*y+p*r*s*y+p*s*y+r*J, m*n*E*H+A*E*H*I+E*H*J, m*n*A*H+m*A*H, m*n*s+n*s, m*n*p*y+m*n*p, k*J*K+J*K, k*I*J+t*I*J, k*H*J+A*H*J+H*J, k*H*I+F*H, k*G+G, k*F+F, k*E*K+n*E*K+x*F*K+E*F*J*K+E*J*K+E*K+F*K, k*E*I+t*E*I*J+E*I*J+E*I, k*D*I+D*I, k*D*E, k*C*K+k*K+n*E*K+x*F*K+C*K+E*F*J*K+E*J*K+F*K+K, k*C*J+k*C+k*E*J+k*E+k*J+k+C*J+C+E*J+E+J+1, k*C*I+k*I+t*C*I+t*C*J+t*C+t*E*I*J+t*E*I+t*E*J+t*E+t*I+t*J+t+C*J+C+E*I*J+E*I+E*J+E+J+1, k*C*H+k*E*H+k*H+A*C*H+A*E*H*I+A*E*H+A*H+C*H+E*H+H, k*C*E+k*C+k*E+k+t*C*E+t*C*J+t*E*F+t*E*J+t*J+C*J+C+E*F+E*J+E+J+1, k*C*D+k*D+t*C*D*E+t*C*D*J+t*D*E+t*D*J+C*D*J+C*D+D*J+D, k*B*J+k*J+A*B*J+A*J+B*J+J, k*B*I+k*I+B*I*J+B*I+F*J+I*J+I, k*B*H*K+n*E*H*K+A*H*I*K+B*E*H*K+B*H*I*K+E*H*J*K, k*B*E+k*E+A*B*E+A*E*I+A*E+B*E+E, k*B*C+k*B+k*C+k+A*B*C+A*B+A*C+A+B*C+B+C+1, k*A*K+A*F*K+A*J*K, k*A*J+A*B*J, k*A*E+n*A+z*A*B+A*B*E+A*B*J+A*B+A*E*I+A, k*A*C+k*A+n*A+z*A*B+A*B*C+A*B*J+A, k*z*J+z*A*J+z*B*J+z*J, k*z*I+F, k*z*E+z*A*E+z*B*E+z*E+A*E*I+B*E*I+E*F+E*I, k*z*C+k*z+z*A*C+z*A+z*C+z, k*z*B*K+B*I*K+F*K+I*K, k*z*A*B, k*x+k*E+k+n*E+E, k*v*I+v*I, k*t*J+k*J, k*t*E+k*E+t*E*J+t*E+E*J+E, k*t*C+k*t+k*C+k+t*C*J+t*C+t*J+t+C*J+C+J+1, k*t*B*H+k*B*H+t*B*H+B*H, k*t*z*B+k*z*B+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*F+A*F+B*I*J+B*I+E*F*J+I*J+I, k*s*K+k*B*K+n*s*K+n*E*K+s*J*K+s*K+t*C*K+t*E*K+t*F*K+t*K+v*B*I*K+z*B*K+A*F*K+B*K+C*J*K+F*J*K+J*K, k*s*J+k*C+k*E*J+k*E+k*J+k+s*t*J+t*B*J+t*C*J+t*J+A*B*J+A*J+B*J+C*J+C+E*J+E+1, k*s*H, k*s*E+k*C+k*E+k+n*s+s*t*E+s*t*J+s*t+s*E*J+s*E+s*J+t*B*E+t*B*J+t*B+t*C+t*E*J+t*E+t*F+t+v*B*E*I+x*F+z*B*J+z*B+A*B*E+A*E*I+A*E+A*F+B*E*I+B*E*J+B*I*J+B*I+C*J+C+E*F*J+E*F+E*I+E+I*J+I+J+1, k*s*B+k*s, k*s*A+k*A*B+n*s*A+n*A+s*A*J+s*A+s*B+s+A*B*J+A, k*s*z+k*s+k*z*B+k*B+k*C+k+n*s+s*t*J+s+t*B*J+t*C+t*E*J+t*E+t*F+t+x*F+z*B*J+z*B+A*F+B*I*J+B*I+B*J+B+C*J+C+E*F*J+I*J+I+J+1, k*s*t+k*s+k*t*B+k*B+s*t*J+s*t+s*J+s+t*B*J+t*B+B*J+B, k*r*J+r*B*J+s*t*J+s*B+s*J+s+t*B*J+t*C*J+t*J+A*B*C+A*B+A*C+A+B*J+C*J+J, k*r*E+n*r*E+r*B*E*J+r*E+s*t*E+s*t*J+s*t+s*B*E+s*J+s+t*B*E+t*B*J+t*B+t*C*E+t*E*F+t*E*J+v*B*E*I+A*B*C*E+A*B*E+A*C*E+A*E+B*E*I+B*E+B*J+B+C*E+E*I+E*J, k*r*C+k*r+n*r*E+r*B*J+r*C+r*E*J+r*J+r+s*t*J+s*B+s*J+s+t*B*J+t*C*J+t*J+v*B*E*I+A*B*C+A*B+A*C+A+B*E*I+B*J+C*J+E*F+E*I+J, k*r*B*H+k*B*H+k*E*H+n*r*E*H+r*B*H+r*E*H*J+A*E*H*J+B*E*H*I+B*H+E*H, k*r*s+k*r*B+k*s+k*B+k*C+k+n*r*E+n*s+r*s+r*B*J+r*B+r*E*J+r*J+s*t*J+s*E*J+s*E+s*J+t*B*J+t*C+t*E*J+t*E+t*F+t+x*F+z*B*J+z*B+A*F+B*E*J+B*E+B*I*J+B*I+C*J+C+E*F*J+I*J+I+J+1, k*p*I+p*I+t*I*J+I*J, k*p*s*y+k*p*s+k*p*y*z*B+k*p*y*B+k*p*y*E+k*p*z*B+k*p*B+k*p*E+p*s*y+p*s+p*y*z*B+p*y*B+p*y*E+p*z*B+p*B+p*E, k*n+k*z*B+k*A*B+k*A+k*E*J+k*E+k*J+n*E+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+A*F+B*I*J+B*I+E+F+I*J+I, k*m*I+k*I+m*I+I, k*m*E*H+m*E*H+A*E*H*J, k*m*C+k*m*E+k*m+k*C+k*E+k+m*C+m*E+m+C+E+1, k*m*s+k*m*B+k*m*E+k*C+k*E+k+m*s+m*B+m*E+n*s+s*t*J+s*E*J+s*E+s*J+s+t*B*J+t*C+t*E*J+t*E+t*F+t+v*B*E*I+x*F+z*B*J+z*B+A*F+B*E*I+B*E*J+B*E+B*I*J+B*I+B+C*J+C+E*F*J+E*F+E*I+E+I*J+I+J+1, k*m*p*y*E+k*m*p*E+m*p*y*E+m*p*E, k*m*p*y*B+k*m*p*E+k*p*r*y*B+k*p*y*B+k*p*y*E+m*p*y*B+m*p*E+n*p*r*y*E+p*r*y*B+p*v*B*E*I+p*y*B+p*y*E+p*B*E*I+r*E*J+s*t*J+s*B+s*J+s+t*B*J+t*C*J+t*J+A*B*C+A*B+A*C+A+B*J+C*J+E*F*J+E*I*J+J, k*m*p*y*A+k*m*p*A*B+k*m*p*A+k*p*r*y*A*B+k*p*y*A*B+m*n*p*A+m*p*y*A+n*p*r*y*A+n*p*y*A+p*r*y*A+p*y*A+r*A*J+r*B*J+r*J, j+r+1, i*K+t*C*K+t*E*K+t*F*K+t*K+x*F*K+C*J*K+E*F*J*K+E*J*K+F*J*K+F*K+J*K, i*J+k*J+J, i*I+t*C*J+t*C+t*E*J+t*E+t*F+t*I*J+t*J+t+x*F+E*F*J+F*J+F+I*J, i*G+B*G, i*F+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*F+E*F*J+F*J+F, i*E*H+i*H+A*E*H+A*H+B*E*H+B*H, i*C+i*E+i+k*C+k*E+k+x*F+C+E*F*J+E+F+1, i*B+i+A*B+A, i*A+A*B+A, i*z+i*E+i+k*E*J+k*J+x*F+z*A*B+z*A+z*B*J+A*B*J+A*B+A*J+A+E*F*J+E*J+F+J, i*t+i+t*J+J, i*s+i*E+i+k*C+k*E+k+s*t*J+s*E*J+s*E+s+t*B*J+t*C*J+t*J+v*B*E*I+x*F+A*B+A+B*E*I+B*E*J+B*E+B+C*J+C+E*F*J+E*F+E*I+E+F+J+1, i*r+r*A*B+r*A+s*t*J+s*B+s*E*J+s*E+t*B*J+t*C*J+t*J+v*B*E*I+A*B*C+A*B+A*C+A+B*E*I+B*E*J+B*E+B+C*J+E*F+E*I+J, i*p*y*E+i*p*y+i*p*E+i*p+p*y*z*B+p*y*A*B+p*y*A+p*z*B+p*A*B+p*A, i*n+i, i*m*H+m*A*H, i*m*E+i*m+i*E+i+m*A*B+m*A+s*E*J+s*E+s*J+s+A*B+A+B*E*J+B*E+B*J+B, i*m*p*y+i*m*p, i*k+k*z*B+k*A*B+k*A+k*E*J+k*E+n*E+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+A*F+B*I*J+B*I+E+F+I*J+I, h*J, h*F+h*I, h*E*I, h*D*I, h*D*E*H, h*C*I+h*C+h*E+h*I+h, h*C*E, h*C*D+h*D*E+h*D, h*B*H, h*B*D*E+h*D*E, h*B*C+h*B*E+h*B, h*A*I*K, h*A*H*K, h*A*E*H, h*A*C+h*A*E+h*A, h*A*B*E+h*A*E, h*z*I+h*I, h*z*D*E, h*z*C+h*z*E+h*z+h*C+h*E+h, h*z*B*K+h*B*I*K, h*z*B*E, h*z*A*K, h*z*A*E, h*z*A*B, h*y*C+h*y*E+h*y+h*C+h*E+h, h*x*I+h*I, h*x*E*H+h*E*H, h*x*C+h*x*E+h*x+h*C+h*E+h, h*x*B*E+h*x*E+h*B*E+h*E, h*x*z*E+h*z*E, h*v*B*I, h*v*A*I, h*t*D*E+h*D*E, h*t*C+h*t*E+h*t*I+h*t+h*A*I+h*B*I, h*t*z*B+h*z*B, h*t*x*E+h*t*E+h*x*E+h*E, h*s*H, h*s*E+h*s+h*z*B+h*B*E+h*B, h*s*D+h*z*B*D+h*B*D+h*D*E, h*s*B+h*s, h*s*A*K+h*A*E*K+h*A*K, h*s*z, h*s*x+h*s+h*x*z*B+h*x*B+h*x*E+h*z*B+h*B+h*E, h*r*B*I+h*B*I, h*r*B*E+h*r*B+h*B*E+h*B, h*r*B*D+h*r*D*E+h*B*D+h*D*E, h*r*A*E*K+h*r*A*K+h*A*E*K+h*A*K, h*r*A*B+h*r*A*E+h*A*B+h*A*E, h*r*x*B+h*r*x*E+h*r*B+h*r*E+h*x*B+h*x*E+h*B+h*E, h*r*s+h*s, h*q*B*I, h*q*B*G, h*q*B*E+h*q*B, h*q*B*D+h*q*D*E, h*q*A*E*K+h*q*A*K, h*q*A*B+h*q*A*E, h*q*z*B, h*q*x*B+h*q*x*E+h*q*B+h*q*E, h*q*s, h*q*r*D*E, h*q*r*B, h*q*r*A*K, h*q*r*A*E, h*q*r*x*E+h*q*r*E, h*n*E*H, h*n*A*H+h*A*H, h*n*s, h*n*q*r*E, h*n*q*r*A+h*q*r*A, h*m, h*k*I+h*I, h*k*E*H+h*E*H, h*k*C+h*k*E+h*k+h*C+h*E+h, h*k*s+h*k*z*B+h*k*B+h*k*E+h*s+h*z*B+h*B+h*E, h*k*r*B+h*k*B+h*k*E+h*n*r*E+h*r*B+h*B+h*E, h*k*q*B+h*k*q*E+h*q*B+h*q*E, h*i*H+h*A*H, h*i*E+h*i+h*z*B+h*A*B+h*A, g*J*K, g*I+g*J+g+n+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*F+E*F*J+F*J+F+J, g*G+B*G, g*F+g*J+g+n+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*F+E*F*J+F*J+F+J, g*E*K+g*K+n*E*K+n*K+t*C*K+t*E*K+t*F*K+t*K+C*K+E*J*K+E*K+F*K+J*K+K, g*E*J+g*E+g*J+g+n*E+n+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+C*J+C+E+F+1, g*D*K+n*D*K+t*C*D*K+t*D*F*K+t*D*K+C*D*K+D*F*K+D*J*K+D*K, g*D*J+g*D+n*D+t*C*D*J+t*C*D+t*D*F+t*D*J+t*D+C*D*J+C*D+D*F+D, g*C*K+g*K+n*E*K+x*F*K+E*F*J*K+E*J*K+F*K, g*C*J+g*C+g*J+g+n*E+x*F+E*F*J+E*J+F, g*C*H+g*H+n*E*H+A*E*H*I+B*E*H*I+E*H*J, g*C*E+g*C+g*E+g, g*B*K+n*E*K+x*F*K+z*B*K+E*F*J*K+E*J*K+F*K, g*B*J+g*B+n*E+x*F+z*B*J+z*B+E*F*J+E*J+F, g*B*H+n*E*H+A*E*H*I+B*E*H+B*H+E*H*J, g*B*C+g*B+g*C+g, g*A+g*B+g*J+n*A+n*E+x*F+z*B*J+z*B+A*J+E*F*J+E*J+F, g*z*K+g*K+n*z*K+n*K+z*J*K+J*K, g*z*J+g*z+g*J+g+n*z+n+z*J+J, g*z*C+g*z, g*z*B+z*B, g*y*K+g*K+n*y*K+n*K+y*J*K+J*K, g*y*J+g*y+g*J+g+n*y+n+y*J+J, g*x*K+g*K+n*x*K+n*K+E*J*K, g*x*J+g*x+g*J+g+n*x+n+x*J+J, g*t+k*J+n+t*J+J, g*s+g*B+g*C+g+n*s+s*J+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*F+z*B*J+z*B+A*F+B*I*J+B*I+E*F*J+I*J+I, g*r*J, g*r*E+g*r+n*r*E+n*r+r*A*F+r*C*J+r*C+r*E+r*F+r+v*B*E*I+v*B*I+B*E*I+B*I*J+B*I+E*F+E*I+F*J+F+I*J+I, g*r*D+n*r*D+r*A*D*F+r*C*D*J+r*C*D+r*D*E*J+r*D*E+r*D*F+r*D+v*B*D*I+B*D*I+D*F+D*I, g*r*C+g*r+n*r*E+r*E*J+v*B*E*I+B*E*I+E*F+E*I, g*r*B+n*r*E+r*E*J+s*E*J+s*E+s*J+s+B*E*J+B*E+B*J+B, g*r*z+g*r+n*r*z+n*r+r*z*J+r*J, g*r*y+g*r+n*r*y+n*r, g*r*x+g*r+n*r*x+n*r+r*x*J+r*J, g*p*y+g*p+n*p*y+n*p, g*n+g*J+n+J, g*m*E+g*m+g*E+g+m*n*E+m*n+m*C+m*E+m*I+m+n*E+n+v*A*I+v*B*E*I+v*B*I+A*F+B*E*I+B*I*J+B*I+C+E*F+E*I+E+F*J+F+I*J+1, g*m*D+g*D+m*n*D+m*C*D+m*D*E+m*D*I+m*D+n*D+v*A*D*I+v*B*D*I+A*D*F+B*D*I+C*D+D*E+D*F+D, g*m*C+g*m+g*C+g+m*n*E+n*E+v*B*E*I+B*E*I+E*F+E*I, g*m*B+g*B+m*n*E+n*E+s*E*J+s*E+s*J+s+B*E*J+B*E+B*J+B, g*m*z+g*m+g*z+g+m*n*z+m*n+n*z+n, g*m*y+g*m+g*y+g+m*n*y+m*n+n*y+n, g*m*x+g*m+g*x+g+m*n*x+m*n+n*x+n, g*m*p*v+g*m*p+g*p*v*J+g*p*J+m*n*p*v+m*n*p+p*v*J+p*J, g*k+g*J+g+k*z*B+k*A*B+k*A+k*E*J+k*E+n*E+n+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+A*F+B*I*J+B*I+E+F+I*J+I+J, g*i+g*J+i+k*J+J, g*h+h*n+h*A*I+h*B*I+h*C+h*E+h*I+h, f*I+m*I+v*I+I, f*F+m*I+v*I+I, f*D*E*K+D*E*K, f*D*E*J+f*D*E+D*E*J+D*E, f*C*J+f*C+f*E*J+f*E+f*J+f+m*C+m*E+m+v*C*J+v*C+v*E*J+v*E+v*J+v+C+E+1, f*B*G+B*G, f*B*E*H+f*B*H+B*E*H+B*H, f*B*D*H+B*D*H, f*A*H*K+A*H*K, f*A*E*K+f*A*J*K+A*E*K+A*J*K, f*A*E*H*J+f*A*E*H+f*A*H*J+f*A*H+A*E*H*J+A*E*H+A*H*J+A*H, f*A*D*H*J+f*A*D*H+A*D*H*J+A*D*H, f*A*D*E+f*B*D*E+f*D*E+A*D*E+B*D*E+D*E, f*A*B*J+f*A*B+f*A*J+f*A+A*B*J+A*B+A*J+A, f*z*B+z*B, f*z*A*K+z*A*K, f*z*A*J+f*z*A+z*A*J+z*A, f*y*J+f*J+y*J+J, f*y*D*E+f*D*E+y*D*E+D*E, f*y*C+f*y*E+f*y+f*C+f*E+f+y*C+y*E+y+C+E+1, f*y*A*E*H+f*y*A*H+f*A*E*H+f*A*H+y*A*E*H+y*A*H+A*E*H+A*H, f*y*A*D*H+f*A*D*H+y*A*D*H+A*D*H, f*y*A*B+f*y*A+f*A*B+f*A+y*A*B+y*A+A*B+A, f*y*z*A+f*z*A+y*z*A+z*A, f*x*E*K+f*E*J*K+f*E*K+x*E*K+E*J*K+E*K, f*x*E*J+f*x*E+f*E*J+f*E+x*E*J+x*E+E*J+E, f*x*B*H+f*B*H+x*B*H+B*H, f*x*A*H+f*x*H*J+f*A*E*H+f*A*H*J+f*E*H*J+f*H*J+x*A*H+x*H*J+A*E*H+A*H*J+E*H*J+H*J, f*x*A*E+f*x*B*E+f*x*E+f*A*E*J+f*A*E+f*B*E*J+f*B*E+f*E*J+f*E+x*A*E+x*B*E+x*E+A*E*J+A*E+B*E*J+B*E+E*J+E, f*x*A*B+f*x*A+f*x*B*J+f*x*J+f*A*B*E+f*A*B+f*A*E+f*A+f*B*E*J+f*B*J+f*E*J+f*J+x*A*B+x*A+x*B*J+x*J+A*B*E+A*B+A*E+A+B*E*J+B*J+E*J+J, f*x*z*A+f*x*z*J+f*z*A*E+f*z*A+f*z*E*J+f*z*J+x*z*A+x*z*J+z*A*E+z*A+z*E*J+z*J, f*x*y*E+f*x*E+f*y*E+f*E+x*y*E+x*E+y*E+E, f*t*H*J+f*A*H*J+f*H*J+t*H*J+A*H*J+H*J, f*t*D*E+t*D*E, f*t*C*K+f*t*E*K+f*t*K+f*C*K+f*E*K+f*K+m*t*I*K+m*C*K+m*E*K+m*K+t*v*I*K+t*C*K+t*E*K+t*F*K+t*I*K+t*K+v*C*J*K+v*C*K+v*E*J*K+v*E*K+v*I*J*K+v*J*K+v*K+C*J*K+C*K+E*J*K+E*K+F*J*K+J*K+K, f*t*B*J+f*t*J+f*A*B+f*A+f*B*J+f*J+t*B*J+t*J+A*B+A+B*J+J, f*t*z*J+f*z*A+f*z*J+t*z*J+z*A+z*J, f*t*x*J+f*t*E*J+f*t*J+t*x*J+t*E*J+t*J, f*t*x*E+f*t*E*J+f*t*E+t*x*E+t*E*J+t*E, f*t*x*C+f*t*x+f*t*C*E+f*t*C+f*t*E*J+f*t+m*t*E*I+t*v*E*I+t*x*C+t*x+t*C*E+t*C+t*E*F+t*E*I+t*E*J+t+v*E*I*J+E*F*J, f*s*E*J+f*s*E+f*s*J+f*s+f*B*E*J+f*B*E+f*B*J+f*B+s*E*J+s*E+s*J+s+B*E*J+B*E+B*J+B, f*s*D*J+f*s*D+f*B*D*J+f*B*D+s*D*J+s*D+B*D*J+B*D, f*s*B*D*E+f*B*C*D*E+s*D*E*J+s*D*E, f*s*A*J*K+f*s*A*K+f*A*J*K+f*A*K+s*A*J*K+s*A*K+A*J*K+A*K, f*s*A*D*K+f*A*D*K+s*A*D*K+A*D*K, f*s*y*H+f*s*H+s*y*H+s*H, f*s*y*E+f*s*y+f*s*E+f*s+f*y*B*E+f*y*B+f*B*E+f*B+s*y*E+s*y+s*E+s+y*B*E+y*B+B*E+B, f*s*y*D+f*s*D+f*y*B*D+f*B*D+s*y*D+s*D+y*B*D+B*D, f*s*y*B+f*s*y+f*s*B+f*s+s*y*B+s*y+s*B+s, f*s*y*A*K+f*s*A*K+f*y*A*K+f*A*K+s*y*A*K+s*A*K+y*A*K+A*K, f*s*y*z+f*s*z+s*y*z+s*z, f*s*x*K+f*s*E*K+f*s*J*K+f*x*B*K+f*B*E*K+f*B*J*K+s*x*K+s*E*K+s*J*K+x*B*K+B*E*K+B*J*K, f*s*x*J+f*s*x+f*s*J+f*s+f*x*B*J+f*x*B+f*B*J+f*B+s*x*J+s*x+s*J+s+x*B*J+x*B+B*J+B, f*s*x*B+f*s*B*E+f*s*B+f*s*J+f*s+f*x*B*C+f*B*C*E+f*B*C+f*B*E*J+f*B*E+f*B*J+f*B+s*x*J+s*x+s*E*J+s*E+x*B*J+x*B+x*E*J+x*E+x*F+B*E*I+B*E*J+B*E+E*F*J+E*F+E*I+E*J+E+F, f*s*x*A+f*s*x+f*s*A*E+f*s*A*J+f*s*E+f*s*J+f*x*A+f*x*B*C+f*x*B*J+f*x*J+f*A*E+f*A*J+f*B*C*E+f*B*C+f*B*E+f*B+f*E*J+f*J+s*x*A+s*x*B+s*x*J+s*A*E+s*A*J+s*B*E+s*B+s*E*J+s+x*A+x*B+x*E*J+x*E+x*F+x*J+A*E+A*J+B*E*I+B*E+B*J+E*F*J+E*F+E*I+E+F+J, f*s*x*y+f*s*x+f*s*y+f*s+f*x*y*B+f*x*B+f*y*B+f*B+s*x*y+s*x+s*y+s+x*y*B+x*B+y*B+B, f*s*t*J+f*s*B+f*s*J+f*s+f*t*C+f*t*E*J+f*t*E+f*t*J+f*t+f*A*B*C+f*A*C+f*C+f*E*J+f*E+f*J+f+m*t*I+m*C+m*E+m+t*v*I+t*B*J+t*C*J+t*C+t*E*J+t*E+t*F+t*I+t+v*C*J+v*C+v*E*J+v*E+v*I*J+v*J+v+A*B+A+B*J+C+E+F*J+J+1, f*s*t*E+f*s*t+f*s*B*E+f*s*B+f*t*B*E+f*t*B+f*t*C*E+f*t*C+f*t*E+f*t+f*A*B*C*E+f*A*B*C+f*A*B*E+f*A*B+f*A*C*E+f*A*C+f*A*E+f*A+f*B*E+f*B+f*C*E+f*C+f*E+f+m*t*E*I+m*t*I+m*C+m*E*I+m*E+m+t*v*E*I+t*v*I+t*C*J+t*C+t*E*I+t*E*J+t*E+t*F+t*I+t*J+t+v*C*J+v*C+v*E*I+v*E*J+v*E+v*I*J+v*J+v+C+E*I+E+F*J+1, f*s*t*D+f*s*B*D+f*t*B*D+f*t*C*D+f*t*D+f*A*B*C*D+f*A*B*D+f*A*C*D+f*A*D+f*B*D+f*C*D+f*D+m*t*D*I+m*C*D+m*D*E+m*D+t*v*D*I+t*C*D*J+t*C*D+t*D*F+t*D*I+t*D*J+t*D+v*C*D*J+v*C*D+v*D*E*J+v*D*E+v*D*I*J+v*D*J+v*D+C*D+D*E*J+D*F*J+D, f*r*J, f*r*D*E+r*D*E*J+r*D*E, f*r*C+f*r*E+f*r+r*v*C*J+r*v*C+r*v*E*J+r*v*E+r*v*J+r*v+r*C*J+r*C+r*E*J+r*E+r*J+r, f*r*B*E+f*r*B+s*E*J+s*E+s*J+s+B*E*J+B*E+B*J+B, f*r*B*D+r*D*E*J+r*D*E+s*D*E*J+s*D*E+s*D*J+s*D+B*D*J+B*D+D*E*J+D*E, f*r*A*K+r*A*E*K+r*A*J*K, f*r*A*E+f*r*A+r*A*B+r*A+r*B*J+r*J, f*r*A*D+r*A*B*D+r*A*D*E+r*A*D+r*B*D*J+r*D*E*J+r*D*J+s*B*D*E+s*D*E*J+s*D*E+B*C*D*E, f*r*A*B+f*r*A+r*A*B+r*A+r*B*J+r*J, f*r*z*A+r*z*A+r*z*J, f*r*x*E+f*r*E+r*x*E*J+r*x*E+r*E*J+r*E, f*r*x*B+f*r*B+r*x*E*J+r*x*E+r*E*J+r*E+s*x*E*J+s*x*E+s*x*J+s*x+s*E*J+s*E+s*J+s+x*B*J+x*B+x*E*J+x*E+A*E*I+B*J+B+E*J+E, f*r*x*A+f*r*A+r*x*A*B+r*x*A*E+r*x*A+r*x*E*J+r*x*J+r*A*B+r*A*E*J+r*A*E+r*A+r*B*E*J+r*J+s*x*B*E+s*x*B+s*x*E*J+s*x*E+s*x*J+s*x+s*B*E+s*B+s*E*J+s*E+s*J+s+x*B*C*E+x*B*C+x*B*J+x*B+x*E*J+x*E+A*E*I+B*C*E+B*C+B*J+B+E*J+E, f*r*s, f*p*J+f*J+p*J+J, f*p*D*E+f*D*E+p*D*E+D*E, f*p*C+f*p*E+f*p+f*C+f*E+f+m*p*C+m*p*E+m*p+m*C+m*E+m+p*v*C*J+p*v*C+p*v*E*J+p*v*E+p*v*J+p*v+p*C*J+p*C+p*E*J+p*E+p*J+p+v*C*J+v*C+v*E*J+v*E+v*J+v+C*J+C+E*J+E+J+1, f*p*B*E+f*p*B+f*B*E*J+f*B*J+p*r*y*B*E+p*r*y*B+p*s*E*J+p*s*E+p*s*J+p*s+p*y*B*E+p*y*B+p*B*E*J+p*B*E+p*B*J+p*B+r*B*E*J+r*B*J+B*E*J+B*J, f*p*B*D+f*B*D*J+p*r*y*B*D+p*r*y*D*E+p*s*D*E*J+p*s*D*E+p*s*D*J+p*s*D+p*y*B*D+p*y*D*E+p*B*D*J+p*B*D+p*D*E*J+r*B*D*J+r*D*E*J+B*D*J, f*p*A*K+f*A*J*K+p*r*y*A*E*K+p*r*y*A*K+p*y*A*E*K+p*y*A*K+p*A*E*K+A*J*K, f*p*A*E+f*p*A+f*A*E*J+f*A*J+p*r*y*A*B+p*r*y*A*E+p*y*A*B+p*y*A*E+p*A*B+p*A+r*A*E*J+r*A*J+r*B*J+r*J+A*E*J+A*J, f*p*A*D+f*A*D*J+p*r*y*A*B*D+p*r*y*A*D*E+p*y*A*B*D+p*y*A*D*E+p*A*B*D+p*A*D*E+p*A*D+r*A*D*J+r*B*D*J+r*D*E*J+r*D*J+s*B*D*E+s*D*E*J+s*D*E+A*D*J+B*C*D*E, f*p*A*B+f*p*A+f*A*B+f*A+p*A*B+p*A+A*B+A, f*p*z*A+f*z*A+p*z*A+z*A, f*p*x*E+f*p*E+f*x*E+f*E+p*x*E+p*E+x*E+E, f*p*x*B+f*p*B+f*x*B*J+f*B*J+p*r*x*y*B+p*r*x*y*E+p*r*y*B+p*r*y*E+p*s*x*E*J+p*s*x*E+p*s*x*J+p*s*x+p*s*E*J+p*s*E+p*s*J+p*s+p*x*y*B+p*x*y*E+p*x*B*J+p*x*B+p*x*E*J+p*y*B+p*y*E+p*A*E*I+p*B*J+p*B+p*E*J+r*x*E*J+r*B*E*J+r*E*J+s*x*B+s*x*J+s*x+s*B*E+s*B+s*E*J+s*E+s*J+s+x*B*C+x*B+x*E*J+x*E+x*F+B*C*E+B*C+B*E*I+B+E*F*J+E*F+E*I+E*J+E+F, f*p*x*A+f*p*A+f*x*B*J+f*x*J+f*A*E*J+f*B*E*J+f*B*J+f*E*J+f*J+p*r*x*y*A*B+p*r*x*y*A*E+p*r*y*A*B+p*r*y*A*E+p*x*y*A*B+p*x*y*A*E+p*x*A*B+p*x*A*E+p*x*A+p*y*A*B+p*y*A*E+p*A*B+p*A*E+p*A+r*x*E*J+r*B*E*J+r*E*J+s*x*B*E+s*x*E*J+s*x*E+x*B*C*E+x*B*J+x*F+x*J+A*E*I+A*E*J+B*E*I+B*E*J+B*J+E*F*J+E*F+E*I+E*J+F+J, f*p*s+f*s*J+p*r*s*y+p*s*y+r*J+s*J, f*n+f*J+n+J, f*m*D*E+f*D*E+m*D*E+D*E, f*m*C+f*m*E+f*m+f*C+f*E+f+m*v*C+m*v*E+m*v+m*C+m*E+m+v*C+v*E+v+C+E+1, f*m*B*E+f*m*B+f*B*E+f*B+s*E*J+s*E+s*J+s+B*E*J+B*E+B*J+B, f*m*B*D+f*B*D+m*D*E+s*D*E*J+s*D*E+s*D*J+s*D+B*D*J+B*D+D*E*J, f*m*A*K+f*A*K+m*A*E*K+A*E*K, f*m*A*H+f*A*H*J+m*A*H+A*H*J, f*m*A*E+f*m*A+f*A*E+f*A+m*A*B+m*A+A*B+A, f*m*A*D+f*A*D+m*A*B*D+m*A*D*E+m*A*D+A*B*D+A*D*E+A*D, f*m*A*B+f*m*A+f*A*B+f*A+m*A*B+m*A+A*B+A, f*m*z*A+f*z*A+m*z*A+z*A, f*m*x*E+f*m*E+f*x*E+f*E+m*x*E+m*E+x*E+E, f*m*x*B+f*m*B+f*x*B+f*B+m*x*E+m*E+s*x*E*J+s*x*E+s*x*J+s*x+s*E*J+s*E+s*J+s+x*B*J+x*B+x*E*J+A*E*I+B*J+B+E*J, f*m*x*A+f*m*A+f*x*A+f*A+m*x*A*B+m*x*A*E+m*x*A+m*A*B+m*A*E+m*A+x*A*B+x*A*E+x*A+A*B+A*E+A, f*m*s+f*s, f*m*p+f*J+m*p*y+J, f*k*J+f*t*J+k*J+t*J, f*k*E+f*t*E*J+f*E*J+f*E+k*E+t*E*J+E*J+E, f*k*C+f*k+f*t*C+f*t*E*J+f*t*E+f*t+f*E*J+f*E+k*C+k+m*t*I+m*C+m*E+m+t*v*I+t*C+t*E*J+t*E+t*F+t*I+t+v*C*J+v*C+v*E*J+v*E+v*I*J+v*J+v+C*J+E+F*J+J, f*k*B*H+f*B*H+k*B*H+B*H, f*k*A*H+f*A*H*J+f*A*H+k*A*H+A*H*J+A*H, f*k*A*B+f*k*A+k*A*B+k*A, f*k*z*A+k*z*A, f*k*s+f*k*B+f*s*B+f*t*C+f*t*E*J+f*t*E+f*t+f*A*B*C+f*A*B+f*A*C+f*A+f*B+f*C+f*E*J+f*E+f+k*s+k*B+m*t*I+m*C+m*E+m+s*t*J+s*J+s+t*v*I+t*B*J+t*C*J+t*C+t*E*J+t*E+t*F+t*I+t*J+t+v*C*J+v*C+v*E*J+v*E+v*I*J+v*J+v+B*J+B+C+E+F*J+1, f*k*r*B+f*r*B+k*s+k*B+k*C+k+n*r*E+n*s+r*E*J+s*t*J+s*E*J+s*E+s*J+t*B*J+t*C+t*E*J+t*E+t*F+t+x*F+z*B*J+z*B+A*F+B*E*J+B*E+B*I*J+B*I+C*J+C+E*F*J+I*J+I+J+1, f*k*r*A+f*r*A+k*r*A*B+k*r*A+n*r*A+r*A*J, f*k*p*B+f*p*B+f*t*J+f*A*B+f*A+f*J+k*p*r*y*B+k*p*s+k*p*y*B+k*p*y*E+k*p*B+k*p*C+k*p*E+k*p+n*p*r*y*E+n*p*s+p*r*y*B+p*s*B+p*s*E*J+p*s*E+p*s+p*t*C*J+p*t*C+p*t*E*J+p*t*E+p*t*I+p*t*J+p*t+p*x*I+p*y*B+p*y*E+p*z*B*J+p*z*B+p*A*B*C+p*A*B+p*A*C+p*A*I+p*A+p*B*E*J+p*B*E+p*B*I+p*B*J+p*C+p*E*I*J+p*E+p*I*J+p*I+p+r*E*J+t*I*J+t*J+x*I*J+A*B+A*I*J+A+B*I*J+E*I*J+J, f*k*p*A+f*p*A+f*A*B+f*A+k*p*r*y*A*B+k*p*y*A*B+k*p*A*B+k*p*A+n*p*r*y*A+n*p*y*A+n*p*A+p*r*y*A+p*y*A+r*A*J+r*B*J+r*J+A*B+A, f*k*m*B+f*k*B+f*m*B+f*B+k*m*E+k*s+k*B+k*C+k*E+k+m*E+n*s+s*t*J+s*E*J+s*E+s*J+t*B*J+t*C+t*E*J+t*E+t*F+t+v*B*E*I+x*F+z*B*J+z*B+A*F+B*E*I+B*E*J+B*E+B*I*J+B*I+C*J+C+E*F*J+E*F+E*I+E+I*J+I+J+1, f*k*m*A+f*k*A+f*m*A+f*A+k*m*A*B+k*m*A+k*A*B+k*A+m*n*A+n*A, f*i+f*t*J+f*J+i+t*J+J, f*h*D*E+h*D*E, f*h*C+f*h*E+f*h+h*v*C+h*v*E+h*v+h*C+h*E+h, f*h*B*E+f*h*B+h*z*B, f*h*B*D+h*z*B*D+h*D*E, f*h*A*K+h*A*E*K, f*h*A*H+h*A*H, f*h*A*E+f*h*A+h*A*B+h*A, f*h*A*D+h*A*B*D+h*A*D*E+h*A*D, f*h*A*B+f*h*A+h*A*B+h*A, f*h*z*A+h*z*A, f*h*x*E+f*h*E+h*x*E+h*E, f*h*x*B+f*h*B+h*x*z*B+h*x*E+h*z*B+h*E, f*h*x*A+f*h*A+h*x*A*B+h*x*A*E+h*x*A+h*A*B+h*A*E+h*A, f*h*s, f*h*q*r*A+h*q*r*A, f*h*k*B+f*h*B+h*k*z*B+h*k*E+h*z*B+h*E, f*h*k*A+f*h*A+h*k*A*B+h*k*A+h*n*A, f*g+g*m+g*v*J+g*v+g*J+g+m*n+n*v+v*J+J, e, d*I+m*v*I+I, d*F+m*v*I+F, d*D*E*H*K, d*D*E*H*J+d*D*E*H, d*C*J+d*C+d*E*J+d*E+d*J+d+m*v*C+m*v*E+m*v+v*C*J+v*E*J+v*J+C*J+C+E*J+E+J+1, d*B*G, d*B*E*H+d*B*H+B*E*H*I+B*H*I, d*B*D*H+B*D*H*I, d*B*C+d*B*E*J+d*B*E+d*B*J+d*B+v*B*E*I+B*E*I+C*F+C*I+E*F*J+E*I*J+F*J+F+I*J+I, d*A*H*K, d*A*D*E*H+d*D*E*H, d*A*C*K+d*A*E*K+d*A*K, d*z*B+v*B*I+B*I*J+B*I+F*J+F+I*J+I, d*z*A*K, d*x*E*H*K+d*E*H*J*K+d*E*H*K+B*E*H*K+B*H*I*K+B*H*K, d*x*E*H*J+d*x*E*H+d*E*H*J+d*E*H+B*E*H*I, d*x*B*H+d*B*H+B*E*H*I, d*x*A*E*H+d*x*E*H+d*A*E*H*J+d*A*E*H+d*E*H*J+d*E*H+B*E*H*I, d*v*D*E*H+d*D*E*H, d*v*C*D*E+d*C*D*E, d*v*B*D*E+d*v*D*E+d*B*D*E+d*D*E, d*v*A*J*K+d*A*J*K, d*v*A*E+d*A*E, d*v*z*D*E+d*z*D*E, d*v*x*E*H+d*v*E*H*J+d*v*E*H+d*x*E*H+d*E*H*J+d*E*H+B*E*H*I, d*v*x*C*E+d*x*C*E+v*B*E*I+v*E*I*J+v*E*I+x*C*E*I+x*F+B*E*I+E*F*J+E*F+E*I+F, d*v*x*B*E+d*v*x*E+d*v*B*E*J+d*v*B*E+d*v*E*J+d*v*E+d*x*B*E+d*x*E+d*B*E*J+d*B*E+d*E*J+d*E, d*v*x*z*E+d*v*z*E*J+d*v*z*E+d*x*z*E+d*z*E*J+d*z*E+v*B*E*I+B*E*I+E*F+E*I, d*t*C*K+d*t*E*K+d*t*K+d*C*K+d*E*K+d*K+m*t*v*I*K+m*v*C*K+m*v*E*K+m*v*K+t*F*K+v*B*E*I*K+v*B*I*K+v*C*J*K+v*E*J*K+v*I*J*K+v*J*K+B*E*I*K+B*I*K+C*J*K+C*K+E*F*K+E*I*K+E*J*K+E*K+F*J*K+F*K+I*K+J*K+K, d*t*B*J+d*t*C+d*t*E*J+d*t*E+d*t+d*A*B*J+d*A*C+d*A*E*J+d*A*E+d*A+d*B*J+d*C+d*E*J+d*E+d+m*t*v*I+m*v*C+m*v*E+m*v+t*C*I+t*C*J+t*C+t*E*I*J+t*E*I+t*E*J+t*E+t*F+t*I*J+t*I+t*J+t+v*A*I+v*B*E*I+v*B*I+v*C*J+v*E*J+v*I*J+v*J+A*C*I+A*F+A*I*J+A*I+B*E*I+B*I*J+B*I+C*I+E*F+E*I*J+F, d*t*B*D*E+d*t*C*D*E+d*t*D*E, d*t*x*B*E+d*t*x*C+d*t*x+d*t*B*E+d*t*C+d*t+d*A*B*E+d*A*C*E+d*A*E+d*B*E*J+d*C*E+d*E*J+m*v*E*I+t*C*E*I+t*E*F+v*B*E*I+v*E*I*J+B*E*I+C*E*I+E*F*J+E*F+E*I, d*t*v*E*H*J+d*t*E*H*J+d*v*E*H*J+d*E*H*J, d*t*v*C*E+d*t*C*E+d*v*C*E+d*C*E+t*v*E*I+t*C*E*I+v*E*I+C*E*I, d*t*v*z*E*J+d*t*z*E*J+d*v*z*E*J+d*z*E*J, d*t*v*x*C+d*t*v*x*E+d*t*v*x+d*t*v*C+d*t*v*E+d*t*v+d*t*x*C+d*t*x*E+d*t*x+d*t*C+d*t*E+d*t+d*v*C*E+d*C*E+t*C*E*I+t*E*F+v*B*E*I+v*E*I*J+v*E*I+B*E*I+C*E*I+E*F*J+E*F+E*I, d*s*E*J+d*s*E+d*s*J+d*s+d*B*E*J+d*B*E+d*B*J+d*B+v*B*E*I+v*B*I+B*E*I+B*I*J+B*I+E*F+E*I+F*J+F+I*J+I, d*s*B*D*E+d*s*D*J+d*s*D+d*B*D*J+d*B*D+d*D*E*J+d*D*E+v*B*D*I+B*D*I+D*F+D*I, d*s*x*B+d*s*x*J+d*s*x+d*s*B*E+d*s*B+d*B*E*J+d*B*E+d*B*J+d*B+v*B*E*I+v*B*I+x*B*I+x*I*J+x*I+A*E*I+B*E*I+B*I*J+E*F*J+E*F+E*I*J+E*I+F*J+F, d*s*v*E*H+d*s*E*H, d*s*v*D*E+d*s*v*D*J+d*s*v*D+d*s*D*E+d*s*D*J+d*s*D+d*v*B*D*J+d*v*B*D+d*v*D*E*J+d*v*D*E+d*B*D*J+d*B*D+d*D*E*J+d*D*E+v*B*D*I+B*D*I+D*F+D*I, d*s*v*B*E+d*s*v*E+d*s*B*E+d*s*E, d*s*v*A*K+d*s*A*K+d*v*A*K+d*A*K, d*s*v*A*J+d*s*v*A+d*s*A*J+d*s*A+d*v*A*B*J+d*v*A*B+d*A*B*J+d*A*B, d*s*v*z*E+d*s*z*E, d*s*v*x*E+d*s*v*x*J+d*s*v*x+d*s*x*E+d*s*x*J+d*s*x+d*v*x*B*J+d*v*x*B+d*v*x*E*J+d*v*x*E+d*v*B*E*J+d*v*B*E+d*v*E*J+d*v*E+d*x*B*J+d*x*B+d*x*E*J+d*x*E+d*B*E*J+d*B*E+d*E*J+d*E+v*B*E*I+v*B*I+x*E*I+x*F+B*E*I+B*I*J+B*I+E*F*J+E*F+E*I*J+F*J+I*J+I, d*s*v*x*A+d*s*v*x*J+d*s*v*A+d*s*v*E+d*s*v+d*s*x*A+d*s*x*J+d*s*A+d*s*E+d*s+d*v*x*A*B+d*v*A*B+d*v*B*E*J+d*v*B*E+d*v*B*J+d*v*B+d*x*A*B+d*A*B+d*B*E*J+d*B*E+d*B*J+d*B+v*B*E*I+v*B*I+x*B*I+x*I*J+x*I+A*E*I+B*E*I+B*I*J+E*F*J+E*F+E*I*J+E*I+F*J+F, d*s*t*J+d*s*B+d*s*J+d*s+t*I*J+B*I*J+F*J, d*s*t*E+d*s*t+d*s*B*E+d*s*B+d*t*B*E+d*t*B+d*t*C*E+d*t*C+d*t*E+d*t+d*A*B*E+d*A*B*J+d*A*C*E+d*A*C+d*A*E*J+d*A+d*B*E+d*B+d*C*E+d*C+d*E+d+m*t*v*E*I+m*t*v*I+m*v*C+m*v*E*I+m*v*E+m*v+t*C*E*I+t*C*I+t*C*J+t*C+t*E*I+t*E*J+t*E+t*F+t*I+t*J+t+v*A*I+v*B*E*I+v*B*I+v*C*J+v*E*J+v*I*J+v*J+A*C*I+A*F+A*I*J+A*I+B*E*I+B*I+C*E*I+C*I+E*F+F*J+F, d*s*t*D+d*s*B*D+d*t*B*D+d*t*C*D+d*t*D+d*A*B*D*J+d*A*C*D+d*A*D*E+d*A*D+d*B*D*E+d*B*D+d*C*D+d*D*E+d*D+m*t*v*D*I+m*v*C*D+m*v*D*E+m*v*D+t*D*F+v*A*D*I+v*B*D*I+v*C*D*J+v*D*E*J+v*D*I*J+v*D*J+A*C*D*I+A*D*F+A*D*I*J+A*D*I+B*D*I+C*D*J+C*D+D*E*J+D*E+D*F*J+D*F+D*I+D*J+D, d*r*J+r*I*J, d*r*D*E, d*r*C+d*r*E+d*r+r*C*I+r*E*I+r*I, d*r*B*E+d*r*B+d*B*E*J+d*B*E+d*B*J+d*B+r*B*I+r*E*F+r*E*I+v*B*I+B*I*J+B*I+E*F*J+E*F+E*I*J+E*I, d*r*B*D+d*B*D*J+d*B*D+d*D*E*J+d*D*E+r*D*F+r*D*I+D*F*J+D*F+D*I*J+D*I, d*r*A*K+d*A*E*K+d*A*K+r*A*I*K+A*F*K+A*I*K, d*r*A*E+d*r*A+d*A*B*J+d*A*B+d*A*E*J+d*A*E+r*A*I+A*F+A*I*J+A*I, d*r*A*D+d*A*B*D*J+d*A*B*D+d*A*D*E+d*B*D*E+d*D*E+r*A*D*I+A*D*F+A*D*I*J+A*D*I, d*r*A*B+d*r*A+r*A*F+r*B*I+r*F+r*I+v*B*I+B*I*J+B*I+F*J+F+I*J+I, d*r*z*A+r*z*I+r*A*F+r*F, d*r*x*E+d*r*E+r*E*I*J+v*B*E*I+x*C*E*I+x*E*I+B*E*I+E*F*J+E*I*J, d*r*x*B+d*r*B+d*x*B*J+d*x*B+d*x*E*J+d*x*E+d*B*J+d*B+d*E*J+d*E+r*x*I*J+r*x*I+r*E*I*J+r*I*J+r*I+x*B*I+x*C*I+x*F+x*I*J+x*I+A*E*I+B*I+C*E*I+C*I+E*F*J+E*F+E*I*J+F+I*J+I, d*r*x*A+d*r*A+d*x*A*B+d*x*A*E+d*x*B*E+d*x*E+d*A*B*E+d*A*B+d*B*E*J+d*B*E+d*E*J+d*E+r*x*A*I+r*A*I+x*A*I+x*B*I+x*I+A*I+B*I+E*F*J+I, d*r*v*A+d*r*A+d*v*A*B*J+d*v*A*B+d*A*B*J+d*A*B+r*A*I+A*F+A*I*J+A*I, d*p*y*D*E+d*p*D*E, d*p*y*A*E*K+d*p*A*E*K, d*p*y*A*E*H+d*p*y*A*H+d*p*A*E*H+d*p*A*H, d*p*y*A*D*H+d*p*A*D*H, d*p*y*A*B+d*p*y*A+d*p*A*B+d*p*A, d*p*y*z*A+d*p*z*A, d*p*x*y*E+d*p*x*E+d*p*y*E+d*p*E, d*p*x*y*A*H+d*p*x*A*H+d*p*y*A*H+d*p*A*H, d*p*v*y*A*H+d*p*v*A*H+d*p*y*A*H+d*p*A*H, d*p*s*v*y*A+d*p*s*v*A+d*p*s*y*A+d*p*s*A+d*p*v*y*A+d*p*v*A+d*p*y*A+d*p*A, d*p*r*s*y+d*p*r*s+d*p*s*y+d*p*s, d*n*E*H+d*E*H*J+B*E*H*I, d*n*A*H+d*A*E*H*J+d*A*E*H+d*A*H, d*n*v*A+d*n*A+d*v*A*B*J+d*v*A*B+d*v*A+d*A*B*J+d*A*B+d*A, d*n*r+r*A*F+r*I*J+v*B*I+B*I*J+B*I+F*J+F+I*J+I, d*n*p*y+d*n*p, d*m*D*E+d*D*E, d*m*C+d*m*E+d*m+d*C+d*E+d+m*C+m*E+m+C+E+1, d*m*B*E+d*m*B+d*B*E*J+d*B*J+E*F*J+E*F+E*I*J+E*I+F*J+F+I*J+I, d*m*B*D+d*B*D*J+d*D*E*J+d*D*E+D*F*J+D*F+D*I*J+D*I, d*m*A*K+d*A*E*K+A*F*K+A*I*K, d*m*A*H+d*A*E*H*J+d*A*E*H+d*A*H+v*A*H*I+A*H*I+F*H+H*I, d*m*A*E+d*m*A+d*A*B*J+d*A*B+d*A*E*J+d*A+v*A*I+A*I*J+A*I, d*m*A*D+d*A*B*D*J+d*A*B*D+d*A*D*E+d*A*D+d*B*D*E+d*D*E+v*A*D*I+A*D*I*J+A*D*I, d*m*A*B+d*m*A+d*A*B+d*A+v*A*I+A*F, d*m*z*A+d*z*A+v*A*I+A*F, d*m*x*E+d*m*E+d*x*E+d*E+v*B*E*I+B*E*I+E*F+E*I, d*m*x*B+d*m*B+d*x*B*J+d*x*E*J+d*x*E+d*B*J+d*E*J+d*E+v*B*E*I+x*F+x*I*J+x*I+B*E*I+E*F*J+E*F+E*I+F+I*J+I, d*m*x*A+d*m*A+d*x*A*B+d*x*A*E+d*x*A+d*x*B*E+d*x*E+d*A*B*E+d*A*B+d*A+d*B*E*J+d*B*E+d*E*J+d*E+x*A*I+x*B*I+x*I+A*I+B*I+E*F*J+I, d*m*v*A+d*m*A+d*v*A*B*J+d*v*A*B+d*v*A+d*A*B*J+d*A*B+d*A+v*A*I+A*I*J+A*I, d*m*s+d*r*s+d*s+r*F+r*I, d*m*p*y+d*m*p, d*m*n+d*n+v*A*I+v*B*I+A*F+B*I*J+B*I+F*J+F+I*J+I, d*k*E*J+d*k*J+d*t*C*E+d*t*C+d*t*E+d*t+d*A*C*E+d*A*C+d*A*E*J+d*A*E+d*A*J+d*A+d*C*E+d*C+d*E*J+d*E+d*J+d+m*t*v*E*I+m*t*v*I+m*v*C+m*v*E*I+m*v*E+m*v+t*C*E*I+t*C*I+t*C*J+t*C+t*E*I*J+t*E*I+t*E*J+t*E+t*F+t*I*J+t*I+t*J+t+v*A*I+v*B*E*I+v*B*I+v*C*J+v*E*J+v*I*J+v*J+A*C*I+A*F+A*I*J+A*I+B*E*I+B*I*J+B*I+C*E*I+C*I+E*F+E*I*J+F, d*k*E*H+d*A*E*H*J+d*E*H, d*k*D*J+d*t*C*D*E+d*t*C*D+d*t*D*E+d*t*D+d*A*C*D+d*A*D*E+d*A*D*J+d*A*D+d*B*D*E+d*C*D+d*D*E*J+d*D*J+d*D+m*t*v*D*I+m*v*C*D+m*v*D*E+m*v*D+t*D*F+v*A*D*I+v*B*D*I+v*C*D*J+v*D*E*J+v*D*I*J+v*D*J+A*C*D*I+A*D*F+A*D*I*J+A*D*I+B*D*I+C*D*J+C*D+D*E*J+D*E+D*F*J+D*F+D*I+D*J+D, d*k*C+d*k*E+d*k+d*t*C*E+d*t*C+d*t*E+d*t+d*A*C*E+d*C*E+m*t*v*E*I+m*t*v*I+m*v*C+m*v*E*I+m*v*E+m*v+t*C*E*I+t*F+v*A*I+v*B*E*I+v*B*I+v*C*J+v*E*J+v*I*J+v*J+A*F+B*E*I+B*I*J+B*I+C*E*I+C*J+C+E*F+E*I+E*J+E+F+I*J+I+J+1, d*k*B*H+d*B*H, d*k*s+d*k*B+d*k*E+d*n*s+d*s*B+d*s*J+d*t*C*E+d*t*C+d*t*E+d*t+d*A*B*J+d*A*C*E+d*A*C+d*A*E*J+d*A*E+d*A+d*B+d*C*E+d*C+d+m*t*v*E*I+m*t*v*I+m*v*C+m*v*E*I+m*v*E+m*v+t*C*E*I+t*C*I+t*C*J+t*C+t*E*I*J+t*E*I+t*E*J+t*E+t*F+t*I+t*J+t+v*A*I+v*B*E*I+v*B*I+v*C*J+v*E*J+v*I*J+v*J+A*C*I+A*F+A*I*J+A*I+B*E*I+B*I+C*E*I+C*I+E*F+E*I*J+F*J+F, d*k*r*B+d*k*B+d*k*E+d*r*B+d*t*C*E+d*t*C+d*t*E+d*t+d*A*B*J+d*A*C*E+d*A*C+d*A*E*J+d*A*E+d*A+d*B+d*C*E+d*C+d+k*r*I+k*I+m*t*v*E*I+m*t*v*I+m*v*C+m*v*E*I+m*v*E+m*v+r*B*I+r*F+t*C*E*I+t*C*I+t*C*J+t*C+t*E*I*J+t*E*I+t*E*J+t*E+t*F+t*I+t*J+t+v*A*I+v*B*E*I+v*C*J+v*E*J+v*I*J+v*J+A*C*I+A*F+A*I*J+A*I+B*E*I+C*E*I+C*I+E*F+E*I*J+F*J, d*k*r*A+d*k*A*B+d*n*A+d*r*A+d*A*B*J+d*A+k*r*A*I+k*A*I+r*A*I+A*I+B*I*J+F*J+I*J, d*k*p*y*E+d*k*p*E+d*p*y*E+d*p*E, d*k*p*y*A*H+d*k*p*A*H+d*p*y*A*H+d*p*A*H, d*k*m*E+d*k*E+d*m*E+d*E, d*k*m*B+d*k*E+d*m*B+d*t*C*E+d*t*C+d*t*E+d*t+d*A*B*J+d*A*C*E+d*A*C+d*A*E*J+d*A*E+d*A+d*C*E+d*C+d+k*I+m*t*v*E*I+m*t*v*I+m*v*C+m*v*E*I+m*v*E+m*v+t*C*E*I+t*C*I+t*C*J+t*C+t*E*I*J+t*E*I+t*E*J+t*E+t*F+t*I+t*J+t+v*A*I+v*B*E*I+v*B*I+v*C*J+v*E*J+v*I*J+v*J+A*C*I+A*F+A*I*J+A*I+B*E*I+B*I*J+B*I+C*E*I+C*I+E*F+E*I*J+F+I*J+I, d*k*m*A+d*k*A*B+d*k*A+d*m*A+d*n*A+d*A*B*J+k*A*I+A*I+B*I*J+F*J+I*J, d*i*H+d*A*E*H*J+d*A*E*H+d*A*H+B*H*I, d*i*E+d*i+d*t*C*E+d*t*C+d*t*E+d*t+d*A*B*J+d*A*B+d*A*C*E+d*A*C+d*A*E*J+d*A*E+d*C*E+d*C+d*E+d+m*t*v*E*I+m*t*v*I+m*v*C+m*v*E*I+m*v*E+m*v+t*C*E*I+t*C*I+t*C*J+t*C+t*E*I*J+t*E*I+t*E*J+t*E+t*F+t*I*J+t*I+t*J+t+v*A*I+v*C*J+v*E*J+v*I*J+v*J+A*C*I+A*F+A*I*J+A*I+C*E*I+C*I+E*I*J+E*I+F*J+I*J+I, d*i*p*y+d*i*p, d*i*m+d*i+v*A*I+v*B*I+A*F+B*I*J+B*I+F*J+F+I*J+I, d*h*D*E, d*h*C+d*h*E+d*h+h*C+h*E+h, d*h*A*H+h*A*H*I, d*h*A*E*K, d*h*A*B+d*h*A+h*A*I, d*h*z*A+h*A*I, d*h*x*E+d*h*E, d*h*s*v*A+d*h*s*A+d*h*v*A+d*h*A+h*A*I, d*h*q*A*K, d*h*q*A*E+d*h*q*A+h*q*A*I, d*h*q*A*D, d*h*q*x*A+d*h*q*A, d*h*q*v*A+d*h*q*A+h*q*A*I, d*h*q*r*A+h*q*r*A*I, d*h*n+h*A*I+h*B*I, d*h*k*E+d*h*E, d*h*k*q*A+d*h*q*A, d*h*i+h*A*I+h*B*I, d*g*K+d*n*K+d*J*K+g*m*v*K+g*K+m*n*v*K+n*K+v*J*K+J*K, d*g*J+d*g+d*n+d*J+g*m*v+g*v*J+g*J+g+m*n*v+n+v*J+J, d*g*B+d*g*C+d*g+v*B*I+B*I*J+B*I+F*J+F+I*J+I, d*g*v*E*H+d*g*E*H+g*v*H*J+g*v*H+g*H*J+g*H+n*v*H+n*H+v*A*H*I+v*C*H*J+v*C*H+v*E*H*J+v*E*H+v*H*I*J+v*H*I+v*H+A*H*I+B*H*I+C*H*J+C*H+E*H*J+E*H+F*H+H*I*J+H, d*g*v*C+d*g*v*E+d*g*v+d*g*C+d*g*E+d*g+g*v*J+g*v+g*J+g+n*v+n+v*A*I+v*B*I+v*C*J+v*C+v*E*J+v*E+v*I*J+v*I+v+A*F+B*I*J+B*I+C*J+C+E*J+E+I*J+I+1, d*g*v*z*E+d*g*z*E+g*v*J+g*v+g*J+g+n*v+n+v*A*I+v*B*I+v*C*J+v*C+v*E*J+v*E+v*I*J+v*I+v+A*F+B*I*J+B*I+C*J+C+E*J+E+I*J+I+1, d*g*r+g*r+n*r+r*A*F+r*J+v*B*I+B*I*J+B*I+F*J+F+I*J+I, d*g*m+d*g+g*m+g+m*n+n+v*A*I+v*B*I+A*F+B*I*J+B*I+F*J+F+I*J+I, d*f+d*m+v*I+I, c*I*J+c*I+c*J+c+I*J+I+J+1, c*G+G, c*F*J+c*F+c*J+c+F*J+F+J+1, c*F*H+c*H*I, c*E+c*F+c*J+C*J+C+E*F*J+J+1, c*D*J+c*D+C*D*J+C*D+D*E*J+D*E+D*F*J+D*F, c*D*I*K+c*D*K+D*I*J*K+D*I*K+D*J*K+D*K, c*D*F+c*D*I+D*F*J+D*F+D*I*J+D*I, c*C*J+c*C+c*J+c+C*J+C+J+1, c*B*I+c*B*J+c*B+c*F+c*I+B*I*J+B*I+B*J+B+F*J+F+I*J+I, c*B*D*K+B*D*I*K+B*D*J*K+B*D*K+D*F*K+D*I*K, c*B*D*H+B*D*H*I+B*D*H, c*A*J+c*A+c*B*J+c*B+c*J+c+g*J+g+n+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*F+A*F+A*J+A+B*I*J+B*I+B*J+B+E*F*J+I*J+I+1, c*A*I+c*B*J+c*B+c*J+c+g*J+g+n+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*F+A*F+A*I*J+A*I+B*I*J+B*I+B*J+B+E*F*J+I*J+I+1, c*A*H*K+c*B*H*K+c*H*J*K+c*H*K+g*H*K+n*H*K+A*H*K+B*H*K+H*K, c*A*F+c*B*J+c*B+c*J+c+g*J+g+n+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*F+B*I*J+B*I+B*J+B+E*F*J+I*J+I+1, c*A*D+c*B*D+c*D*I+c*D+A*D*F+A*D*J+A*D+B*D*I+B*D*J+B*D+D*F+D*I*J+D*J+D, c*z*J+c*z+c*J+c+z*J+z+J+1, c*z*I+c*F, c*z*D*K+c*D*K+z*D*J*K+z*D*K+D*J*K+D*K, c*z*B*K+c*B*J*K+c*B*K+z*B*K+B*J*K+B*K, c*z*A*K+c*B*J*K+c*B*K+c*J*K+c*K+g*K+n*K+t*C*K+t*E*K+t*F*K+t*K+x*F*K+z*A*K+A*F*K+B*I*K+B*J*K+B*K+C*J*K+E*F*J*K+E*J*K+F*J*K+I*K+J*K+K, c*z*A*B+z*A*B, c*y*J+c*y+c*J+c+y*J+y+J+1, c*y*D*K+c*D*K+y*D*J*K+y*D*K+D*J*K+D*K, c*y*B*H+c*B*H+y*B*H+B*H, c*x*K+c*F*K+c*J*K+c*K+g*K+n*K+t*C*K+t*E*K+t*F*K+t*K+x*K+C*J*K+E*F*J*K+F*K+K, c*x*J+c*x+g*J+g+n+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*J+x+F*J+J, c*x*H*I+c*H*I+c*H*J+c*H+g*H*J+g*H+n*H+A*H*I+B*H*I+E*H*I*J+F*H+H*I+H, c*x*F+c*F+c*J+c+g*J+g+n+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*F+F*J+F+1, c*x*B*H+x*B*H+B*E*H*I+B*H*I, c*x*A+c*x*B+c*x+c*F+c*J+c+g*J+g+n+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*A+x*B+x*F+x+A*E*J+A*F+A*J+B*E*J+B*I*J+B*I+B*J+E*J+F*J+F+I*J+I+J+1, c*v*H*J+c*v*H+c*B*H+c*H*J+c*H+g*v*H*J+g*v*H+g*H*J+g*H+n*v*H+n*H+v*A*H*I+v*H+A*H*I+B*H+F*H+H*I+H, c*v*B*J+c*v*B+c*v*J+c*v+c*B*J+c*B+c*J+c+g*v*J+g*v+g*J+g+n*v+n+v*A*I+v*B*J+v*B+v+A*F+B*J+B+1, c*v*B*H+v*B*H, c*t*K+c*K+g*K+n*K+t*K+J*K+K, c*t*J+c*t+c*J+c+g*J+g+n+t*J+t+1, c*t*H*I+c*H*I+g*H*J+g*H+n*H+t*H*I+F*H+H*J, c*t*F+c*F+g*J+g+n+t*F+F+J, c*t*D*I+c*D*I+t*C*D*J+t*C*D+t*D*F+t*D*I+t*D*J+t*D+C*D*J+C*D+D*F+D*I+D*J+D, c*t*C*D+c*t*D+c*C*D*I+c*D*I+t*C*D*E+t*C*D*J+t*C*D+t*D*E+t*D*J+t*D+C*D*J+C*D+D*J+D, c*t*B*H+c*B*H+t*B*H+B*H, c*t*z*B*D+c*t*B*D+t*B*D*J+t*B*D+t*C*D*E+t*C*D*J+t*C*D+t*D*E+t*D*F+t*D*J+t*D+z*B*D*J+z*B*D+A*D*F+B*D*I+D*F*J+D*F+D*I, c*t*x*I+c*F+c*J+c+t*x*I+t*C*J+t*C+t*E*I*J+t*E*J+t*E+t*F+t*I*J+t*J+t+E*F*J+F+J+1, c*t*x*C+c*t*x+c*C*F+c*F+t*x*C+t*x+t*C*E+t*C*J+t*E*F+t*E*J+t*J+C*E*F+C*J+C+E*J+E+J+1, c*t*x*z*B+c*t*x*B+t*x*B*J+t*x*B+t*x*C+t*x+t*C*E+t*C*J+t*E*F+t*E+t*F+t*J+x*z*B*J+x*z*B+x*E*J+x*E+A*F+B*I*J+B*I+E*F*J+E*J+E+F+I*J+I, c*t*v*I+c*v*I+g*v*J+g*v+n*v+t*v*I+v*I+v*J, c*s*J+c*s+c*v*J+c*v+c*B*J+c*B+c*J+c+g*v*J+g*v+g*J+g+n*v+n+s*J+s+v*A*I+v+A*F+B*J+B+1, c*s*D*K+s*D*J*K+s*D*K+v*B*D*I*K, c*s*y*B*D+c*s*B*D+c*y*B*C*D+c*B*C*D+s*y*D*J+s*y*D+s*D*J+s*D+y*B*D*J+y*B*D+y*D*E*J+y*D*E+B*D*J+B*D+D*E*J+D*E, c*s*t*D+c*t*B*D+s*D*J+s*D+t*C*D*E+t*D*E+v*B*D*I+B*D*I+B*D*J+B*D+D*E*J+D*E+D*F+D*I, c*s*t*y+c*s*t+c*s*y*B+c*s*B+c*t*y*B+c*t*y*C+c*t*y+c*t*B+c*t*C+c*t+c*y*A*B*C+c*y*A*B+c*y*A*C+c*y*A+c*y*B+c*y*C+c*y+c*A*B*C+c*A*B+c*A*C+c*A+c*B+c*C+c+s*t*y+s*t+s*y*B+s*B+t*y*B+t*y*C+t*y+t*B+t*C+t+y*A*B*C+y*A*B+y*A*C+y*A+y*B+y*C+y+A*B*C+A*B+A*C+A+B+C+1, c*r*J+c*r+c*v*J+c*v+c*J+c+g*r+g*v*J+g*v+g*J+g+n*r+n*v+n+r*A*F+r+v*A*I+v+A*F+1, c*r*H*K+c*B*H*K+g*r*H*K+n*r*H*K+r*H*K+B*H*K, c*r*H*I+c*B*H+g*r*H+n*r*H+r*A*H*I+r*H*I+r*H*J+B*H, c*r*F+c*v*J+c*v+c*J+c+g*r+g*v*J+g*v+g*J+g+n*r+n*v+n+r*A*F+r*F+r*J+v*A*I+v+A*F+1, c*r*D*K+r*C*D*J*K+r*C*D*K+r*D*E*K+r*D*F*K, c*r*D*I+r*C*D*J+r*C*D+r*D*E*J+r*D*E+r*D*F+r*D*I*J+r*D*I+r*D*J+r*D, c*r*C*H+c*B*H+g*r*H+n*r*H+r*A*H*I+r*C*H+r*H*I*J+r*H*J+B*H, c*r*C*D+r*D*E*J+r*D*E+r*D*F+r*D*J+r*D, c*r*B*K+c*r*K+g*r*K+n*r*K+r*B*K+r*J*K+r*K, c*r*B*H+c*B*H+r*B*H+B*H, c*r*B*D+c*s*B*D+c*B*C*D+r*B*D*J+r*B*D+s*D*J+s*D+B*D*J+B*D+D*E*J+D*E, c*r*A*H+c*r*H+c*B*H+g*r*H+n*r*H+r*A*H+r*H*J+r*H+B*H, c*r*A*B+c*r*A+c*r*B+c*r+g*r+n*r+r*A*B+r*A+r*B+r*J+r, c*r*z*K+c*v*J*K+c*v*K+c*J*K+c*K+g*r*K+g*v*K+g*K+n*r*K+n*v*K+n*K+r*z*K+r*J*K+v*K+K, c*r*z*A+c*r*z+c*v*J+c*v+c*J+c+g*r+g*v*J+g*v+g*J+g+n*r+n*v+n+r*z*A+r*z+r*J+v*A*I+v+A*F+1, c*r*y+c*r+r*y+r, c*r*x*I+c*x*C*I+c*x*I+c*C*F+c*F+g*r+n*r+r*x*I*J+r*x*I+r*A*F+r*J+v*B*E*I+v*B*I+B*E*I+B*I*J+B*I+C*E*F+E*F*J+E*F+E*I*J+F*J+F+I*J+I, c*r*x*B+c*s*x*B+c*x*B*C+r*x*B+r*B*E*J+r*B*J+s*x*B+s*B*E+s*B+s*E*J+s*E+s*J+s+x*B*C+B*C*E+B*C+B*E*J+B*E+B*J+B, c*r*v*I+g*r*v+n*r*v+r*v*I+r*v*J, c*r*t+c*r*B+c*s*t+c*s*B+c*t*B+c*t*C+c*t+c*A*B*C+c*A*B+c*A*C+c*A+c*B+c*C+c+r*t+r*A*F+r*B+s*t+s*B+t*B+t*C+t+A*B*C+A*B+A*C+A+B+C+1, c*r*s+c*r+c*v*J+c*v+c*J+c+g*r+g*v*J+g*v+g*J+g+n*r+n*v+n+r*s+r*A*F+r*J+r+v*A*I+v+A*F+1, c*p*y+c*p+p*y+p, c*p*x*I+c*p*I+c*p*J+c*p+c*x*I+c*I+c*J+c+g*p*J+g*p+g*J+g+n*p+n+p*t*C*J+p*t*C+p*t*E*J+p*t*E+p*t*I+p*t*J+p*t+p*x*I+p*I*J+p*I+p+t*C*J+t*C+t*E*J+t*E+t*F+t*I*J+t*J+t+x*I+F*J+I*J+I+1, c*p*v*I+c*p*v*J+c*p*v+c*p*I+c*p*J+c*p+c*v*I+c*v*J+c*v+c*I+c*J+c+p*v*I+p*v*J+p*v+p*I+p*J+p+v*I+v*J+v+I+J+1, c*p*v*D*K+c*p*D*K+c*v*D*K+c*D*K+p*v*C*D*J*K+p*v*C*D*K+p*v*D*E*K+p*v*D*I*K+p*C*D*J*K+p*C*D*K+p*D*E*K+p*D*I*K+v*C*D*J*K+v*C*D*K+v*D*E*K+v*D*I*K+C*D*J*K+C*D*K+D*E*K+D*F*J*K+D*F*K+D*I*J*K, c*p*t*I+c*p*I+c*t*I+c*I+g*p*J+g*p+g*J+g+n*p+n+p*t*I+p*I+p*J+t*I+I+J, c*p*s*B*D+c*p*B*C*D+c*s*B*D+c*B*C*D+p*s*D*J+p*s*D+p*v*B*D*I+p*B*D*I+p*B*D*J+p*B*D+p*D*E*J+p*D*E+s*D*J+s*D+v*B*D*I+B*D*I+B*D*J+B*D+D*E*J+D*E+D*F*J+D*F+D*I*J+D*I, c*p*s*t+c*p*s*B+c*p*t*B+c*p*t*C+c*p*t+c*p*A*B*C+c*p*A*B+c*p*A*C+c*p*A+c*p*B+c*p*C+c*p+c*s*t+c*s*B+c*t*B+c*t*C+c*t+c*A*B*C+c*A*B+c*A*C+c*A+c*B+c*C+c+p*s*t+p*s*B+p*t*B+p*t*C+p*t+p*A*B*C+p*A*B+p*A*C+p*A+p*B+p*C+p+s*t+s*B+t*B+t*C+t+A*B*C+A*B+A*C+A+B+C+1, c*p*r+c*p*v*J+c*p*v+c*p*J+c*p+c*r+c*v*J+c*v+c*J+c+g*p*r+g*p*v*J+g*p*v+g*p*J+g*p+g*r+g*v*J+g*v+g*J+g+n*p*r+n*p*v+n*p+n*r+n*v+n+p*r*A*I+p*r+p*v*A*I+p*v+p*A*I+p+r*A*F+r*A*I*J+r+v*A*I+v+A*F+A*I*J+1, c*n+c+g*J+g+J+1, c*m+c*v*J+c*v+c*J+g*m+g*v*J+g*v+g*J+m*n+m+n*v+v, c*k+c*t*z*B+c*t*B+c*A*B+c*A+c*B+c*J+g*J+g+k*C+k*E*J+k*E+k*J+n+t*B*J+t*B+t*C*J+t*J+z*B*J+z*B+A*B*J+A*B+A*J+A+B*J+B+C*J+C+E*J+E+1, c*i+c*t*z*B+c*t*B+c*A*B+c*A+c*B+i+k*C+k*E*J+k*E+k*J+k+t*B*J+t*B+t*C*J+t*J+z*B*J+z*B+A*B*J+A*B+A*J+A+B*J+B+C*J+C+E*J+E+1, c*h+h*C+h*E+h*I, c*g*K+c*J*K+c*K+g*K+J*K+K, c*g*J+c*g+c*J+c+g*J+g+J+1, c*g*C+c*g+g*C+g, c*g*B+c*z*B+g*C+g, c*g*r+c*v*J+c*v+c*J+c+g*v*J+g*v+g*J+g+n*r+n*v+n+r*A*F+r*J+v*A*I+v+A*F+1, c*f*J+c*f+c*J+c+f*J+f+g*m+g*v*J+g*v+g*J+m*n+n*v+v*J+1, c*f*D*K+c*v*D*K+f*D*J*K+f*D*K+m*C*D*K+m*D*E*K+m*D*I*K+m*D*K+v*D*J*K+v*D*K+C*D*K+D*E*K+D*I*K+D*K, c*f*B*H+c*B*H+f*B*H+B*H, c*f*y+c*f+c*y+c+f*y+f+y+1, c*f*t*H+c*f*A*H+c*f*H+c*t*H+c*A*H+c*H+f*t*H+f*A*H+f*H+g*m*H+g*v*H*J+g*v*H+g*H*J+m*n*H+n*v*H+t*H+v*H*J+A*H+H*J+H, c*f*t*D+c*t*D+c*v*D*I+c*D*I+f*t*D*J+f*t*D+t*D*J+t*D+v*C*D*J+v*C*D+v*D*E*J+v*D*E+v*D*J+v*D+C*D*J+C*D+D*E*J+D*E+D*F*J+D*F+D*I*J+D*I+D*J+D, c*f*t*B+c*f*t+c*f*A*B+c*f*A+c*f*B+c*f+c*t*B+c*t+c*A*B+c*A+c*B+c+f*t*B+f*t+f*A*B+f*A+f*B+f+g*m+g*v*J+g*v+g*J+m*n+n*v+t*B+t+v*J+A*B+A+B+J+1, c*f*t*z+c*f*z*A+c*f*z+c*t*z+c*z*A+c*z+f*t*z+f*z*A+f*z+g*m+g*v*J+g*v+g*J+m*n+n*v+t*z+v*J+z*A+z+J, c*f*t*x+c*t*x+c*v*I+c*v*J+c*v+c*F+c*J+c+f*t*x+f*t*E*J+f*t*J+t*x+t*E*J+t*J+v*E*I*J+v*I*J+v*I+v*J+v+E*F*J+F*J+F+J+1, c*f*s*B*D+c*f*B*C*D+s*D*J+s*D+v*B*D*I+B*D*I+B*D*J+B*D+D*E*J+D*E+D*F+D*I, c*f*s*t+c*f*s*B+c*f*t*C+c*f*A*B*C+c*f*A*C+c*f*C+c*t*B+c*t+c*A*B+c*A+c*B+c+f*s*t+f*s*B+f*t*C+f*A*B*C+f*A*C+f*C+g*m+g*v*J+g*v+g*J+m*n+n*v+t*B+t+v*J+A*B+A+B+J+1, c*f*r+c*v*J+c*v+c*J+c+f*r+g*r*v+g*r+g*v*J+g*v+g*J+g+n*r*v+n*r+n*v+n+r*v*J+r*A*F+r*J+v*A*I+v+A*F+1, c*f*p+c*f+c*p+c+f*p+f+g*m*p+g*m+g*p*v*J+g*p*v+g*p*J+g*v*J+g*v+g*J+m*n*p+m*n+n*p*v+n*v+p*v*J+p*J+p+v*J+J+1, c*d*J+c*d+c*v*J+c*v+c*J+c+d*J+d+g*m*v+g*v+m*n*v+n*v+v*J+v+J+1, c*d*D*K+c*v*D*K+c*D*K+d*D*J*K+d*D*K+m*v*C*D*K+m*v*D*E*K+m*v*D*I*K+m*v*D*K+v*C*D*K+v*D*E*K+v*D*I*K+v*D*J*K+D*J*K+D*K, c*d*B*H+c*B*H+d*B*H+B*H, c*d*v*D*H+c*d*D*H+c*v*D*H*I+c*D*H*I+d*v*D*H*J+d*v*D*H+d*D*H*J+d*D*H+v*D*H*I*J+v*D*H*I+D*H*I*J+D*H*I, c*d*v*C*D+c*d*C*D+c*v*C*D*I+c*C*D*I+d*v*D*E*J+d*v*D*E+d*v*D*J+d*v*D+d*D*E*J+d*D*E+d*D*J+d*D+v*D*I*J+v*D*I+D*I*J+D*I, c*d*v*B*D+c*d*v*D+c*d*B*D+c*d*D+c*v*D*I+c*D*I+d*v*B*D*J+d*v*B*D+d*v*D*J+d*v*D+d*B*D*J+d*B*D+d*D*J+d*D+v*B*D*I+v*D*I*J+v*D*I+B*D*I+D*F+D*I*J, c*d*v*A+c*d*A+d*v*A*J+d*v*A+d*A*J+d*A+v*A*I+A*F, c*d*v*z*D+c*d*z*D+c*v*D*I+c*D*I+d*v*z*D*J+d*v*z*D+d*z*D*J+d*z*D+v*D*I*J+v*D*I+D*I*J+D*I, c*d*v*x*H+c*d*x*H+c*v*H*I+c*B*H+c*H*I+d*v*x*H*J+d*v*x*H+d*x*H*J+d*x*H+v*E*H*I*J+v*H*I*J+v*H*I+B*E*H*I+B*H*I+B*H+E*H*I*J+H*I*J+H*I, c*d*v*x*C+c*d*x*C+c*v*C*I+c*v*J+c*v+c*x*C*I+d*v*x*E*J+d*v*x*E+d*v*x*J+d*v*x+d*x*E*J+d*x*E+d*x*J+d*x+g*v*J+g*v+n*v+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+v*A*I+v*B*I+v*C*E*I+v*I*J+v*I+v+x*I*J+x*I+A*F+B*I*J+B*I+E*I*J+E*I+F+I*J+I, c*d*v*x*B+c*d*v*x+c*d*x*B+c*d*x+c*v*I+c*v*J+c*v+c*F+c*J+c+d*v*x*B*J+d*v*x*B+d*v*x*J+d*v*x+d*x*B*J+d*x*B+d*x*J+d*x+g*v*J+g*v+g*J+g+n*v+n+v*A*I+v*E*I*J+v*I*J+v*I+v+A*F+E*F*J+F*J+F+1, c*d*v*x*z+c*d*x*z+c*v*I+c*v*J+c*v+c*F+c*J+c+d*v*x*z*J+d*v*x*z+d*x*z*J+d*x*z+g*v*J+g*v+g*J+g+n*v+n+v*A*I+v*B*E*I+v*B*I+v*E*I*J+v*I*J+v*I+v+A*F+B*E*I+B*I*J+B*I+E*F*J+E*F+E*I+I*J+I+1, c*d*t*B+c*d*t*C+c*d*t+c*d*A*B+c*d*A*C+c*d*A+c*d*B+c*d*C+c*d+c*t*C*I+c*C*I+d*t*B+d*t*C+d*t+d*A*B+d*A*C+d*A+d*B+d*C+d+g*J+g+n+t*C*I+A*C*I+A*I*J+A*I+C*I+J, c*d*t*v*H+c*d*t*H+c*d*v*H+c*d*H+d*t*v*H*J+d*t*v*H+d*t*H*J+d*t*H+d*v*H*J+d*v*H+d*H*J+d*H+g*v*H*J+g*v*H+g*H*J+g*H+n*v*H+n*H+v*H*J+H*J, c*d*t*v*D+c*d*t*D+c*v*D*I+c*D*I+d*t*v*D*E+d*t*v*D*J+d*t*v*D+d*t*D*E+d*t*D*J+d*t*D+d*v*D*E*J+d*v*D*E+d*D*E*J+d*D*E+v*A*D*I+v*B*D*I+v*C*D*J+v*C*D+v*D*E*J+v*D*E+v*D*J+v*D+A*D*F+B*D*I+C*D*J+C*D+D*E*J+D*E+D*F*J+D*I*J+D*J+D, c*d*t*v*C+c*d*t*C+c*d*v*C+c*d*C+c*t*C*I+c*C*I+d*t*v*E*J+d*t*v*E+d*t*v*J+d*t*v+d*t*E*J+d*t*E+d*t*J+d*t+d*v*E*J+d*v*E+d*v*J+d*v+d*E*J+d*E+d*J+d+g*v*J+g*v+n*v+t*v*I+t*C*J+t*C+t*E*I*J+t*E*I+t*E*J+t*E+t*F+t*I*J+t*I+t*J+t+v*A*I+v*B*E*I+v*B*I+v*C*J+v*C+v*E*J+v*E+v*I*J+v+A*F+B*E*I+B*I*J+B*I+E*F+E*I*J+F, c*d*t*v*z+c*d*t*z+c*d*v*z+c*d*z+d*t*v*z*J+d*t*v*z+d*t*z*J+d*t*z+d*v*z*J+d*v*z+d*z*J+d*z+g*v*J+g*v+g*J+g+n*v+n+v*J+J, c*d*t*v*x+c*d*t*x+c*v*I+c*v*J+c*v+c*F+c*J+c+d*t*v*x*E+d*t*v*x*J+d*t*v*x+d*t*v*E*J+d*t*v*E+d*t*x*E+d*t*x*J+d*t*x+d*t*E*J+d*t*E+d*v*x*E*J+d*v*x*E+d*v*E*J+d*v*E+d*x*E*J+d*x*E+d*E*J+d*E+v*A*I+v*B*E*I+v*B*I+v*E*I*J+v*I*J+v*I+v*J+v+A*F+B*E*I+B*I*J+B*I+E*F*J+E*F+E*I+I*J+I+J+1, c*d*s*B*D+d*s*D*J+d*s*D, c*d*s*v*H+c*d*s*H, c*d*s*v*D+c*d*s*D+d*s*v*D*J+d*s*v*D+d*s*D*J+d*s*D, c*d*s*v*B+c*d*s*v+c*d*s*B+c*d*s, c*d*s*v*z+c*d*s*z, c*d*s*v*x+c*d*s*x+c*x*I+c*F+c*J+c+d*s*v*x*J+d*s*v*x+d*s*x*J+d*s*x+g*J+g+n+t*C*J+t*C+t*E*J+t*E+t*F+t*J+t+x*I*J+x*I+E*F*J+F+1, c*d*s*t+c*d*s*B+c*t*I+c*I+d*s*t+d*s*B+g*J+g+n+t*I+B*I*J+F*J+I*J+I+J, c*d*r+c*r*I+d*r+r*I, c*d*g*v+c*d*g+c*v*J+c*v+c*J+c+d*g*v*E+d*g*E+d*n*v*E+d*n*v+d*n*E+d*n+d*v*E*J+d*v*J+d*E*J+d*J+v*A*I+v*B*E*I+v*B*I+v*C*J+v*C+v*E*J+v*E+v*I*J+v*I+A*F+B*E*I+B*I*J+B*I+C*J+C+E*F+E*I+E*J+E+I*J+I, b*J+r*J, b*F+b*I+r*F+r*I, b*D*E+r*D*E*J, b*C*I+b*C+b*E*I+b*E+b*I+b+r*C*I+r*C+r*E*I+r*E+r*I+r, b*C*H+b*E*H+b*H*I+b*H+r*E*H*J+r*H*I*J+r*H*J, b*C*E+b*E*I+r*C*E+r*E*I, b*B*I+r*B*I+v*B*I+B*I*J+B*I+F*J+F+I*J+I, b*B*H, b*B*G, b*B*E+b*B+r*B*E+r*B+s*E*J+s*E+s*J+s+B*E*J+B*E+B*J+B, b*B*D+r*B*D+r*D*E*J+r*D*E+s*D*E*J+s*D*E+s*D*J+s*D+B*D*J+B*D+D*E*J+D*E, b*B*C+r*B*E*J+r*B*E+r*B*J+r*B+r*C*J+s*E*J+s*E+s*J+s+B*E*J+B*E+B*J+B, b*A*K+r*A*E*K+r*A*J*K+r*A*K, b*A*I+r*A*F+r*A*I, b*A*H+r*H*J, b*A*E+b*A+r*A*B+r*A*E+r*B*J+r*J, b*A*D+r*A*B*D+r*A*D*E+r*B*D*J+r*D*E*J+r*D*J+s*B*D*E+s*D*E*J+s*D*E+B*C*D*E, b*A*C+r*A*B+r*A*C+r*A+r*B*J+r*J, b*A*B+b*A+r*B*J+r*J, b*z*I+b*I+r*z*I+r*I, b*z*C+b*z*E+b*z+b*C+b*E+b+r*z*E*J+r*z*J+r*C*J+r*E*J+r*J, b*z*B, b*z*A+r*z*J, b*y*C+b*y*E+b*y+b*C+b*E+b, b*x*I+b*I+r*x*I+r*I+v*B*E*I+B*E*I+E*F+E*I, b*x*E+b*E+r*x*E*J+r*E*J, b*x*C+b*x+b*C+b+r*x*J+r*C*E+r*E*F+r*J, b*x*B+b*B+r*x*B+r*x*E*J+r*x*E+r*B+r*E*J+r*E+s*x*E*J+s*x*E+s*x*J+s*x+s*E*J+s*E+s*J+s+x*B*J+x*B+x*E*J+x*E+A*E*I+B*J+B+E*J+E, b*x*A+b*A+r*x*A*B+r*x*A*E+r*x*E*J+r*x*J+r*A*B+r*A*E*J+r*A*E+r*B*E*J+r*J+s*x*B*E+s*x*B+s*x*E*J+s*x*E+s*x*J+s*x+s*B*E+s*B+s*E*J+s*E+s*J+s+x*B*C*E+x*B*C+x*B*J+x*B+x*E*J+x*E+A*E*I+B*C*E+B*C+B*J+B+E*J+E, b*v*I+b*I+r*F+r*I, b*v*C+b*v*E+b*v+b*C+b*E+b+r*v*C*J+r*v*E*J+r*v*J+r*C*J+r*E*J+r*J, b*t*C+b*t*E+b*t*I+b*t+r*t*F+r*t*I+r*B*E*J+r*B*J+r*C*J+s*t*E+s*t+s*B*E+s*B+t*B*E+t*B+t*C*E+t*C*J+t*E*F+t*E*J+t*J+A*B*C*E+A*B*C+A*B*E+A*B+A*C*E+A*C+A*E+A+B*E+B+C*E+C*J+E*F+E*J+J, b*s+r*s, b*r+b, b*n+r*J, b*m+m*s+r*s+s, b*k*I+b*I+k*r*I+r*I, b*k*E+b*E+r*B*E*J+r*E*J+s*t*E+s*t*J+s*t+s*B*E+s*J+s+t*B*E+t*B*J+t*B+t*C*E+t*E*F+t*E*J+A*B*C*E+A*B*E+A*C*E+A*E+B*E+B*J+B+C*E+E*F+E*J, b*k*C+b*k+b*C+b+r*B*J+r*J+s*t*J+s*B+s*J+s+t*B*J+t*C*J+t*J+A*B*C+A*B+A*C+A+B*J+C*J+J, b*k*B+b*B+k*r*B+k*s+k*B+k*C+k+n*r*E+n*s+r*B+r*E*J+s*t*J+s*E*J+s*E+s*J+t*B*J+t*C+t*E*J+t*E+t*F+t+x*F+z*B*J+z*B+A*F+B*E*J+B*E+B*I*J+B*I+C*J+C+E*F*J+I*J+I+J+1, b*k*A+b*A+k*r*A*B+n*r*A+r*A*J+r*A, b*i+r*B*J+r*J+s*t*J+s*B+s*J+s+t*B*J+t*C*J+t*J+A*B*C+A*B+A*C+A+B*J+C*J+J, b*h*q*B, b*h*q*A, b*g*H, b*g*E+b*g+b*C+b*E+b*I+b+r*C*J+r*E*J+r*F+r*I+r*J, b*g*D+b*C*D+b*D*I+b*D+r*C*D*J+r*D*F+r*D*I+r*D*J, b*g*C+b*g, b*g*B, b*g*z+b*g, b*g*y+b*g, b*g*x+b*g, b*g*v+b*g, b*f, b*d*C+b*d*E+b*d+b*C+b*E+b+r*C*I+r*C+r*E*I+r*E+r*I+r, b*d*v*A+b*d*A+d*v*A*B*J+d*v*A*B+d*A*B*J+d*A*B+r*A*F+r*A*I+A*F+A*I*J+A*I, b*d*g+b*g, b*c+b*g+b+c*r+c*v*J+c*v+c*J+c+g*r+g*v*J+g*v+g*J+g+n*r+n*v+n+r*A*F+r*J+r+v*A*I+v+A*F+1}

II1 = ideal(i*r*B+i*s*C+s*B*C+i*s+s*B+B*C+s,b*f+b*o,c*F*J+E*F*J+c*E+C*J+F*J+c+C+F,k*o*p,k*z*F+k*z*G+F*G+F+G,e*f*x*J+e*f*t+e*t*x+e*x+e*J,f*m*p+m*p*y+f*m*J+J,i*t*A*B+i*t*A*G+t*A*B*G+t*A*B+A*B*G+i*B+i+A,e*y*z+e*y,h*k*D*I+h*k*E*I+k*D*E+k*D*I+D*I,f*m*v*I+d*v*I+d*f+d*m+I,e*x+k*x+k*E+n*E+k+E,e*z+e,o*y,e,i*y*F*K+e*i*F+i*y*F+e*i*K+e*y*K+e*y+i*F+i*K,d*e*i*u+u,d*f*s*C,i*k*n*t+i*k*n*J+i*n*t+k*n*t+i*t*J+i+J,e*s*F+s*u*H,h*q*r*B*H+h*q*r*B+h*B*H,d*v*A*E+d*e*v+e*v*A+d*A*E+e*A*E+e*v,b*r*B*H+b*r*G*H+b*r*B+r*B*G+b*G*H+b*B+B*G+G*H,b*g*m*r*s+b*g*s+b*r*s+b*r+m*r+m*s+b+s,h*m*u*B+h*u*H+m*B*H+h*m+u*H,f*g*n+c*n+c+g+n+1,e*m*o*r*I+e*m*o*r+e*m*r*I+e*m+o,x*A*K,c*g*z*B+c*g*B+g*z*B+z*B+g*C+g,i*n*t*J+i*n*t+g*t+t*J+i+n,j+r+1,s*v*y*I+s*y*I+y*F*I+F*I+F+I,e*h*m*q*v+e*h*m*q+e*h*q+e*m*v+h*m*v+e,k*r*A*G+r*x*G+G,b*e*H*I+b*g*H*I+e*H*I,E*F*G*H)

  GG = sort flatten entries gens gbBoolean II1
  assert(GG == correctLexBasis)
///
end

uninstallPackage "BooleanGB"

viewHelp gbBool

restart
installPackage "BooleanGB"
check "BooleanGB"



end -- test file disabled by Dan; these tests should be moved to the package BooleanGB

restart
R=ZZ/2[a1,a2,a3,a4,b1,b2,b3,b4,c1,c2,c3,c4,d1,d2,d3,d4,e1,e2,e3,e4,f1,f2,f3,f4,g1,g2,g3,g4,h1,h2,h3,h4,i1,i2,i3,i4,j1,j2,j3,j4,k1,k2,k3,k4,l1,l2,l3,l4,m1,m2,m3,m4,n1,n2,n3,n4,o1,o2,o3,o4,p1,p2,p3,p4, MonomialOrder=>Lex];
l =  apply( gens R, x -> x^2+x);
RQ = R/l
I1= ideal {a1+a2+a3+a4-1, b1+b2+b3+b4-1, c1+c2+c3+c4-1, d1+d2+d3+d4-1, e1+e2+e3+e4-1, f1+f2+f3+f4-1, g1+g2+g3+g4-1, h1+h2+h3+h4-1, i1+i2+i3+i4-1, j1+j2+j3+j4-1, k1+k2+k3+k4-1, l1+l2+l3+l4-1, m1+m2+m3+m4-1, n1+n2+n3+n4-1, o1+o2+o3+o4-1, p1+p2+p3+p4-1, a1*b1+a2*b2+a3*b3+a4*b4, a1*c1+a2*c2+a3*c3+a4*c4, a1*d1+a2*d2+a3*d3+a4*d4, a1*e1+a2*e2+a3*e3+a4*e4, a1*f1+a2*f2+a3*f3+a4*f4, a1*i1+a2*i2+a3*i3+a4*i4, a1*m1+a2*m2+a3*m3+a4*m4, b1*c1+b2*c2+b3*c3+b4*c4, b1*d1+b2*d2+b3*d3+b4*d4, b1*e1+b2*e2+b3*e3+b4*e4, b1*f1+b2*f2+b3*f3+b4*f4, b1*j1+b2*j2+b3*j3+b4*j4, b1*n1+b2*n2+b3*n3+b4*n4, c1*d1+c2*d2+c3*d3+c4*d4, c1*g1+c2*g2+c3*g3+c4*g4, c1*h1+c2*h2+c3*h3+c4*h4, c1*k1+c2*k2+c3*k3+c4*k4, c1*o1+c2*o2+c3*o3+c4*o4, d1*g1+d2*g2+d3*g3+d4*g4, d1*h1+d2*h2+d3*h3+d4*h4, d1*l1+d2*l2+d3*l3+d4*l4, d1*p1+d2*p2+d3*p3+d4*p4, e1*f1+e2*f2+e3*f3+e4*f4, e1*g1+e2*g2+e3*g3+e4*g4, e1*h1+e2*h2+e3*h3+e4*h4, e1*i1+e2*i2+e3*i3+e4*i4, e1*m1+e2*m2+e3*m3+e4*m4, f1*g1+f2*g2+f3*g3+f4*g4, f1*h1+f2*h2+f3*h3+f4*h4, f1*j1+f2*j2+f3*j3+f4*j4, f1*n1+f2*n2+f3*n3+f4*n4, g1*h1+g2*h2+g3*h3+g4*h4, g1*k1+g2*k2+g3*k3+g4*k4, g1*o1+g2*o2+g3*o3+g4*o4, h1*l1+h2*l2+h3*l3+h4*l4, h1*p1+h2*p2+h3*p3+h4*p4, i1*j1+i2*j2+i3*j3+i4*j4, i1*k1+i2*k2+i3*k3+i4*k4, i1*l1+i2*l2+i3*l3+i4*l4, i1*m1+i2*m2+i3*m3+i4*m4, i1*n1+i2*n2+i3*n3+i4*n4, j1*k1+j2*k2+j3*k3+j4*k4, j1*l1+j2*l2+j3*l3+j4*l4, j1*m1+j2*m2+j3*m3+j4*m4, j1*n1+j2*n2+j3*n3+j4*n4, k1*l1+k2*l2+k3*l3+k4*l4, k1*o1+k2*o2+k3*o3+k4*o4, k1*p1+k2*p2+k3*p3+k4*p4, l1*o1+l2*o2+l3*o3+l4*o4, l1*p1+l2*p2+l3*p3+l4*p4, m1*n1+m2*n2+m3*n3+m4*n4, m1*o1+m2*o2+m3*o3+m4*o4, m1*p1+m2*p2+m3*p3+m4*p4, n1*o1+n2*o2+n3*o3+n4*o4, n1*p1+n2*p2+n3*p3+n4*p4, o1*p1+o2*p2+o3*p3+o4*p4}
I2 = ideal {d1, d2, d3, d4-1, e1, e2, e3, e4-1, g1, g2-1, g3, g4, j1, j2, j3-1, j4, l1-1, l2, l3, l4, m1-1, m2, m3, m4}
I = I1+I2
I = I1
--time C = gens gb I;
time B = gbBoolean I;
end
--B
--assert( sort gens B - sort C == 0 )


end -- test file disabled by Dan; these tests should be moved to the package BooleanGB

-- Example: 16 node Conway game of life, on a torus
--  load "~/2010March/Macaulay2/packages/Macaulay2Doc/test/franzi-brp-cgl.m2"
R = ZZ/2[vars(0..15), MonomialOrder=>Lex]
l = apply(gens R, x-> x^2+x);
QR = R/l;

L = {a*b*d*e*f*h*m*n + a*b*d*e*f*h*m*p + a*b*d*e*f*h*m + a*b*d*e*f*h*n*p +
a*b*d*e*f*h*n + a*b*d*e*f*h*p + a*b*d*e*f*m*n*p + a*b*d*e*f*m*n +
a*b*d*e*f*m*p + a*b*d*e*f*n*p + a*b*d*e*h*m*n*p + a*b*d*e*h*m*n +
a*b*d*e*h*m*p + a*b*d*e*h*n*p + a*b*d*e*m*n*p + a*b*d*e + a*b*d*f*h*m*n*p +
a*b*d*f*h*m*n + a*b*d*f*h*m*p + a*b*d*f*h*n*p + a*b*d*f*m*n*p + a*b*d*f +
a*b*d*h*m*n*p + a*b*d*h + a*b*d*m + a*b*d*n + a*b*d*p + a*b*d +
a*b*e*f*h*m*n*p + a*b*e*f*h*m*n + a*b*e*f*h*m*p + a*b*e*f*h*n*p +
a*b*e*f*m*n*p + a*b*e*f + a*b*e*h*m*n*p + a*b*e*h + a*b*e*m + a*b*e*n +
a*b*e*p + a*b*e + a*b*f*h*m*n*p + a*b*f*h + a*b*f*m + a*b*f*n + a*b*f*p +
a*b*f + a*b*h*m + a*b*h*n + a*b*h*p + a*b*h + a*b*m*n + a*b*m*p + a*b*m +
a*b*n*p + a*b*n + a*b*p + a*d*e*f*h*m*n*p + a*d*e*f*h*m*n + a*d*e*f*h*m*p +
a*d*e*f*h*n*p + a*d*e*f*m*n*p + a*d*e*f + a*d*e*h*m*n*p + a*d*e*h + a*d*e*m +
a*d*e*n + a*d*e*p + a*d*e + a*d*f*h*m*n*p + a*d*f*h + a*d*f*m + a*d*f*n +
a*d*f*p + a*d*f + a*d*h*m + a*d*h*n + a*d*h*p + a*d*h + a*d*m*n + a*d*m*p +
a*d*m + a*d*n*p + a*d*n + a*d*p + a*e*f*h*m*n*p + a*e*f*h + a*e*f*m + a*e*f*n
+ a*e*f*p + a*e*f + a*e*h*m + a*e*h*n + a*e*h*p + a*e*h + a*e*m*n + a*e*m*p +
a*e*m + a*e*n*p + a*e*n + a*e*p + a*f*h*m + a*f*h*n + a*f*h*p + a*f*h +
a*f*m*n + a*f*m*p + a*f*m + a*f*n*p + a*f*n + a*f*p + a*h*m*n + a*h*m*p +
a*h*m + a*h*n*p + a*h*n + a*h*p + a*m*n*p + a*m*n + a*m*p + a*n*p +
b*d*e*f*h*m*n + b*d*e*f*h*m*p + b*d*e*f*h*n*p + b*d*e*f*m*n*p + b*d*e*h*m*n*p
+ b*d*e + b*d*f*h*m*n*p + b*d*f + b*d*h + b*d*m + b*d*n + b*d*p +
b*e*f*h*m*n*p + b*e*f + b*e*h + b*e*m + b*e*n + b*e*p + b*f*h + b*f*m + b*f*n
+ b*f*p + b*h*m + b*h*n + b*h*p + b*m*n + b*m*p + b*n*p + d*e*f*h*m*n*p +
d*e*f + d*e*h + d*e*m + d*e*n + d*e*p + d*f*h + d*f*m + d*f*n + d*f*p + d*h*m
+ d*h*n + d*h*p + d*m*n + d*m*p + d*n*p + e*f*h + e*f*m + e*f*n + e*f*p +
e*h*m + e*h*n + e*h*p + e*m*n + e*m*p + e*n*p + f*h*m + f*h*n + f*h*p + f*m*n
+ f*m*p + f*n*p + h*m*n + h*m*p + h*n*p + m*n*p, a*b*c*e*f*g*m*n +
a*b*c*e*f*g*m*o + a*b*c*e*f*g*m + a*b*c*e*f*g*n*o + a*b*c*e*f*g*n +
a*b*c*e*f*g*o + a*b*c*e*f*m*n*o + a*b*c*e*f*m*n + a*b*c*e*f*m*o +
a*b*c*e*f*n*o + a*b*c*e*g*m*n*o + a*b*c*e*g*m*n + a*b*c*e*g*m*o +
a*b*c*e*g*n*o + a*b*c*e*m*n*o + a*b*c*e + a*b*c*f*g*m*n*o + a*b*c*f*g*m*n +
a*b*c*f*g*m*o + a*b*c*f*g*n*o + a*b*c*f*m*n*o + a*b*c*f + a*b*c*g*m*n*o +
a*b*c*g + a*b*c*m + a*b*c*n + a*b*c*o + a*b*c + a*b*e*f*g*m*n*o +
a*b*e*f*g*m*n + a*b*e*f*g*m*o + a*b*e*f*g*n*o + a*b*e*f*m*n*o + a*b*e*f +
a*b*e*g*m*n*o + a*b*e*g + a*b*e*m + a*b*e*n + a*b*e*o + a*b*e + a*b*f*g*m*n*o
+ a*b*f*g + a*b*f*m + a*b*f*n + a*b*f*o + a*b*f + a*b*g*m + a*b*g*n + a*b*g*o
+ a*b*g + a*b*m*n + a*b*m*o + a*b*m + a*b*n*o + a*b*n + a*b*o + a*c*e*f*g*m*n
+ a*c*e*f*g*m*o + a*c*e*f*g*n*o + a*c*e*f*m*n*o + a*c*e*g*m*n*o + a*c*e +
a*c*f*g*m*n*o + a*c*f + a*c*g + a*c*m + a*c*n + a*c*o + a*e*f*g*m*n*o + a*e*f
+ a*e*g + a*e*m + a*e*n + a*e*o + a*f*g + a*f*m + a*f*n + a*f*o + a*g*m +
a*g*n + a*g*o + a*m*n + a*m*o + a*n*o + b*c*e*f*g*m*n*o + b*c*e*f*g*m*n +
b*c*e*f*g*m*o + b*c*e*f*g*n*o + b*c*e*f*m*n*o + b*c*e*f + b*c*e*g*m*n*o +
b*c*e*g + b*c*e*m + b*c*e*n + b*c*e*o + b*c*e + b*c*f*g*m*n*o + b*c*f*g +
b*c*f*m + b*c*f*n + b*c*f*o + b*c*f + b*c*g*m + b*c*g*n + b*c*g*o + b*c*g +
b*c*m*n + b*c*m*o + b*c*m + b*c*n*o + b*c*n + b*c*o + b*e*f*g*m*n*o + b*e*f*g
+ b*e*f*m + b*e*f*n + b*e*f*o + b*e*f + b*e*g*m + b*e*g*n + b*e*g*o + b*e*g +
b*e*m*n + b*e*m*o + b*e*m + b*e*n*o + b*e*n + b*e*o + b*f*g*m + b*f*g*n +
b*f*g*o + b*f*g + b*f*m*n + b*f*m*o + b*f*m + b*f*n*o + b*f*n + b*f*o +
b*g*m*n + b*g*m*o + b*g*m + b*g*n*o + b*g*n + b*g*o + b*m*n*o + b*m*n + b*m*o
+ b*n*o + c*e*f*g*m*n*o + c*e*f + c*e*g + c*e*m + c*e*n + c*e*o + c*f*g +
c*f*m + c*f*n + c*f*o + c*g*m + c*g*n + c*g*o + c*m*n + c*m*o + c*n*o + e*f*g
+ e*f*m + e*f*n + e*f*o + e*g*m + e*g*n + e*g*o + e*m*n + e*m*o + e*n*o +
f*g*m + f*g*n + f*g*o + f*m*n + f*m*o + f*n*o + g*m*n + g*m*o + g*n*o + m*n*o,
b*c*d*f*g*h*n*o + b*c*d*f*g*h*n*p + b*c*d*f*g*h*n + b*c*d*f*g*h*o*p +
b*c*d*f*g*h*o + b*c*d*f*g*h*p + b*c*d*f*g*n*o*p + b*c*d*f*g*n*o +
b*c*d*f*g*n*p + b*c*d*f*g*o*p + b*c*d*f*h*n*o*p + b*c*d*f*h*n*o +
b*c*d*f*h*n*p + b*c*d*f*h*o*p + b*c*d*f*n*o*p + b*c*d*f + b*c*d*g*h*n*o*p +
b*c*d*g*h*n*o + b*c*d*g*h*n*p + b*c*d*g*h*o*p + b*c*d*g*n*o*p + b*c*d*g +
b*c*d*h*n*o*p + b*c*d*h + b*c*d*n + b*c*d*o + b*c*d*p + b*c*d +
b*c*f*g*h*n*o*p + b*c*f*g*h*n*o + b*c*f*g*h*n*p + b*c*f*g*h*o*p +
b*c*f*g*n*o*p + b*c*f*g + b*c*f*h*n*o*p + b*c*f*h + b*c*f*n + b*c*f*o +
b*c*f*p + b*c*f + b*c*g*h*n*o*p + b*c*g*h + b*c*g*n + b*c*g*o + b*c*g*p +
b*c*g + b*c*h*n + b*c*h*o + b*c*h*p + b*c*h + b*c*n*o + b*c*n*p + b*c*n +
b*c*o*p + b*c*o + b*c*p + b*d*f*g*h*n*o + b*d*f*g*h*n*p + b*d*f*g*h*o*p +
b*d*f*g*n*o*p + b*d*f*h*n*o*p + b*d*f + b*d*g*h*n*o*p + b*d*g + b*d*h + b*d*n
+ b*d*o + b*d*p + b*f*g*h*n*o*p + b*f*g + b*f*h + b*f*n + b*f*o + b*f*p +
b*g*h + b*g*n + b*g*o + b*g*p + b*h*n + b*h*o + b*h*p + b*n*o + b*n*p + b*o*p
+ c*d*f*g*h*n*o*p + c*d*f*g*h*n*o + c*d*f*g*h*n*p + c*d*f*g*h*o*p +
c*d*f*g*n*o*p + c*d*f*g + c*d*f*h*n*o*p + c*d*f*h + c*d*f*n + c*d*f*o +
c*d*f*p + c*d*f + c*d*g*h*n*o*p + c*d*g*h + c*d*g*n + c*d*g*o + c*d*g*p +
c*d*g + c*d*h*n + c*d*h*o + c*d*h*p + c*d*h + c*d*n*o + c*d*n*p + c*d*n +
c*d*o*p + c*d*o + c*d*p + c*f*g*h*n*o*p + c*f*g*h + c*f*g*n + c*f*g*o +
c*f*g*p + c*f*g + c*f*h*n + c*f*h*o + c*f*h*p + c*f*h + c*f*n*o + c*f*n*p +
c*f*n + c*f*o*p + c*f*o + c*f*p + c*g*h*n + c*g*h*o + c*g*h*p + c*g*h +
c*g*n*o + c*g*n*p + c*g*n + c*g*o*p + c*g*o + c*g*p + c*h*n*o + c*h*n*p +
c*h*n + c*h*o*p + c*h*o + c*h*p + c*n*o*p + c*n*o + c*n*p + c*o*p +
d*f*g*h*n*o*p + d*f*g + d*f*h + d*f*n + d*f*o + d*f*p + d*g*h + d*g*n + d*g*o
+ d*g*p + d*h*n + d*h*o + d*h*p + d*n*o + d*n*p + d*o*p + f*g*h + f*g*n +
f*g*o + f*g*p + f*h*n + f*h*o + f*h*p + f*n*o + f*n*p + f*o*p + g*h*n + g*h*o
+ g*h*p + g*n*o + g*n*p + g*o*p + h*n*o + h*n*p + h*o*p + n*o*p,
a*c*d*e*g*h*m*o + a*c*d*e*g*h*m*p + a*c*d*e*g*h*m + a*c*d*e*g*h*o*p +
a*c*d*e*g*h*o + a*c*d*e*g*h*p + a*c*d*e*g*m*o*p + a*c*d*e*g*m*o +
a*c*d*e*g*m*p + a*c*d*e*g*o*p + a*c*d*e*h*m*o*p + a*c*d*e*h*m*o +
a*c*d*e*h*m*p + a*c*d*e*h*o*p + a*c*d*e*m*o*p + a*c*d*e + a*c*d*g*h*m*o*p +
a*c*d*g*h*m*o + a*c*d*g*h*m*p + a*c*d*g*h*o*p + a*c*d*g*m*o*p + a*c*d*g +
a*c*d*h*m*o*p + a*c*d*h + a*c*d*m + a*c*d*o + a*c*d*p + a*c*d + a*c*e*g*h*m*o
+ a*c*e*g*h*m*p + a*c*e*g*h*o*p + a*c*e*g*m*o*p + a*c*e*h*m*o*p + a*c*e +
a*c*g*h*m*o*p + a*c*g + a*c*h + a*c*m + a*c*o + a*c*p + a*d*e*g*h*m*o*p +
a*d*e*g*h*m*o + a*d*e*g*h*m*p + a*d*e*g*h*o*p + a*d*e*g*m*o*p + a*d*e*g +
a*d*e*h*m*o*p + a*d*e*h + a*d*e*m + a*d*e*o + a*d*e*p + a*d*e + a*d*g*h*m*o*p
+ a*d*g*h + a*d*g*m + a*d*g*o + a*d*g*p + a*d*g + a*d*h*m + a*d*h*o + a*d*h*p
+ a*d*h + a*d*m*o + a*d*m*p + a*d*m + a*d*o*p + a*d*o + a*d*p + a*e*g*h*m*o*p
+ a*e*g + a*e*h + a*e*m + a*e*o + a*e*p + a*g*h + a*g*m + a*g*o + a*g*p +
a*h*m + a*h*o + a*h*p + a*m*o + a*m*p + a*o*p + c*d*e*g*h*m*o*p +
c*d*e*g*h*m*o + c*d*e*g*h*m*p + c*d*e*g*h*o*p + c*d*e*g*m*o*p + c*d*e*g +
c*d*e*h*m*o*p + c*d*e*h + c*d*e*m + c*d*e*o + c*d*e*p + c*d*e + c*d*g*h*m*o*p
+ c*d*g*h + c*d*g*m + c*d*g*o + c*d*g*p + c*d*g + c*d*h*m + c*d*h*o + c*d*h*p
+ c*d*h + c*d*m*o + c*d*m*p + c*d*m + c*d*o*p + c*d*o + c*d*p + c*e*g*h*m*o*p
+ c*e*g + c*e*h + c*e*m + c*e*o + c*e*p + c*g*h + c*g*m + c*g*o + c*g*p +
c*h*m + c*h*o + c*h*p + c*m*o + c*m*p + c*o*p + d*e*g*h*m*o*p + d*e*g*h +
d*e*g*m + d*e*g*o + d*e*g*p + d*e*g + d*e*h*m + d*e*h*o + d*e*h*p + d*e*h +
d*e*m*o + d*e*m*p + d*e*m + d*e*o*p + d*e*o + d*e*p + d*g*h*m + d*g*h*o +
d*g*h*p + d*g*h + d*g*m*o + d*g*m*p + d*g*m + d*g*o*p + d*g*o + d*g*p +
d*h*m*o + d*h*m*p + d*h*m + d*h*o*p + d*h*o + d*h*p + d*m*o*p + d*m*o + d*m*p
+ d*o*p + e*g*h + e*g*m + e*g*o + e*g*p + e*h*m + e*h*o + e*h*p + e*m*o +
e*m*p + e*o*p + g*h*m + g*h*o + g*h*p + g*m*o + g*m*p + g*o*p + h*m*o + h*m*p
+ h*o*p + m*o*p, a*b*d*e*f*h*i*j + a*b*d*e*f*h*i*l + a*b*d*e*f*h*i +
a*b*d*e*f*h*j*l + a*b*d*e*f*h*j + a*b*d*e*f*h*l + a*b*d*e*f*i*j*l +
a*b*d*e*f*i*j + a*b*d*e*f*i*l + a*b*d*e*f*j*l + a*b*d*e*h*i*j*l +
a*b*d*e*h*i*j + a*b*d*e*h*i*l + a*b*d*e*h*j*l + a*b*d*e*i*j*l + a*b*d*e +
a*b*d*f*h*i*j + a*b*d*f*h*i*l + a*b*d*f*h*j*l + a*b*d*f*i*j*l + a*b*d*h*i*j*l
+ a*b*d + a*b*e*f*h*i*j*l + a*b*e*f*h*i*j + a*b*e*f*h*i*l + a*b*e*f*h*j*l +
a*b*e*f*i*j*l + a*b*e*f + a*b*e*h*i*j*l + a*b*e*h + a*b*e*i + a*b*e*j +
a*b*e*l + a*b*e + a*b*f*h*i*j*l + a*b*f + a*b*h + a*b*i + a*b*j + a*b*l +
a*d*e*f*h*i*j*l + a*d*e*f*h*i*j + a*d*e*f*h*i*l + a*d*e*f*h*j*l +
a*d*e*f*i*j*l + a*d*e*f + a*d*e*h*i*j*l + a*d*e*h + a*d*e*i + a*d*e*j +
a*d*e*l + a*d*e + a*d*f*h*i*j*l + a*d*f + a*d*h + a*d*i + a*d*j + a*d*l +
a*e*f*h*i*j*l + a*e*f*h + a*e*f*i + a*e*f*j + a*e*f*l + a*e*f + a*e*h*i +
a*e*h*j + a*e*h*l + a*e*h + a*e*i*j + a*e*i*l + a*e*i + a*e*j*l + a*e*j +
a*e*l + a*f*h + a*f*i + a*f*j + a*f*l + a*h*i + a*h*j + a*h*l + a*i*j + a*i*l
+ a*j*l + b*d*e*f*h*i*j*l + b*d*e*f*h*i*j + b*d*e*f*h*i*l + b*d*e*f*h*j*l +
b*d*e*f*i*j*l + b*d*e*f + b*d*e*h*i*j*l + b*d*e*h + b*d*e*i + b*d*e*j +
b*d*e*l + b*d*e + b*d*f*h*i*j*l + b*d*f + b*d*h + b*d*i + b*d*j + b*d*l +
b*e*f*h*i*j*l + b*e*f*h + b*e*f*i + b*e*f*j + b*e*f*l + b*e*f + b*e*h*i +
b*e*h*j + b*e*h*l + b*e*h + b*e*i*j + b*e*i*l + b*e*i + b*e*j*l + b*e*j +
b*e*l + b*f*h + b*f*i + b*f*j + b*f*l + b*h*i + b*h*j + b*h*l + b*i*j + b*i*l
+ b*j*l + d*e*f*h*i*j*l + d*e*f*h + d*e*f*i + d*e*f*j + d*e*f*l + d*e*f +
d*e*h*i + d*e*h*j + d*e*h*l + d*e*h + d*e*i*j + d*e*i*l + d*e*i + d*e*j*l +
d*e*j + d*e*l + d*f*h + d*f*i + d*f*j + d*f*l + d*h*i + d*h*j + d*h*l + d*i*j
+ d*i*l + d*j*l + e*f*h*i + e*f*h*j + e*f*h*l + e*f*h + e*f*i*j + e*f*i*l +
e*f*i + e*f*j*l + e*f*j + e*f*l + e*h*i*j + e*h*i*l + e*h*i + e*h*j*l + e*h*j
+ e*h*l + e*i*j*l + e*i*j + e*i*l + e*j*l + f*h*i + f*h*j + f*h*l + f*i*j +
f*i*l + f*j*l + h*i*j + h*i*l + h*j*l + i*j*l, a*b*c*e*f*g*i*j +
a*b*c*e*f*g*i*k + a*b*c*e*f*g*i + a*b*c*e*f*g*j*k + a*b*c*e*f*g*j +
a*b*c*e*f*g*k + a*b*c*e*f*i*j*k + a*b*c*e*f*i*j + a*b*c*e*f*i*k +
a*b*c*e*f*j*k + a*b*c*e*g*i*j + a*b*c*e*g*i*k + a*b*c*e*g*j*k + a*b*c*e*i*j*k
+ a*b*c*f*g*i*j*k + a*b*c*f*g*i*j + a*b*c*f*g*i*k + a*b*c*f*g*j*k +
a*b*c*f*i*j*k + a*b*c*f + a*b*c*g*i*j*k + a*b*c + a*b*e*f*g*i*j*k +
a*b*e*f*g*i*j + a*b*e*f*g*i*k + a*b*e*f*g*j*k + a*b*e*f*i*j*k + a*b*e*f +
a*b*e*g*i*j*k + a*b*e + a*b*f*g*i*j*k + a*b*f*g + a*b*f*i + a*b*f*j + a*b*f*k
+ a*b*f + a*b*g + a*b*i + a*b*j + a*b*k + a*c*e*f*g*i*j*k + a*c*e*f*g*i*j +
a*c*e*f*g*i*k + a*c*e*f*g*j*k + a*c*e*f*i*j*k + a*c*e*f + a*c*e*g*i*j*k +
a*c*e + a*c*f*g*i*j*k + a*c*f*g + a*c*f*i + a*c*f*j + a*c*f*k + a*c*f + a*c*g
+ a*c*i + a*c*j + a*c*k + a*e*f*g*i*j*k + a*e*f*g + a*e*f*i + a*e*f*j +
a*e*f*k + a*e*f + a*e*g + a*e*i + a*e*j + a*e*k + a*f*g*i + a*f*g*j + a*f*g*k
+ a*f*g + a*f*i*j + a*f*i*k + a*f*i + a*f*j*k + a*f*j + a*f*k + a*g*i + a*g*j
+ a*g*k + a*i*j + a*i*k + a*j*k + b*c*e*f*g*i*j*k + b*c*e*f*g*i*j +
b*c*e*f*g*i*k + b*c*e*f*g*j*k + b*c*e*f*i*j*k + b*c*e*f + b*c*e*g*i*j*k +
b*c*e + b*c*f*g*i*j*k + b*c*f*g + b*c*f*i + b*c*f*j + b*c*f*k + b*c*f + b*c*g
+ b*c*i + b*c*j + b*c*k + b*e*f*g*i*j*k + b*e*f*g + b*e*f*i + b*e*f*j +
b*e*f*k + b*e*f + b*e*g + b*e*i + b*e*j + b*e*k + b*f*g*i + b*f*g*j + b*f*g*k
+ b*f*g + b*f*i*j + b*f*i*k + b*f*i + b*f*j*k + b*f*j + b*f*k + b*g*i + b*g*j
+ b*g*k + b*i*j + b*i*k + b*j*k + c*e*f*g*i*j*k + c*e*f*g + c*e*f*i + c*e*f*j
+ c*e*f*k + c*e*f + c*e*g + c*e*i + c*e*j + c*e*k + c*f*g*i + c*f*g*j +
c*f*g*k + c*f*g + c*f*i*j + c*f*i*k + c*f*i + c*f*j*k + c*f*j + c*f*k + c*g*i
+ c*g*j + c*g*k + c*i*j + c*i*k + c*j*k + e*f*g*i + e*f*g*j + e*f*g*k + e*f*g
+ e*f*i*j + e*f*i*k + e*f*i + e*f*j*k + e*f*j + e*f*k + e*g*i + e*g*j + e*g*k
+ e*i*j + e*i*k + e*j*k + f*g*i*j + f*g*i*k + f*g*i + f*g*j*k + f*g*j + f*g*k
+ f*i*j*k + f*i*j + f*i*k + f*j*k + g*i*j + g*i*k + g*j*k + i*j*k,
b*c*d*f*g*h*j*k + b*c*d*f*g*h*j*l + b*c*d*f*g*h*j + b*c*d*f*g*h*k*l +
b*c*d*f*g*h*k + b*c*d*f*g*h*l + b*c*d*f*g*j*k*l + b*c*d*f*g*j*k +
b*c*d*f*g*j*l + b*c*d*f*g*k*l + b*c*d*f*h*j*k + b*c*d*f*h*j*l + b*c*d*f*h*k*l
+ b*c*d*f*j*k*l + b*c*d*g*h*j*k*l + b*c*d*g*h*j*k + b*c*d*g*h*j*l +
b*c*d*g*h*k*l + b*c*d*g*j*k*l + b*c*d*g + b*c*d*h*j*k*l + b*c*d +
b*c*f*g*h*j*k*l + b*c*f*g*h*j*k + b*c*f*g*h*j*l + b*c*f*g*h*k*l +
b*c*f*g*j*k*l + b*c*f*g + b*c*f*h*j*k*l + b*c*f + b*c*g*h*j*k*l + b*c*g*h +
b*c*g*j + b*c*g*k + b*c*g*l + b*c*g + b*c*h + b*c*j + b*c*k + b*c*l +
b*d*f*g*h*j*k*l + b*d*f*g*h*j*k + b*d*f*g*h*j*l + b*d*f*g*h*k*l +
b*d*f*g*j*k*l + b*d*f*g + b*d*f*h*j*k*l + b*d*f + b*d*g*h*j*k*l + b*d*g*h +
b*d*g*j + b*d*g*k + b*d*g*l + b*d*g + b*d*h + b*d*j + b*d*k + b*d*l +
b*f*g*h*j*k*l + b*f*g*h + b*f*g*j + b*f*g*k + b*f*g*l + b*f*g + b*f*h + b*f*j
+ b*f*k + b*f*l + b*g*h*j + b*g*h*k + b*g*h*l + b*g*h + b*g*j*k + b*g*j*l +
b*g*j + b*g*k*l + b*g*k + b*g*l + b*h*j + b*h*k + b*h*l + b*j*k + b*j*l +
b*k*l + c*d*f*g*h*j*k*l + c*d*f*g*h*j*k + c*d*f*g*h*j*l + c*d*f*g*h*k*l +
c*d*f*g*j*k*l + c*d*f*g + c*d*f*h*j*k*l + c*d*f + c*d*g*h*j*k*l + c*d*g*h +
c*d*g*j + c*d*g*k + c*d*g*l + c*d*g + c*d*h + c*d*j + c*d*k + c*d*l +
c*f*g*h*j*k*l + c*f*g*h + c*f*g*j + c*f*g*k + c*f*g*l + c*f*g + c*f*h + c*f*j
+ c*f*k + c*f*l + c*g*h*j + c*g*h*k + c*g*h*l + c*g*h + c*g*j*k + c*g*j*l +
c*g*j + c*g*k*l + c*g*k + c*g*l + c*h*j + c*h*k + c*h*l + c*j*k + c*j*l +
c*k*l + d*f*g*h*j*k*l + d*f*g*h + d*f*g*j + d*f*g*k + d*f*g*l + d*f*g + d*f*h
+ d*f*j + d*f*k + d*f*l + d*g*h*j + d*g*h*k + d*g*h*l + d*g*h + d*g*j*k +
d*g*j*l + d*g*j + d*g*k*l + d*g*k + d*g*l + d*h*j + d*h*k + d*h*l + d*j*k +
d*j*l + d*k*l + f*g*h*j + f*g*h*k + f*g*h*l + f*g*h + f*g*j*k + f*g*j*l +
f*g*j + f*g*k*l + f*g*k + f*g*l + f*h*j + f*h*k + f*h*l + f*j*k + f*j*l +
f*k*l + g*h*j*k + g*h*j*l + g*h*j + g*h*k*l + g*h*k + g*h*l + g*j*k*l + g*j*k
+ g*j*l + g*k*l + h*j*k + h*j*l + h*k*l + j*k*l + 1, a*c*d*e*g*h*i*k +
a*c*d*e*g*h*i*l + a*c*d*e*g*h*i + a*c*d*e*g*h*k*l + a*c*d*e*g*h*k +
a*c*d*e*g*h*l + a*c*d*e*g*i*k + a*c*d*e*g*i*l + a*c*d*e*g*k*l +
a*c*d*e*h*i*k*l + a*c*d*e*h*i*k + a*c*d*e*h*i*l + a*c*d*e*h*k*l +
a*c*d*e*i*k*l + a*c*d*g*h*i*k*l + a*c*d*g*h*i*k + a*c*d*g*h*i*l +
a*c*d*g*h*k*l + a*c*d*g*i*k*l + a*c*d*h*i*k*l + a*c*d*h + a*c*d +
a*c*e*g*h*i*k*l + a*c*e*g*h*i*k + a*c*e*g*h*i*l + a*c*e*g*h*k*l +
a*c*e*g*i*k*l + a*c*e*h*i*k*l + a*c*e*h + a*c*e + a*c*g*h*i*k*l + a*c*g*h +
a*c*g + a*c*h*i + a*c*h*k + a*c*h*l + a*c*h + a*c*i + a*c*k + a*c*l +
a*d*e*g*h*i*k*l + a*d*e*g*h*i*k + a*d*e*g*h*i*l + a*d*e*g*h*k*l +
a*d*e*g*i*k*l + a*d*e*h*i*k*l + a*d*e*h + a*d*e + a*d*g*h*i*k*l + a*d*g*h +
a*d*g + a*d*h*i + a*d*h*k + a*d*h*l + a*d*h + a*d*i + a*d*k + a*d*l +
a*e*g*h*i*k*l + a*e*g*h + a*e*g + a*e*h*i + a*e*h*k + a*e*h*l + a*e*h + a*e*i
+ a*e*k + a*e*l + a*g*h*i + a*g*h*k + a*g*h*l + a*g*h + a*g*i + a*g*k + a*g*l
+ a*h*i*k + a*h*i*l + a*h*i + a*h*k*l + a*h*k + a*h*l + a*i*k + a*i*l + a*k*l
+ c*d*e*g*h*i*k*l + c*d*e*g*h*i*k + c*d*e*g*h*i*l + c*d*e*g*h*k*l +
c*d*e*g*i*k*l + c*d*e*h*i*k*l + c*d*e*h + c*d*e + c*d*g*h*i*k*l + c*d*g*h +
c*d*g + c*d*h*i + c*d*h*k + c*d*h*l + c*d*h + c*d*i + c*d*k + c*d*l +
c*e*g*h*i*k*l + c*e*g*h + c*e*g + c*e*h*i + c*e*h*k + c*e*h*l + c*e*h + c*e*i
+ c*e*k + c*e*l + c*g*h*i + c*g*h*k + c*g*h*l + c*g*h + c*g*i + c*g*k + c*g*l
+ c*h*i*k + c*h*i*l + c*h*i + c*h*k*l + c*h*k + c*h*l + c*i*k + c*i*l + c*k*l
+ d*e*g*h*i*k*l + d*e*g*h + d*e*g + d*e*h*i + d*e*h*k + d*e*h*l + d*e*h +
d*e*i + d*e*k + d*e*l + d*g*h*i + d*g*h*k + d*g*h*l + d*g*h + d*g*i + d*g*k +
d*g*l + d*h*i*k + d*h*i*l + d*h*i + d*h*k*l + d*h*k + d*h*l + d*i*k + d*i*l +
d*k*l + e*g*h*i + e*g*h*k + e*g*h*l + e*g*h + e*g*i + e*g*k + e*g*l + e*h*i*k
+ e*h*i*l + e*h*i + e*h*k*l + e*h*k + e*h*l + e*i*k + e*i*l + e*k*l + g*h*i*k
+ g*h*i*l + g*h*i + g*h*k*l + g*h*k + g*h*l + g*i*k + g*i*l + g*k*l + h*i*k*l
+ h*i*k + h*i*l + h*k*l + i*k*l + 1, f*g*h*j*k*l*n*o + f*g*h*j*k*l*n*p +
f*g*h*j*k*l*n + f*g*h*j*k*l*o*p + f*g*h*j*k*l*o + f*g*h*j*k*l*p +
f*g*h*j*k*n*o*p + f*g*h*j*k*n*o + f*g*h*j*k*n*p + f*g*h*j*k*o*p +
f*g*h*j*l*n*o + f*g*h*j*l*n*p + f*g*h*j*l*o*p + f*g*h*j*n*o*p +
f*g*h*k*l*n*o*p + f*g*h*k*l*n*o + f*g*h*k*l*n*p + f*g*h*k*l*o*p +
f*g*h*k*n*o*p + f*g*h*k + f*g*h*l*n*o*p + f*g*h + f*g*j*k*l*n*o*p +
f*g*j*k*l*n*o + f*g*j*k*l*n*p + f*g*j*k*l*o*p + f*g*j*k*n*o*p + f*g*j*k +
f*g*j*l*n*o*p + f*g*j + f*g*k*l*n*o*p + f*g*k*l + f*g*k*n + f*g*k*o + f*g*k*p
+ f*g*k + f*g*l + f*g*n + f*g*o + f*g*p + f*h*j*k*l*n*o*p + f*h*j*k*l*n*o +
f*h*j*k*l*n*p + f*h*j*k*l*o*p + f*h*j*k*n*o*p + f*h*j*k + f*h*j*l*n*o*p +
f*h*j + f*h*k*l*n*o*p + f*h*k*l + f*h*k*n + f*h*k*o + f*h*k*p + f*h*k + f*h*l
+ f*h*n + f*h*o + f*h*p + f*j*k*l*n*o*p + f*j*k*l + f*j*k*n + f*j*k*o +
f*j*k*p + f*j*k + f*j*l + f*j*n + f*j*o + f*j*p + f*k*l*n + f*k*l*o + f*k*l*p
+ f*k*l + f*k*n*o + f*k*n*p + f*k*n + f*k*o*p + f*k*o + f*k*p + f*l*n + f*l*o
+ f*l*p + f*n*o + f*n*p + f*o*p + g*h*j*k*l*n*o*p + g*h*j*k*l*n*o +
g*h*j*k*l*n*p + g*h*j*k*l*o*p + g*h*j*k*n*o*p + g*h*j*k + g*h*j*l*n*o*p +
g*h*j + g*h*k*l*n*o*p + g*h*k*l + g*h*k*n + g*h*k*o + g*h*k*p + g*h*k + g*h*l
+ g*h*n + g*h*o + g*h*p + g*j*k*l*n*o*p + g*j*k*l + g*j*k*n + g*j*k*o +
g*j*k*p + g*j*k + g*j*l + g*j*n + g*j*o + g*j*p + g*k*l*n + g*k*l*o + g*k*l*p
+ g*k*l + g*k*n*o + g*k*n*p + g*k*n + g*k*o*p + g*k*o + g*k*p + g*l*n + g*l*o
+ g*l*p + g*n*o + g*n*p + g*o*p + h*j*k*l*n*o*p + h*j*k*l + h*j*k*n + h*j*k*o
+ h*j*k*p + h*j*k + h*j*l + h*j*n + h*j*o + h*j*p + h*k*l*n + h*k*l*o +
h*k*l*p + h*k*l + h*k*n*o + h*k*n*p + h*k*n + h*k*o*p + h*k*o + h*k*p + h*l*n
+ h*l*o + h*l*p + h*n*o + h*n*p + h*o*p + j*k*l*n + j*k*l*o + j*k*l*p + j*k*l
+ j*k*n*o + j*k*n*p + j*k*n + j*k*o*p + j*k*o + j*k*p + j*l*n + j*l*o + j*l*p
+ j*n*o + j*n*p + j*o*p + k*l*n*o + k*l*n*p + k*l*n + k*l*o*p + k*l*o + k*l*p
+ k*n*o*p + k*n*o + k*n*p + k*o*p + l*n*o + l*n*p + l*o*p + n*o*p + 1,
e*g*h*i*k*l*m*o + e*g*h*i*k*l*m*p + e*g*h*i*k*l*m + e*g*h*i*k*l*o*p +
e*g*h*i*k*l*o + e*g*h*i*k*l*p + e*g*h*i*k*m*o + e*g*h*i*k*m*p + e*g*h*i*k*o*p
+ e*g*h*i*l*m*o*p + e*g*h*i*l*m*o + e*g*h*i*l*m*p + e*g*h*i*l*o*p +
e*g*h*i*m*o*p + e*g*h*k*l*m*o*p + e*g*h*k*l*m*o + e*g*h*k*l*m*p +
e*g*h*k*l*o*p + e*g*h*k*m*o*p + e*g*h*l*m*o*p + e*g*h*l + e*g*h +
e*g*i*k*l*m*o*p + e*g*i*k*l*m*o + e*g*i*k*l*m*p + e*g*i*k*l*o*p +
e*g*i*k*m*o*p + e*g*i*l*m*o*p + e*g*i*l + e*g*i + e*g*k*l*m*o*p + e*g*k*l +
e*g*k + e*g*l*m + e*g*l*o + e*g*l*p + e*g*l + e*g*m + e*g*o + e*g*p +
e*h*i*k*l*m*o*p + e*h*i*k*l*m*o + e*h*i*k*l*m*p + e*h*i*k*l*o*p +
e*h*i*k*m*o*p + e*h*i*l*m*o*p + e*h*i*l + e*h*i + e*h*k*l*m*o*p + e*h*k*l +
e*h*k + e*h*l*m + e*h*l*o + e*h*l*p + e*h*l + e*h*m + e*h*o + e*h*p +
e*i*k*l*m*o*p + e*i*k*l + e*i*k + e*i*l*m + e*i*l*o + e*i*l*p + e*i*l + e*i*m
+ e*i*o + e*i*p + e*k*l*m + e*k*l*o + e*k*l*p + e*k*l + e*k*m + e*k*o + e*k*p
+ e*l*m*o + e*l*m*p + e*l*m + e*l*o*p + e*l*o + e*l*p + e*m*o + e*m*p + e*o*p
+ g*h*i*k*l*m*o*p + g*h*i*k*l*m*o + g*h*i*k*l*m*p + g*h*i*k*l*o*p +
g*h*i*k*m*o*p + g*h*i*l*m*o*p + g*h*i*l + g*h*i + g*h*k*l*m*o*p + g*h*k*l +
g*h*k + g*h*l*m + g*h*l*o + g*h*l*p + g*h*l + g*h*m + g*h*o + g*h*p +
g*i*k*l*m*o*p + g*i*k*l + g*i*k + g*i*l*m + g*i*l*o + g*i*l*p + g*i*l + g*i*m
+ g*i*o + g*i*p + g*k*l*m + g*k*l*o + g*k*l*p + g*k*l + g*k*m + g*k*o + g*k*p
+ g*l*m*o + g*l*m*p + g*l*m + g*l*o*p + g*l*o + g*l*p + g*m*o + g*m*p + g*o*p
+ h*i*k*l*m*o*p + h*i*k*l + h*i*k + h*i*l*m + h*i*l*o + h*i*l*p + h*i*l +
h*i*m + h*i*o + h*i*p + h*k*l*m + h*k*l*o + h*k*l*p + h*k*l + h*k*m + h*k*o +
h*k*p + h*l*m*o + h*l*m*p + h*l*m + h*l*o*p + h*l*o + h*l*p + h*m*o + h*m*p +
h*o*p + i*k*l*m + i*k*l*o + i*k*l*p + i*k*l + i*k*m + i*k*o + i*k*p + i*l*m*o
+ i*l*m*p + i*l*m + i*l*o*p + i*l*o + i*l*p + i*m*o + i*m*p + i*o*p + k*l*m*o
+ k*l*m*p + k*l*m + k*l*o*p + k*l*o + k*l*p + k*m*o + k*m*p + k*o*p + l*m*o*p
+ l*m*o + l*m*p + l*o*p + m*o*p + 1, e*f*h*i*j*l*m*n + e*f*h*i*j*l*m*p +
e*f*h*i*j*l*m + e*f*h*i*j*l*n*p + e*f*h*i*j*l*n + e*f*h*i*j*l*p +
e*f*h*i*j*m*n*p + e*f*h*i*j*m*n + e*f*h*i*j*m*p + e*f*h*i*j*n*p +
e*f*h*i*l*m*n*p + e*f*h*i*l*m*n + e*f*h*i*l*m*p + e*f*h*i*l*n*p +
e*f*h*i*m*n*p + e*f*h*i + e*f*h*j*l*m*n + e*f*h*j*l*m*p + e*f*h*j*l*n*p +
e*f*h*j*m*n*p + e*f*h*l*m*n*p + e*f*h + e*f*i*j*l*m*n*p + e*f*i*j*l*m*n +
e*f*i*j*l*m*p + e*f*i*j*l*n*p + e*f*i*j*m*n*p + e*f*i*j + e*f*i*l*m*n*p +
e*f*i*l + e*f*i*m + e*f*i*n + e*f*i*p + e*f*i + e*f*j*l*m*n*p + e*f*j + e*f*l
+ e*f*m + e*f*n + e*f*p + e*h*i*j*l*m*n*p + e*h*i*j*l*m*n + e*h*i*j*l*m*p +
e*h*i*j*l*n*p + e*h*i*j*m*n*p + e*h*i*j + e*h*i*l*m*n*p + e*h*i*l + e*h*i*m +
e*h*i*n + e*h*i*p + e*h*i + e*h*j*l*m*n*p + e*h*j + e*h*l + e*h*m + e*h*n +
e*h*p + e*i*j*l*m*n*p + e*i*j*l + e*i*j*m + e*i*j*n + e*i*j*p + e*i*j +
e*i*l*m + e*i*l*n + e*i*l*p + e*i*l + e*i*m*n + e*i*m*p + e*i*m + e*i*n*p +
e*i*n + e*i*p + e*j*l + e*j*m + e*j*n + e*j*p + e*l*m + e*l*n + e*l*p + e*m*n
+ e*m*p + e*n*p + f*h*i*j*l*m*n*p + f*h*i*j*l*m*n + f*h*i*j*l*m*p +
f*h*i*j*l*n*p + f*h*i*j*m*n*p + f*h*i*j + f*h*i*l*m*n*p + f*h*i*l + f*h*i*m +
f*h*i*n + f*h*i*p + f*h*i + f*h*j*l*m*n*p + f*h*j + f*h*l + f*h*m + f*h*n +
f*h*p + f*i*j*l*m*n*p + f*i*j*l + f*i*j*m + f*i*j*n + f*i*j*p + f*i*j +
f*i*l*m + f*i*l*n + f*i*l*p + f*i*l + f*i*m*n + f*i*m*p + f*i*m + f*i*n*p +
f*i*n + f*i*p + f*j*l + f*j*m + f*j*n + f*j*p + f*l*m + f*l*n + f*l*p + f*m*n
+ f*m*p + f*n*p + h*i*j*l*m*n*p + h*i*j*l + h*i*j*m + h*i*j*n + h*i*j*p +
h*i*j + h*i*l*m + h*i*l*n + h*i*l*p + h*i*l + h*i*m*n + h*i*m*p + h*i*m +
h*i*n*p + h*i*n + h*i*p + h*j*l + h*j*m + h*j*n + h*j*p + h*l*m + h*l*n +
h*l*p + h*m*n + h*m*p + h*n*p + i*j*l*m + i*j*l*n + i*j*l*p + i*j*l + i*j*m*n
+ i*j*m*p + i*j*m + i*j*n*p + i*j*n + i*j*p + i*l*m*n + i*l*m*p + i*l*m +
i*l*n*p + i*l*n + i*l*p + i*m*n*p + i*m*n + i*m*p + i*n*p + j*l*m + j*l*n +
j*l*p + j*m*n + j*m*p + j*n*p + l*m*n + l*m*p + l*n*p + m*n*p, e*f*g*i*j*k*m*n
+ e*f*g*i*j*k*m*o + e*f*g*i*j*k*m + e*f*g*i*j*k*n*o + e*f*g*i*j*k*n +
e*f*g*i*j*k*o + e*f*g*i*j*m*n*o + e*f*g*i*j*m*n + e*f*g*i*j*m*o +
e*f*g*i*j*n*o + e*f*g*i*k*m*n + e*f*g*i*k*m*o + e*f*g*i*k*n*o + e*f*g*i*m*n*o
+ e*f*g*j*k*m*n*o + e*f*g*j*k*m*n + e*f*g*j*k*m*o + e*f*g*j*k*n*o +
e*f*g*j*m*n*o + e*f*g*j + e*f*g*k*m*n*o + e*f*g + e*f*i*j*k*m*n*o +
e*f*i*j*k*m*n + e*f*i*j*k*m*o + e*f*i*j*k*n*o + e*f*i*j*m*n*o + e*f*i*j +
e*f*i*k*m*n*o + e*f*i + e*f*j*k*m*n*o + e*f*j*k + e*f*j*m + e*f*j*n + e*f*j*o
+ e*f*j + e*f*k + e*f*m + e*f*n + e*f*o + e*g*i*j*k*m*n*o + e*g*i*j*k*m*n +
e*g*i*j*k*m*o + e*g*i*j*k*n*o + e*g*i*j*m*n*o + e*g*i*j + e*g*i*k*m*n*o +
e*g*i + e*g*j*k*m*n*o + e*g*j*k + e*g*j*m + e*g*j*n + e*g*j*o + e*g*j + e*g*k
+ e*g*m + e*g*n + e*g*o + e*i*j*k*m*n*o + e*i*j*k + e*i*j*m + e*i*j*n +
e*i*j*o + e*i*j + e*i*k + e*i*m + e*i*n + e*i*o + e*j*k*m + e*j*k*n + e*j*k*o
+ e*j*k + e*j*m*n + e*j*m*o + e*j*m + e*j*n*o + e*j*n + e*j*o + e*k*m + e*k*n
+ e*k*o + e*m*n + e*m*o + e*n*o + f*g*i*j*k*m*n*o + f*g*i*j*k*m*n +
f*g*i*j*k*m*o + f*g*i*j*k*n*o + f*g*i*j*m*n*o + f*g*i*j + f*g*i*k*m*n*o +
f*g*i + f*g*j*k*m*n*o + f*g*j*k + f*g*j*m + f*g*j*n + f*g*j*o + f*g*j + f*g*k
+ f*g*m + f*g*n + f*g*o + f*i*j*k*m*n*o + f*i*j*k + f*i*j*m + f*i*j*n +
f*i*j*o + f*i*j + f*i*k + f*i*m + f*i*n + f*i*o + f*j*k*m + f*j*k*n + f*j*k*o
+ f*j*k + f*j*m*n + f*j*m*o + f*j*m + f*j*n*o + f*j*n + f*j*o + f*k*m + f*k*n
+ f*k*o + f*m*n + f*m*o + f*n*o + g*i*j*k*m*n*o + g*i*j*k + g*i*j*m + g*i*j*n
+ g*i*j*o + g*i*j + g*i*k + g*i*m + g*i*n + g*i*o + g*j*k*m + g*j*k*n +
g*j*k*o + g*j*k + g*j*m*n + g*j*m*o + g*j*m + g*j*n*o + g*j*n + g*j*o + g*k*m
+ g*k*n + g*k*o + g*m*n + g*m*o + g*n*o + i*j*k*m + i*j*k*n + i*j*k*o + i*j*k
+ i*j*m*n + i*j*m*o + i*j*m + i*j*n*o + i*j*n + i*j*o + i*k*m + i*k*n + i*k*o
+ i*m*n + i*m*o + i*n*o + j*k*m*n + j*k*m*o + j*k*m + j*k*n*o + j*k*n + j*k*o
+ j*m*n*o + j*m*n + j*m*o + j*n*o + k*m*n + k*m*o + k*n*o + m*n*o,
a*b*d*i*j*l*m*n + a*b*d*i*j*l*m*p + a*b*d*i*j*l*m + a*b*d*i*j*l*n +
a*b*d*i*j*l*p + a*b*d*i*j*m*n*p + a*b*d*i*j*m*n + a*b*d*i*j*m*p +
a*b*d*i*j*n*p + a*b*d*i*l*m*n*p + a*b*d*i*l*m*n + a*b*d*i*l*m*p +
a*b*d*i*l*n*p + a*b*d*i*m*n*p + a*b*d*j*l*m*n*p + a*b*d*j*l*m*n +
a*b*d*j*l*m*p + a*b*d*j*l*n*p + a*b*d*j*m*n*p + a*b*d*l*m*n*p + a*b*d*m +
a*b*d + a*b*i*j*l*m*n*p + a*b*i*j*l*m*n + a*b*i*j*l*m*p + a*b*i*j*l*n*p +
a*b*i*j*m*n*p + a*b*i*l*m*n*p + a*b*i*m + a*b*i + a*b*j*l*m*n*p + a*b*j*m +
a*b*j + a*b*l*m + a*b*l + a*b*m*n + a*b*m*p + a*b*m + a*b*n + a*b*p +
a*d*i*j*l*m*n*p + a*d*i*j*l*m*n + a*d*i*j*l*m*p + a*d*i*j*l*n*p +
a*d*i*j*m*n*p + a*d*i*l*m*n*p + a*d*i*m + a*d*i + a*d*j*l*m*n*p + a*d*j*m +
a*d*j + a*d*l*m + a*d*l + a*d*m*n + a*d*m*p + a*d*m + a*d*n + a*d*p +
a*i*j*l*m*n*p + a*i*j*m + a*i*j + a*i*l*m + a*i*l + a*i*m*n + a*i*m*p + a*i*m
+ a*i*n + a*i*p + a*j*l*m + a*j*l + a*j*m*n + a*j*m*p + a*j*m + a*j*n + a*j*p
+ a*l*m*n + a*l*m*p + a*l*m + a*l*n + a*l*p + a*m*n*p + a*m*n + a*m*p + a*n*p
+ b*d*i*j*l*m*n*p + b*d*i*j*l*m*n + b*d*i*j*l*m*p + b*d*i*j*l*n*p +
b*d*i*j*m*n*p + b*d*i*l*m*n*p + b*d*i*m + b*d*i + b*d*j*l*m*n*p + b*d*j*m +
b*d*j + b*d*l*m + b*d*l + b*d*m*n + b*d*m*p + b*d*m + b*d*n + b*d*p +
b*i*j*l*m*n*p + b*i*j*m + b*i*j + b*i*l*m + b*i*l + b*i*m*n + b*i*m*p + b*i*m
+ b*i*n + b*i*p + b*j*l*m + b*j*l + b*j*m*n + b*j*m*p + b*j*m + b*j*n + b*j*p
+ b*l*m*n + b*l*m*p + b*l*m + b*l*n + b*l*p + b*m*n*p + b*m*n + b*m*p + b*n*p
+ d*i*j*l*m*n*p + d*i*j*m + d*i*j + d*i*l*m + d*i*l + d*i*m*n + d*i*m*p +
d*i*m + d*i*n + d*i*p + d*j*l*m + d*j*l + d*j*m*n + d*j*m*p + d*j*m + d*j*n +
d*j*p + d*l*m*n + d*l*m*p + d*l*m + d*l*n + d*l*p + d*m*n*p + d*m*n + d*m*p +
d*n*p + i*j*l*m + i*j*l + i*j*m*n + i*j*m*p + i*j*m + i*j*n + i*j*p + i*l*m*n
+ i*l*m*p + i*l*m + i*l*n + i*l*p + i*m*n*p + i*m*n + i*m*p + i*n*p + j*l*m*n
+ j*l*m*p + j*l*m + j*l*n + j*l*p + j*m*n*p + j*m*n + j*m*p + j*n*p + l*m*n*p
+ l*m*n + l*m*p + l*n*p + m*n*p, a*b*c*i*j*k*m*n + a*b*c*i*j*k*m +
a*b*c*i*j*k*n*o + a*b*c*i*j*k*n + a*b*c*i*j*k*o + a*b*c*i*j*m*n*o +
a*b*c*i*j*m*n + a*b*c*i*j*m*o + a*b*c*i*j*n*o + a*b*c*i*k*m*n*o +
a*b*c*i*k*m*n + a*b*c*i*k*m*o + a*b*c*i*k*n*o + a*b*c*i*m*n*o +
a*b*c*j*k*m*n*o + a*b*c*j*k*m*n + a*b*c*j*k*m*o + a*b*c*j*k*n*o +
a*b*c*j*m*n*o + a*b*c*k*m*n*o + a*b*c*n + a*b*c + a*b*i*j*k*m*n*o +
a*b*i*j*k*m*n + a*b*i*j*k*m*o + a*b*i*j*k*n*o + a*b*i*j*m*n*o + a*b*i*k*m*n*o
+ a*b*i*n + a*b*i + a*b*j*k*m*n*o + a*b*j*n + a*b*j + a*b*k*n + a*b*k +
a*b*m*n + a*b*m + a*b*n*o + a*b*n + a*b*o + a*c*i*j*k*m*n*o + a*c*i*j*k*m*n +
a*c*i*j*k*m*o + a*c*i*j*k*n*o + a*c*i*j*m*n*o + a*c*i*k*m*n*o + a*c*i*n +
a*c*i + a*c*j*k*m*n*o + a*c*j*n + a*c*j + a*c*k*n + a*c*k + a*c*m*n + a*c*m +
a*c*n*o + a*c*n + a*c*o + a*i*j*k*m*n*o + a*i*j*n + a*i*j + a*i*k*n + a*i*k +
a*i*m*n + a*i*m + a*i*n*o + a*i*n + a*i*o + a*j*k*n + a*j*k + a*j*m*n + a*j*m
+ a*j*n*o + a*j*n + a*j*o + a*k*m*n + a*k*m + a*k*n*o + a*k*n + a*k*o +
a*m*n*o + a*m*n + a*m*o + a*n*o + b*c*i*j*k*m*n*o + b*c*i*j*k*m*n +
b*c*i*j*k*m*o + b*c*i*j*k*n*o + b*c*i*j*m*n*o + b*c*i*k*m*n*o + b*c*i*n +
b*c*i + b*c*j*k*m*n*o + b*c*j*n + b*c*j + b*c*k*n + b*c*k + b*c*m*n + b*c*m +
b*c*n*o + b*c*n + b*c*o + b*i*j*k*m*n*o + b*i*j*n + b*i*j + b*i*k*n + b*i*k +
b*i*m*n + b*i*m + b*i*n*o + b*i*n + b*i*o + b*j*k*n + b*j*k + b*j*m*n + b*j*m
+ b*j*n*o + b*j*n + b*j*o + b*k*m*n + b*k*m + b*k*n*o + b*k*n + b*k*o +
b*m*n*o + b*m*n + b*m*o + b*n*o + c*i*j*k*m*n*o + c*i*j*n + c*i*j + c*i*k*n +
c*i*k + c*i*m*n + c*i*m + c*i*n*o + c*i*n + c*i*o + c*j*k*n + c*j*k + c*j*m*n
+ c*j*m + c*j*n*o + c*j*n + c*j*o + c*k*m*n + c*k*m + c*k*n*o + c*k*n + c*k*o
+ c*m*n*o + c*m*n + c*m*o + c*n*o + i*j*k*n + i*j*k + i*j*m*n + i*j*m +
i*j*n*o + i*j*n + i*j*o + i*k*m*n + i*k*m + i*k*n*o + i*k*n + i*k*o + i*m*n*o
+ i*m*n + i*m*o + i*n*o + j*k*m*n + j*k*m + j*k*n*o + j*k*n + j*k*o + j*m*n*o
+ j*m*n + j*m*o + j*n*o + k*m*n*o + k*m*n + k*m*o + k*n*o + m*n*o,
b*c*d*j*k*l*n*o + b*c*d*j*k*l*n + b*c*d*j*k*l*o*p + b*c*d*j*k*l*o +
b*c*d*j*k*l*p + b*c*d*j*k*n*o*p + b*c*d*j*k*n*o + b*c*d*j*k*n*p +
b*c*d*j*k*o*p + b*c*d*j*l*n*o*p + b*c*d*j*l*n*o + b*c*d*j*l*n*p +
b*c*d*j*l*o*p + b*c*d*j*n*o*p + b*c*d*k*l*n*o*p + b*c*d*k*l*n*o +
b*c*d*k*l*n*p + b*c*d*k*l*o*p + b*c*d*k*n*o*p + b*c*d*l*n*o*p + b*c*d*o +
b*c*d + b*c*j*k*l*n*o*p + b*c*j*k*l*n*o + b*c*j*k*l*n*p + b*c*j*k*l*o*p +
b*c*j*k*n*o*p + b*c*j*l*n*o*p + b*c*j*o + b*c*j + b*c*k*l*n*o*p + b*c*k*o +
b*c*k + b*c*l*o + b*c*l + b*c*n*o + b*c*n + b*c*o*p + b*c*o + b*c*p +
b*d*j*k*l*n*o*p + b*d*j*k*l*n*o + b*d*j*k*l*n*p + b*d*j*k*l*o*p +
b*d*j*k*n*o*p + b*d*j*l*n*o*p + b*d*j*o + b*d*j + b*d*k*l*n*o*p + b*d*k*o +
b*d*k + b*d*l*o + b*d*l + b*d*n*o + b*d*n + b*d*o*p + b*d*o + b*d*p +
b*j*k*l*n*o*p + b*j*k*o + b*j*k + b*j*l*o + b*j*l + b*j*n*o + b*j*n + b*j*o*p
+ b*j*o + b*j*p + b*k*l*o + b*k*l + b*k*n*o + b*k*n + b*k*o*p + b*k*o + b*k*p
+ b*l*n*o + b*l*n + b*l*o*p + b*l*o + b*l*p + b*n*o*p + b*n*o + b*n*p + b*o*p
+ c*d*j*k*l*n*o*p + c*d*j*k*l*n*o + c*d*j*k*l*n*p + c*d*j*k*l*o*p +
c*d*j*k*n*o*p + c*d*j*l*n*o*p + c*d*j*o + c*d*j + c*d*k*l*n*o*p + c*d*k*o +
c*d*k + c*d*l*o + c*d*l + c*d*n*o + c*d*n + c*d*o*p + c*d*o + c*d*p +
c*j*k*l*n*o*p + c*j*k*o + c*j*k + c*j*l*o + c*j*l + c*j*n*o + c*j*n + c*j*o*p
+ c*j*o + c*j*p + c*k*l*o + c*k*l + c*k*n*o + c*k*n + c*k*o*p + c*k*o + c*k*p
+ c*l*n*o + c*l*n + c*l*o*p + c*l*o + c*l*p + c*n*o*p + c*n*o + c*n*p + c*o*p
+ d*j*k*l*n*o*p + d*j*k*o + d*j*k + d*j*l*o + d*j*l + d*j*n*o + d*j*n +
d*j*o*p + d*j*o + d*j*p + d*k*l*o + d*k*l + d*k*n*o + d*k*n + d*k*o*p + d*k*o
+ d*k*p + d*l*n*o + d*l*n + d*l*o*p + d*l*o + d*l*p + d*n*o*p + d*n*o + d*n*p
+ d*o*p + j*k*l*o + j*k*l + j*k*n*o + j*k*n + j*k*o*p + j*k*o + j*k*p +
j*l*n*o + j*l*n + j*l*o*p + j*l*o + j*l*p + j*n*o*p + j*n*o + j*n*p + j*o*p +
k*l*n*o + k*l*n + k*l*o*p + k*l*o + k*l*p + k*n*o*p + k*n*o + k*n*p + k*o*p +
l*n*o*p + l*n*o + l*n*p + l*o*p + n*o*p, a*c*d*i*k*l*m*p + a*c*d*i*k*l*m +
a*c*d*i*k*l*o*p + a*c*d*i*k*l*o + a*c*d*i*k*l*p + a*c*d*i*k*m*o*p +
a*c*d*i*k*m*o + a*c*d*i*k*m*p + a*c*d*i*k*o*p + a*c*d*i*l*m*o*p +
a*c*d*i*l*m*o + a*c*d*i*l*m*p + a*c*d*i*l*o*p + a*c*d*i*m*o*p +
a*c*d*k*l*m*o*p + a*c*d*k*l*m*o + a*c*d*k*l*m*p + a*c*d*k*l*o*p +
a*c*d*k*m*o*p + a*c*d*l*m*o*p + a*c*d*p + a*c*d + a*c*i*k*l*m*o*p +
a*c*i*k*l*m*o + a*c*i*k*l*m*p + a*c*i*k*l*o*p + a*c*i*k*m*o*p + a*c*i*l*m*o*p
+ a*c*i*p + a*c*i + a*c*k*l*m*o*p + a*c*k*p + a*c*k + a*c*l*p + a*c*l +
a*c*m*p + a*c*m + a*c*o*p + a*c*o + a*c*p + a*d*i*k*l*m*o*p + a*d*i*k*l*m*o +
a*d*i*k*l*m*p + a*d*i*k*l*o*p + a*d*i*k*m*o*p + a*d*i*l*m*o*p + a*d*i*p +
a*d*i + a*d*k*l*m*o*p + a*d*k*p + a*d*k + a*d*l*p + a*d*l + a*d*m*p + a*d*m +
a*d*o*p + a*d*o + a*d*p + a*i*k*l*m*o*p + a*i*k*p + a*i*k + a*i*l*p + a*i*l +
a*i*m*p + a*i*m + a*i*o*p + a*i*o + a*i*p + a*k*l*p + a*k*l + a*k*m*p + a*k*m
+ a*k*o*p + a*k*o + a*k*p + a*l*m*p + a*l*m + a*l*o*p + a*l*o + a*l*p +
a*m*o*p + a*m*o + a*m*p + a*o*p + c*d*i*k*l*m*o*p + c*d*i*k*l*m*o +
c*d*i*k*l*m*p + c*d*i*k*l*o*p + c*d*i*k*m*o*p + c*d*i*l*m*o*p + c*d*i*p +
c*d*i + c*d*k*l*m*o*p + c*d*k*p + c*d*k + c*d*l*p + c*d*l + c*d*m*p + c*d*m +
c*d*o*p + c*d*o + c*d*p + c*i*k*l*m*o*p + c*i*k*p + c*i*k + c*i*l*p + c*i*l +
c*i*m*p + c*i*m + c*i*o*p + c*i*o + c*i*p + c*k*l*p + c*k*l + c*k*m*p + c*k*m
+ c*k*o*p + c*k*o + c*k*p + c*l*m*p + c*l*m + c*l*o*p + c*l*o + c*l*p +
c*m*o*p + c*m*o + c*m*p + c*o*p + d*i*k*l*m*o*p + d*i*k*p + d*i*k + d*i*l*p +
d*i*l + d*i*m*p + d*i*m + d*i*o*p + d*i*o + d*i*p + d*k*l*p + d*k*l + d*k*m*p
+ d*k*m + d*k*o*p + d*k*o + d*k*p + d*l*m*p + d*l*m + d*l*o*p + d*l*o + d*l*p
+ d*m*o*p + d*m*o + d*m*p + d*o*p + i*k*l*p + i*k*l + i*k*m*p + i*k*m +
i*k*o*p + i*k*o + i*k*p + i*l*m*p + i*l*m + i*l*o*p + i*l*o + i*l*p + i*m*o*p
+ i*m*o + i*m*p + i*o*p + k*l*m*p + k*l*m + k*l*o*p + k*l*o + k*l*p + k*m*o*p
+ k*m*o + k*m*p + k*o*p + l*m*o*p + l*m*o + l*m*p + l*o*p + m*o*p}

I = ideal L
time C = gens gb( I, Algorithm=>Sugarless); -- this takes 29.3027 seconds, without Sugarless it takes currently takes 4 days or so, at least according to Franzi...!
time B = gbBoolean I; --  -- used 506.507 seconds Revision: 10953
-- Revision: 10961  -- used 249.015 seconds

-- we computed 987516 S Polynomials and added 1078 of them to the intermediate basis.
-- Revision: 10970 -- used 204.691 seconds


assert( sort gens B - sort C == 0 )


end -- test file disabled by Dan; these tests should be moved to the package BooleanGB

restart
R=ZZ/2[a1,a2,a3,a4,b1,b2,b3,b4,c1,c2,c3,c4,d1,d2,d3,d4,e1,e2,e3,e4,f1,f2,f3,f4,g1,g2,g3,g4,h1,h2,h3,h4,i1,i2,i3,i4,j1,j2,j3,j4,k1,k2,k3,k4,l1,l2,l3,l4,m1,m2,m3,m4,n1,n2,n3,n4,o1,o2,o3,o4,p1,p2,p3,p4];
FP =  apply( gens R, i -> i*(i-1));
--FP are the field polynomials
RQ = R/ideal FP
I= ideal {a1+a2+a3+a4-1, b1+b2+b3+b4-1, c1+c2+c3+c4-1, d1+d2+d3+d4-1, e1+e2+e3+e4-1, f1+f2+f3+f4-1, g1+g2+g3+g4-1, h1+h2+h3+h4-1, i1+i2+i3+i4-1, j1+j2+j3+j4-1, k1+k2+k3+k4-1, l1+l2+l3+l4-1, m1+m2+m3+m4-1, n1+n2+n3+n4-1, o1+o2+o3+o4-1, p1+p2+p3+p4-1, a1*b1+a2*b2+a3*b3+a4*b4, a1*c1+a2*c2+a3*c3+a4*c4, a1*d1+a2*d2+a3*d3+a4*d4, a1*e1+a2*e2+a3*e3+a4*e4, a1*f1+a2*f2+a3*f3+a4*f4, a1*i1+a2*i2+a3*i3+a4*i4, a1*m1+a2*m2+a3*m3+a4*m4, b1*c1+b2*c2+b3*c3+b4*c4, b1*d1+b2*d2+b3*d3+b4*d4, b1*e1+b2*e2+b3*e3+b4*e4, b1*f1+b2*f2+b3*f3+b4*f4, b1*j1+b2*j2+b3*j3+b4*j4, b1*n1+b2*n2+b3*n3+b4*n4, c1*d1+c2*d2+c3*d3+c4*d4, c1*g1+c2*g2+c3*g3+c4*g4, c1*h1+c2*h2+c3*h3+c4*h4, c1*k1+c2*k2+c3*k3+c4*k4, c1*o1+c2*o2+c3*o3+c4*o4, d1*g1+d2*g2+d3*g3+d4*g4, d1*h1+d2*h2+d3*h3+d4*h4, d1*l1+d2*l2+d3*l3+d4*l4, d1*p1+d2*p2+d3*p3+d4*p4, e1*f1+e2*f2+e3*f3+e4*f4, e1*g1+e2*g2+e3*g3+e4*g4, e1*h1+e2*h2+e3*h3+e4*h4, e1*i1+e2*i2+e3*i3+e4*i4, e1*m1+e2*m2+e3*m3+e4*m4, f1*g1+f2*g2+f3*g3+f4*g4, f1*h1+f2*h2+f3*h3+f4*h4, f1*j1+f2*j2+f3*j3+f4*j4, f1*n1+f2*n2+f3*n3+f4*n4, g1*h1+g2*h2+g3*h3+g4*h4, g1*k1+g2*k2+g3*k3+g4*k4, g1*o1+g2*o2+g3*o3+g4*o4, h1*l1+h2*l2+h3*l3+h4*l4, h1*p1+h2*p2+h3*p3+h4*p4, i1*j1+i2*j2+i3*j3+i4*j4, i1*k1+i2*k2+i3*k3+i4*k4, i1*l1+i2*l2+i3*l3+i4*l4, i1*m1+i2*m2+i3*m3+i4*m4, i1*n1+i2*n2+i3*n3+i4*n4, j1*k1+j2*k2+j3*k3+j4*k4, j1*l1+j2*l2+j3*l3+j4*l4, j1*m1+j2*m2+j3*m3+j4*m4, j1*n1+j2*n2+j3*n3+j4*n4, k1*l1+k2*l2+k3*l3+k4*l4, k1*o1+k2*o2+k3*o3+k4*o4, k1*p1+k2*p2+k3*p3+k4*p4, l1*o1+l2*o2+l3*o3+l4*o4, l1*p1+l2*p2+l3*p3+l4*p4, m1*n1+m2*n2+m3*n3+m4*n4, m1*o1+m2*o2+m3*o3+m4*o4, m1*p1+m2*p2+m3*p3+m4*p4, n1*o1+n2*o2+n3*o3+n4*o4, n1*p1+n2*p2+n3*p3+n4*p4, o1*p1+o2*p2+o3*p3+o4*p4}
--K1 is the set of Shidoku Polynomials (not quite right for Shidoku, but gives the right answer most of the time! I still need to modify it for ZZ/2 computations.)
--K2 = ideal {d1, d2, d3, d4-1, e1, e2, e3, e4-1, g1, g2-1, g3, g4, j1, j2, j3-1, j4, l1-1, l2, l3, l4, m1-1, m2, m3, m4}
--K2 = ideal {d1, d2-1, d3, d4, e1, e2, e3, e4-1, j1, j2, j3-1, j4, m1-1, m2, m3, m4}
--K2 is the set of initial clues.  This should be a fast computation
--time gens gb (K1+K2) -- used 0.024064 seconds
--time gens gb (K1) -- used seconds
--time C = gens gb I; -- 
gbTrace = 3
time C = gb (I, Algorithm=>Sugarless); -- 
time B = gbBoolean I;
assert( sort gens B - sort gens C == 0 )


end -- test file disabled by Dan; these tests should be moved to the package BooleanGB

-- load "franzi-brp.m2" 
R = ZZ/2[x,y,z, MonomialOrder=>Lex];
l = apply(gens R, x-> x^2+x);
QR = R/l;
I = ideal(x+y, y*z);
time C = gens gb I;
time B = gbBoolean I;
assert( gens B - C == 0 )

R = ZZ/2[x,y,z, MonomialOrder=>Lex];
l = apply(gens R, x-> x^2+x);
QR = R/l;
I = ideal(x+y, y*z, x, y);
time C = gens gb I;
time B = gbBoolean I;
assert( gens B - C == 0 )


restart
R=ZZ/2[a1,a2,a3,a4,b1,b2,b3,b4,c1,c2,c3,c4,d1,d2,d3,d4,e1,e2,e3,e4,f1,f2,f3,f4,g1,g2,g3,g4,h1,h2,h3,h4,i1,i2,i3,i4,j1,j2,j3,j4,k1,k2,k3,k4,l1,l2,l3,l4,m1,m2,m3,m4,n1,n2,n3,n4,o1,o2,o3,o4,p1,p2,p3,p4];
FP =  apply( gens R, i -> i*(i-1));
--FP are the field polynomials
RQ = R/ideal FP
I= ideal {a1+a2+a3+a4-1, b1+b2+b3+b4-1, c1+c2+c3+c4-1, d1+d2+d3+d4-1, e1+e2+e3+e4-1, f1+f2+f3+f4-1, g1+g2+g3+g4-1, h1+h2+h3+h4-1, i1+i2+i3+i4-1, j1+j2+j3+j4-1, k1+k2+k3+k4-1, l1+l2+l3+l4-1, m1+m2+m3+m4-1, n1+n2+n3+n4-1, o1+o2+o3+o4-1, p1+p2+p3+p4-1, a1*b1+a2*b2+a3*b3+a4*b4, a1*c1+a2*c2+a3*c3+a4*c4, a1*d1+a2*d2+a3*d3+a4*d4, a1*e1+a2*e2+a3*e3+a4*e4, a1*f1+a2*f2+a3*f3+a4*f4, a1*i1+a2*i2+a3*i3+a4*i4, a1*m1+a2*m2+a3*m3+a4*m4, b1*c1+b2*c2+b3*c3+b4*c4, b1*d1+b2*d2+b3*d3+b4*d4, b1*e1+b2*e2+b3*e3+b4*e4, b1*f1+b2*f2+b3*f3+b4*f4, b1*j1+b2*j2+b3*j3+b4*j4, b1*n1+b2*n2+b3*n3+b4*n4, c1*d1+c2*d2+c3*d3+c4*d4, c1*g1+c2*g2+c3*g3+c4*g4, c1*h1+c2*h2+c3*h3+c4*h4, c1*k1+c2*k2+c3*k3+c4*k4, c1*o1+c2*o2+c3*o3+c4*o4, d1*g1+d2*g2+d3*g3+d4*g4, d1*h1+d2*h2+d3*h3+d4*h4, d1*l1+d2*l2+d3*l3+d4*l4, d1*p1+d2*p2+d3*p3+d4*p4, e1*f1+e2*f2+e3*f3+e4*f4, e1*g1+e2*g2+e3*g3+e4*g4, e1*h1+e2*h2+e3*h3+e4*h4, e1*i1+e2*i2+e3*i3+e4*i4, e1*m1+e2*m2+e3*m3+e4*m4, f1*g1+f2*g2+f3*g3+f4*g4, f1*h1+f2*h2+f3*h3+f4*h4, f1*j1+f2*j2+f3*j3+f4*j4, f1*n1+f2*n2+f3*n3+f4*n4, g1*h1+g2*h2+g3*h3+g4*h4, g1*k1+g2*k2+g3*k3+g4*k4, g1*o1+g2*o2+g3*o3+g4*o4, h1*l1+h2*l2+h3*l3+h4*l4, h1*p1+h2*p2+h3*p3+h4*p4, i1*j1+i2*j2+i3*j3+i4*j4, i1*k1+i2*k2+i3*k3+i4*k4, i1*l1+i2*l2+i3*l3+i4*l4, i1*m1+i2*m2+i3*m3+i4*m4, i1*n1+i2*n2+i3*n3+i4*n4, j1*k1+j2*k2+j3*k3+j4*k4, j1*l1+j2*l2+j3*l3+j4*l4, j1*m1+j2*m2+j3*m3+j4*m4, j1*n1+j2*n2+j3*n3+j4*n4, k1*l1+k2*l2+k3*l3+k4*l4, k1*o1+k2*o2+k3*o3+k4*o4, k1*p1+k2*p2+k3*p3+k4*p4, l1*o1+l2*o2+l3*o3+l4*o4, l1*p1+l2*p2+l3*p3+l4*p4, m1*n1+m2*n2+m3*n3+m4*n4, m1*o1+m2*o2+m3*o3+m4*o4, m1*p1+m2*p2+m3*p3+m4*p4, n1*o1+n2*o2+n3*o3+n4*o4, n1*p1+n2*p2+n3*p3+n4*p4, o1*p1+o2*p2+o3*p3+o4*p4}
gbTrace = 3
time C = gb (I, Algorithm=>Sugarless); -- 
time B = gbBoolean I;
assert( sort gens B - sort gens C == 0 )


restart
installPackage "BooleanGB"
check "BooleanGB"
