
/* 
 * International Color Consortium color transform expanded support
 *
 * Author:  Graeme W. Gill
 * Date:    8/5/2002
 * Version: 1.00
 *
 * Copyright 2002 Graeme W. Gill
 * All rights reserved.
 * This material is licenced under the GNU AFFERO GENERAL PUBLIC LICENSE Version 3 :-
 * see the License.txt file for licencing details.
 *
 */

/*
 * This is some test code to test the FWA compensation
 * feature of the spectal to CIE conversion.
 *
 */

#define FILTERED_D65	/* Use Spectrolino filtered D65 instead of ideal D65 */

#define DOPLOT		/* Graphs: Black = target curve */
					/*         Red   = uncorrected curve */
					/*         Green = corrected curve */

#define DATAFILE	/* Write the plot data to a data file */

#include <stdio.h>
#include <math.h>
#include "xspect.h"
#include "numlib.h"
#ifdef DOPLOT
#include "plot.h"
#endif


/* Spectrolino filter "D65" illuminant */
static xspect il_sod65 = {
	36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
	100.0,				/* Scale factor */
	{
		17.42, 29.51, 36.54, 38.39, 38.15,
		39.68, 44.7,  53.6,  64.26, 74.01,
		82.39, 89.9, 93.45, 94.05, 91.59,
		89.4, 93.69, 100.85, 99.66, 87.12,
		78.34, 82.95, 91.31, 98.18, 99.99,
		104.38, 120.06, 145.35, 172.8, 190.38,
		192.46, 181.28, 163.7, 143.14, 123.57,
		104.67
	}
};
 
/* Material/illuminant record */
struct {
	char *media;			/* Media/process identifier */
	char *illum;			/* Primary illuminant identifier */
	icxIllumeType ill;		/* Illuminant used to get reflectance spectrum */
	xspect white;			/* Media on its own spectrum */
	xspect black;			/* Media with maximum colorant */
	struct {
		char *pdesc;		/* Description of sample */
		xspect s;			/* Spectrum of it */
	} patches[6];

} matilum[2][2]	= {		/* [Cromalin, Epson 10K] [A, D65] */
	{
		{
			"Cromalin",			/* Paper measured */
			"Illuminant A",		/* Illuminant measured under */
			icxIT_A,
			{	/* White */
				36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
				1.0,				/* Scale factor */
				{
					0.0376, 0.0217, 0.0325, 0.1779, 0.5891,
					0.9366, 1.0307, 1.0241, 0.9996, 0.9738,
					0.9521, 0.9355, 0.9245, 0.9174, 0.9147,
					0.9113, 0.9112, 0.9118, 0.9111, 0.9148,
					0.9147, 0.9162, 0.9172, 0.9185, 0.9197,
					0.9194, 0.9213, 0.9239, 0.9259, 0.9257,
					0.9259, 0.9250, 0.9271, 0.9293, 0.9296, 0.9310
				}
			},
			{	/* Black */
				36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
				1.0,				/* Scale factor */
				{
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000
				}
			},
			{
				{
					"30% Y",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0274, 0.0161, 0.0218, 0.1072, 0.3077,
							0.4280, 0.4532, 0.4530, 0.4508, 0.4498,
							0.4585, 0.5019, 0.6137, 0.7716, 0.8680,
							0.8939, 0.9000, 0.8999, 0.9015, 0.9098,
							0.9153, 0.9193, 0.9212, 0.9230, 0.9245,
							0.9247, 0.9267, 0.9294, 0.9317, 0.9315,
							0.9317, 0.9317, 0.9331, 0.9354, 0.9357, 0.9366
						}
					}
				},
				{
					"60% Y",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0225, 0.0130, 0.0169, 0.0613, 0.1512,
							0.1951, 0.2039, 0.2059, 0.2095, 0.2149,
							0.2301, 0.2850, 0.4283, 0.6585, 0.8249,
							0.8767, 0.8878, 0.8860, 0.8887, 0.9017,
							0.9115, 0.9181, 0.9202, 0.9219, 0.9232,
							0.9229, 0.9247, 0.9269, 0.9289, 0.9283,
							0.9280, 0.9277, 0.9292, 0.9307, 0.9305, 0.9315
						}
					}
				},
				{
					"90% Y",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0195, 0.0114, 0.0140, 0.0314, 0.0568,
							0.0664, 0.0691, 0.0708, 0.0760, 0.0836,
							0.1003, 0.1571, 0.3087, 0.5728, 0.7861,
							0.8602, 0.8759, 0.8737, 0.8781, 0.8952,
							0.9094, 0.9185, 0.9215, 0.9234, 0.9250,
							0.9248, 0.9267, 0.9292, 0.9312, 0.9307,
							0.9305, 0.9300, 0.9313, 0.9328, 0.9321, 0.9339
						}
					}
				},
				{
					"30% K",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0163, 0.0117, 0.0181, 0.1018, 0.3029,
							0.4324, 0.4585, 0.4539, 0.4437, 0.4330,
							0.4247, 0.4180, 0.4140, 0.4113, 0.4108,
							0.4095, 0.4097, 0.4101, 0.4100, 0.4117,
							0.4119, 0.4126, 0.4133, 0.4141, 0.4152,
							0.4156, 0.4169, 0.4187, 0.4200, 0.4206,
							0.4211, 0.4214, 0.4224, 0.4240, 0.4243, 0.4256
						}
					}
				},
				{
					"60% K",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0087, 0.0070, 0.0138, 0.0539, 0.1378,
							0.1802, 0.1872, 0.1840, 0.1794, 0.1748,
							0.1711, 0.1686, 0.1669, 0.1661, 0.1659,
							0.1656, 0.1657, 0.1661, 0.1660, 0.1670,
							0.1670, 0.1675, 0.1680, 0.1686, 0.1693,
							0.1700, 0.1707, 0.1720, 0.1728, 0.1735,
							0.1740, 0.1745, 0.1749, 0.1761, 0.1759, 0.1779
						}
					}
				},
				{
					"90% K",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0044, 0.0040, 0.0090, 0.0243, 0.0430,
							0.0503, 0.0508, 0.0485, 0.0460, 0.0440,
							0.0423, 0.0412, 0.0406, 0.0404, 0.0404,
							0.0404, 0.0407, 0.0409, 0.0411, 0.0415,
							0.0418, 0.0422, 0.0424, 0.0429, 0.0434,
							0.0440, 0.0444, 0.0452, 0.0459, 0.0465,
							0.0469, 0.0473, 0.0477, 0.0483, 0.0489, 0.0496
						}
					}
				}
			}
		},
		{
			"Cromalin",
			"Illuminant filtered D65",
			icxIT_D65,
			{	/* White */
				36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
				1.0,				/* Scale factor */
				{
					0.0096, 0.0082, 0.0242, 0.1791, 0.6547,
					1.0944, 1.2012, 1.1545, 1.0891, 1.0388,
					1.0009, 0.9725, 0.9552, 0.9442, 0.9384,
					0.9313, 0.9266, 0.9235, 0.9213, 0.9250,
					0.9249, 0.9247, 0.9239, 0.9248, 0.9253,
					0.9249, 0.9267, 0.9293, 0.9310, 0.9308,
					0.9307, 0.9311, 0.9309, 0.9335, 0.9263, 0.8774
				}
			},
			{	/* Black */
				36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
				1.0,				/* Scale factor */
				{
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000
				}
			},
			{
				{
					"30% Y",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0054, 0.0050, 0.0157, 0.1084, 0.3443,
							0.4998, 0.5278, 0.5108, 0.4915, 0.4802,
							0.4823, 0.5221, 0.6331, 0.7929, 0.8894,
							0.9116, 0.9134, 0.9095, 0.9097, 0.9176,
							0.9222, 0.9245, 0.9251, 0.9267, 0.9278,
							0.9275, 0.9293, 0.9323, 0.9344, 0.9346,
							0.9348, 0.9346, 0.9344, 0.9354, 0.9252, 0.8748
						}
					}
				},
				{
					"60% Y",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0039, 0.0042, 0.0117, 0.0632, 0.1733,
							0.2333, 0.2430, 0.2364, 0.2315, 0.2315,
							0.2440, 0.2976, 0.4421, 0.6777, 0.8461,
							0.8950, 0.9014, 0.8967, 0.8971, 0.9095,
							0.9187, 0.9234, 0.9242, 0.9263, 0.9270,
							0.9257, 0.9280, 0.9303, 0.9326, 0.9322,
							0.9315, 0.9317, 0.9314, 0.9306, 0.9169, 0.8664
						}
					}
				},
				{
					"90% Y",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0032, 0.0038, 0.0099, 0.0342, 0.0716,
							0.0889, 0.0918, 0.0884, 0.0887, 0.0931,
							0.1086, 0.1652, 0.3190, 0.5892, 0.8060,
							0.8777, 0.8891, 0.8839, 0.8863, 0.9022,
							0.9158, 0.9230, 0.9259, 0.9269, 0.9283,
							0.9266, 0.9304, 0.9327, 0.9350, 0.9344,
							0.9342, 0.9336, 0.9328, 0.9323, 0.9175, 0.8589
						}
					}
				},
				{
					"30% K",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0047, 0.0049, 0.0153, 0.1046, 0.3404,
							0.5036, 0.5317, 0.5102, 0.4828, 0.4620,
							0.4468, 0.4358, 0.4293, 0.4249, 0.4232,
							0.4207, 0.4191, 0.4181, 0.4172, 0.4188,
							0.4190, 0.4194, 0.4191, 0.4196, 0.4203,
							0.4204, 0.4218, 0.4236, 0.4251, 0.4255,
							0.4258, 0.4259, 0.4267, 0.4276, 0.4249, 0.4031
						}
					}
				},
				{
					"60% K",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0042, 0.0047, 0.0126, 0.0586, 0.1584,
							0.2146, 0.2206, 0.2101, 0.1978, 0.1885,
							0.1820, 0.1774, 0.1748, 0.1730, 0.1726,
							0.1714, 0.1710, 0.1703, 0.1704, 0.1712,
							0.1709, 0.1717, 0.1713, 0.1724, 0.1730,
							0.1732, 0.1738, 0.1753, 0.1759, 0.1768,
							0.1766, 0.1778, 0.1774, 0.1775, 0.1774, 0.1707
						}
					}
				},
				{
					"90% K",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0018, 0.0039, 0.0103, 0.0293, 0.0560,
							0.0677, 0.0676, 0.0611, 0.0551, 0.0505,
							0.0475, 0.0452, 0.0441, 0.0434, 0.0430,
							0.0430, 0.0428, 0.0428, 0.0427, 0.0433,
							0.0433, 0.0434, 0.0441, 0.0438, 0.0448,
							0.0457, 0.0453, 0.0468, 0.0471, 0.0477,
							0.0483, 0.0483, 0.0487, 0.0486, 0.0492, 0.0500
						}
					}
				}
			}
		}
	}, 
	{
		{
			"Epson10K",
			"Illuminant A",
			icxIT_A,
			{	/* White */
				36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
				1.0,				/* Scale factor */
				{
					0.4561, 0.4677, 0.5052, 0.6477, 0.8785,
					0.9929, 1.0127, 0.9909, 0.9676, 0.9539,
					0.9420, 0.9330, 0.9277, 0.9246, 0.9236,
					0.9205, 0.9200, 0.9191, 0.9172, 0.9198,
					0.9184, 0.9185, 0.9181, 0.9189, 0.9204,
					0.9207, 0.9228, 0.9256, 0.9277, 0.9276,
					0.9284, 0.9286, 0.9304, 0.9332, 0.9321, 0.9349
				}
			},
			{	/* Black */
				36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
				1.0,				/* Scale factor */
				{
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000
				}
			},
			{
				{
					"30% Y",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.3650, 0.3689, 0.3908, 0.4912, 0.6549,
							0.7395, 0.7667, 0.7745, 0.7868, 0.8103,
							0.8385, 0.8663, 0.8869, 0.8983, 0.9037,
							0.9028, 0.9031, 0.9024, 0.9006, 0.9030,
							0.9018, 0.9016, 0.9007, 0.9011, 0.9013,
							0.9002, 0.9015, 0.9032, 0.9048, 0.9043,
							0.9040, 0.9038, 0.9050, 0.9073, 0.9062, 0.9082
						}
					}
				},
				{
					"60% Y",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.2376, 0.2286, 0.2349, 0.2856, 0.3740,
							0.4290, 0.4642, 0.5005, 0.5513, 0.6172,
							0.6959, 0.7750, 0.8340, 0.8683, 0.8848,
							0.8884, 0.8901, 0.8900, 0.8884, 0.8906,
							0.8892, 0.8893, 0.8884, 0.8885, 0.8886,
							0.8875, 0.8885, 0.8903, 0.8917, 0.8910,
							0.8911, 0.8908, 0.8924, 0.8951, 0.8940, 0.8962
						}
					}
				},
				{
					"90% Y",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0482, 0.0362, 0.0314, 0.0323, 0.0373,
							0.0448, 0.0570, 0.0829, 0.1310, 0.2130,
							0.3465, 0.5227, 0.6838, 0.7871, 0.8409,
							0.8622, 0.8713, 0.8749, 0.8752, 0.8787,
							0.8783, 0.8787, 0.8781, 0.8784, 0.8784,
							0.8773, 0.8780, 0.8794, 0.8802, 0.8789,
							0.8783, 0.8773, 0.8778, 0.8794, 0.8775, 0.8779
						}
					}
				},
				{
					"30% K",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.3304, 0.3398, 0.3632, 0.4562, 0.6018,
							0.6702, 0.6808, 0.6663, 0.6520, 0.6434,
							0.6362, 0.6311, 0.6285, 0.6272, 0.6269,
							0.6253, 0.6253, 0.6253, 0.6247, 0.6274,
							0.6280, 0.6301, 0.6320, 0.6351, 0.6385,
							0.6414, 0.6464, 0.6530, 0.6593, 0.6643,
							0.6692, 0.6727, 0.6770, 0.6811, 0.6826, 0.6856
						}
					}
				},
				{
					"60% K",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.1735, 0.1784, 0.1886, 0.2295, 0.2936,
							0.3232, 0.3279, 0.3227, 0.3177, 0.3152,
							0.3133, 0.3126, 0.3126, 0.3131, 0.3141,
							0.3142, 0.3151, 0.3160, 0.3169, 0.3195,
							0.3217, 0.3251, 0.3287, 0.3335, 0.3386,
							0.3439, 0.3511, 0.3599, 0.3692, 0.3775,
							0.3853, 0.3919, 0.3982, 0.4036, 0.4068, 0.4109
						}
					}
				},
				{
					"90% K",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0197, 0.0198, 0.0201, 0.0201, 0.0208,
							0.0207, 0.0204, 0.0202, 0.0200, 0.0197,
							0.0194, 0.0194, 0.0193, 0.0192, 0.0191,
							0.0191, 0.0190, 0.0190, 0.0190, 0.0192,
							0.0194, 0.0198, 0.0205, 0.0212, 0.0221,
							0.0230, 0.0244, 0.0261, 0.0282, 0.0304,
							0.0326, 0.0346, 0.0366, 0.0383, 0.0396, 0.0412
						}
					}
				}
			}
		},
		{
			"Epson10K",
			"Illuminant filtered D65",
			icxIT_D65,
			{	/* White */
				36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
				1.0,				/* Scale factor */
				{
					0.4452, 0.4631, 0.5041, 0.6692, 0.9582,
					1.1083, 1.1289, 1.0717, 1.0191, 0.9929,
					0.9721, 0.9560, 0.9483, 0.9439, 0.9419,
					0.9374, 0.9357, 0.9337, 0.9313, 0.9340,
					0.9330, 0.9329, 0.9320, 0.9328, 0.9341,
					0.9339, 0.9366, 0.9387, 0.9406, 0.9405,
					0.9404, 0.9405, 0.9414, 0.9455, 0.9369, 0.9289
				}
			},
			{	/* Black */
				36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
				1.0,				/* Scale factor */
				{
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000,
					0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000
				}
			},
			{
				{
					"30% Y",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.3538, 0.3632, 0.3888, 0.5058, 0.7111,
							0.8221, 0.8512, 0.8334, 0.8251, 0.8392,
							0.8609, 0.8832, 0.9013, 0.9113, 0.9157,
							0.9137, 0.9128, 0.9113, 0.9086, 0.9111,
							0.9103, 0.9094, 0.9091, 0.9083, 0.9085,
							0.9080, 0.9091, 0.9104, 0.9118, 0.9104,
							0.9099, 0.9099, 0.9094, 0.9111, 0.9054, 0.8793
						}
					}
				},
				{
					"60% Y",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.2220, 0.2209, 0.2299, 0.2901, 0.4010,
							0.4716, 0.5092, 0.5337, 0.5732, 0.6337,
							0.7095, 0.7848, 0.8422, 0.8755, 0.8907,
							0.8933, 0.8938, 0.8927, 0.8909, 0.8929,
							0.8922, 0.8912, 0.8897, 0.8902, 0.8900,
							0.8893, 0.8896, 0.8915, 0.8932, 0.8918,
							0.8921, 0.8915, 0.8921, 0.8939, 0.8825, 0.8575
						}
					}
				},
				{
					"90% Y",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0326, 0.0287, 0.0269, 0.0295, 0.0376,
							0.0469, 0.0605, 0.0862, 0.1343, 0.2162,
							0.3503, 0.5257, 0.6862, 0.7889, 0.8420,
							0.8632, 0.8719, 0.8753, 0.8753, 0.8784,
							0.8776, 0.8778, 0.8778, 0.8783, 0.8775,
							0.8772, 0.8777, 0.8791, 0.8808, 0.8793,
							0.8784, 0.8770, 0.8758, 0.8757, 0.8626, 0.8106
						}
					}
				},
				{
					"30% K",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.3243, 0.3369, 0.3635, 0.4708, 0.6532,
							0.7439, 0.7542, 0.7167, 0.6839, 0.6673,
							0.6546, 0.6454, 0.6412, 0.6387, 0.6380,
							0.6357, 0.6351, 0.6341, 0.6334, 0.6363,
							0.6367, 0.6388, 0.6400, 0.6431, 0.6465,
							0.6491, 0.6532, 0.6591, 0.6656, 0.6699,
							0.6743, 0.6778, 0.6816, 0.6859, 0.6815, 0.6758
						}
					}
				},
				{
					"60% K",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.1707, 0.1772, 0.1895, 0.2374, 0.3174,
							0.3566, 0.3616, 0.3464, 0.3329, 0.3272,
							0.3228, 0.3201, 0.3196, 0.3198, 0.3208,
							0.3203, 0.3207, 0.3217, 0.3221, 0.3249,
							0.3269, 0.3295, 0.3339, 0.3375, 0.3427,
							0.3483, 0.3545, 0.3635, 0.3726, 0.3808,
							0.3881, 0.3949, 0.4003, 0.4053, 0.4052, 0.3962
						}
					}
				},
				{
					"90% K",
					{
						36, 380.0, 730.0,	/* 36 bands from 380 to 730 in 10nm steps */
						1.0,				/* Scale factor */
						{
							0.0206, 0.0205, 0.0202, 0.0205, 0.0213,
							0.0217, 0.0210, 0.0207, 0.0202, 0.0199,
							0.0198, 0.0197, 0.0197, 0.0195, 0.0196,
							0.0193, 0.0192, 0.0194, 0.0194, 0.0194,
							0.0197, 0.0198, 0.0204, 0.0214, 0.0223,
							0.0236, 0.0242, 0.0265, 0.0283, 0.0306,
							0.0330, 0.0347, 0.0370, 0.0389, 0.0394, 0.0383
						}
					}
				}
			}
		}
	}
};

/* ---------------------------------------------------------- */

int
main(void) {
	int m, ps, ss, pn;
#ifdef DATAFILE
	FILE *df;
#endif

	printf("Hi there\n");

#ifdef DATAFILE
	if ((df = fopen("spectest.dat", "w")) == NULL)
		error ("Unable to open data file '%s'","spectest.dat");
#endif
	/* For each material and illuminant */
	for (m = 0; m < 2; m++) {

		printf("Material '%s'\n", matilum[m][0].media);

		/* For each light source as primary (target/check), other as secondary (instrument) */
		for (ps = 0; ps < 2; ps++) {
			xsp2cie *pcon, *scon;		/* Conversions */
			xspect pill, sill;			/* Illuminants */

			ss = 1 - ps;	/* Opposite */
			
#ifdef FILTERED_D65
			if (matilum[m][ps].ill == icxIT_D65)
				pill = il_sod65;
			else
#endif
				standardIlluminant(&pill, matilum[m][ps].ill, 0);		/* Target/check */

#ifdef FILTERED_D65
			if (matilum[m][ss].ill == icxIT_D65)
				sill = il_sod65;
			else
#endif
				standardIlluminant(&sill, matilum[m][ss].ill, 0);		/* Instrument */

			/* Create two conversions for the target/check illuminant */
			if ((pcon = new_xsp2cie(icxIT_custom, &pill, icxOT_Shaw_Fairchild_2,
			                       NULL, icSigLabData, 1)) == NULL)
				error ("Creating conversion failed");

			if ((scon = new_xsp2cie(icxIT_custom, &pill, icxOT_Shaw_Fairchild_2,
			                       NULL, icSigLabData, 1)) == NULL)
				error ("Creating conversion failed");

			/* Tell the secondary conversion to allow for instrument illuminant */
			if (scon->set_fwa(scon, &sill, NULL, &matilum[m][ss].white))
				error ("Setting FWA compensation failed");

			printf("Primary (Target/Check) '%s', Secondary (Instrument)'%s'\n",
			matilum[m][ps].illum, matilum[m][ss].illum);

			/* For each test patch */
			for (pn = 0; pn < 6; pn++) {
				int i, j;
				double plab[3];
				double slab[3];
				double sclab[3];
				xspect psp;			/* Reference spectrum */
				xspect ssp;			/* un-compensated spectrum */
				xspect scsp;		/* Compensated spectrum */
				double de, cde;
#ifdef DOPLOT
#define	XRES 400
				double xx[XRES];
				double y1[XRES];
				double y2[XRES];
				double y3[XRES];
#endif /* DOPLOT */

	
				/* Compute reference value for target illuminant */
				pcon->sconvert(pcon, &psp, plab, &matilum[m][ps].patches[pn].s);

				/* Compute uncompensated value for target illuminant */
				pcon->sconvert(pcon, &ssp, slab, &matilum[m][ss].patches[pn].s);

				/* Compute compensated value for target illuminant */
				scon->sconvert(scon, &scsp, sclab, &matilum[m][ss].patches[pn].s);

				de = 0.0;
				for (j = 0; j < 3; j++) {
					double tt = plab[j] - slab[j];
					de += tt * tt;
				}
				de = sqrt(de);

				printf("Patch '%s', Ref %f %f %f,  Other %f %f %f DE %f\n",
				matilum[m][ps].patches[pn].pdesc, plab[0], plab[1], plab[2],
				slab[0], slab[1], slab[2], de);

				cde = 0.0;
				for (j = 0; j < 3; j++) {
					double tt = plab[j] - sclab[j];
					cde += tt * tt;
				}
				cde = sqrt(cde);

				printf("Patch '%s', Ref %f %f %f, COther %f %f %f DE %f\n",
				matilum[m][ps].patches[pn].pdesc, plab[0], plab[1], plab[2],
				sclab[0], sclab[1], sclab[2], cde);
				printf("DE change %f\n",cde - de);
				printf("\n");

#ifdef DOPLOT
				for (i = 0; i < psp.spec_n; i++) {
					double ww;

					ww = (psp.spec_wl_long - psp.spec_wl_short)
					   * ((double)i/(psp.spec_n-1.0)) + psp.spec_wl_short;
	
					xx[i] = ww;
					y1[i] = ssp.spec[i];	/* Black - Input */
					y2[i] = scsp.spec[i];	/* Red - Estimate */
					y3[i] = psp.spec[i];	/* Green - Target */
				}
				do_plot(xx,y1,y2,y3,i);
#endif /* DOPLOT */
#ifdef DATAFILE
				fprintf(df,"\nPrimary (Target/Check) '%s', Secondary (Instrument)'%s'\n",
				        matilum[m][ps].illum, matilum[m][ss].illum);
				fprintf(df,"Patch '%s', Ref %f %f %f,  Other %f %f %f DE %f\n",
				        matilum[m][ps].patches[pn].pdesc, plab[0], plab[1], plab[2],
				        slab[0], slab[1], slab[2], de);
				fprintf(df,"Patch '%s', Ref %f %f %f, COther %f %f %f DE %f\n",
				        matilum[m][ps].patches[pn].pdesc, plab[0], plab[1], plab[2],
				        sclab[0], sclab[1], sclab[2], cde);
				fprintf(df,"NM	Ref.	Un-comp.	Comp.\n");
				for (i = 0; i < psp.spec_n; i++) {
					double ww;

					ww = (psp.spec_wl_long - psp.spec_wl_short)
					   * ((double)i/(psp.spec_n-1.0)) + psp.spec_wl_short;
	
					fprintf(df,"%d	%f	%f	%f\n",((int)ww), psp.spec[i], ssp.spec[i], scsp.spec[i]);
				}
#endif /* DATAFILE */
			}
			pcon->del(pcon);
			scon->del(scon);
		}
	}
#ifdef DATAFILE
	fclose(df);
#endif

	return 0;
}








