#include "libmpdclient.h"

#include "gbemol-mpd.h"
#include "gbemol-marshal.h"

/* Properties */
enum {
	GBEMOL_MPD_HOST = 1,    
	GBEMOL_MPD_PASS,         	
	GBEMOL_MPD_PORT,          
	GBEMOL_MPD_TIMEOUT	
};

/* Signals */
typedef enum {
	STATE_CHANGED_SIGNAL,
	SONG_CHANGED_SIGNAL,
	PLAYLIST_CHANGED_SIGNAL,
	REFRESH_SIGNAL,
	ERROR_SIGNAL,
	LAST_SIGNAL
} GbemolMpdSignalType;

struct _GbemolMpdPrivate {
	mpd_Connection *conn;  

	gchar *host;		/* MPD Host */
	gchar *pass;		/* MPD Pass */
	guint port;		/* MPD TCP Port */

	float timeout;		/* Timeout, in seconds */

	gboolean connected;     /* Connected to the daemon? */

	int error;		/* Error code, 0 for no error */

	GList* not_commands;	/* List of not allowed commands */
};

static guint gbemol_mpd_signals [LAST_SIGNAL] = { 0 };

static void gbemol_mpd_class_init (GObjectClass *g_class);
static void gbemol_mpd_init (GbemolMpd *obj);
static void gbemol_mpd_finalize (GObject *object);
static gboolean gbemol_mpd_refresh (GbemolMpd *obj);
static void gbemol_mpd_get_not_commands_list (GbemolMpd *obj);
gboolean gbemol_mpd_check_permission (GbemolMpd *obj, gchar *command);

/* Utils */
static gboolean gbemol_mpd_finish_and_check (GbemolMpd *obj);
static gboolean gbemol_mpd_finish_and_handle (GbemolMpd *obj);


/* Finishes a command and checks if an error has ocurred, true if yes */

GType
gbemol_mpd_get_type (void)
{
	static GType type = 0;
	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (GbemolMpdClass),
			NULL,   /* base_init */
			NULL,   /* base_finalize */
			(GClassInitFunc)gbemol_mpd_class_init,   /* class_init */
			NULL,   /* class_finalize */
			NULL,   /* class_data */
			sizeof (GbemolMpd),
			0,      /* n_preallocs */
			(GInstanceInitFunc)gbemol_mpd_init    /* instance_init */
		};

		type = g_type_register_static (G_TYPE_OBJECT,
				"GbemolMpd",
				&info, 0);
	}
	return type;
}


static void
gbemol_mpd_init (GbemolMpd *obj)               
{
	obj->priv = g_new0 (GbemolMpdPrivate, 1);

	/* Default values */
	obj->status = NULL;

	obj->priv->host = g_strdup ("localhost");
	obj->priv->pass = NULL;
	obj->priv->port = 6600;
	obj->priv->timeout = 1.0;

	obj->priv->connected = FALSE;

	obj->priv->conn = NULL;

	obj->priv->not_commands = NULL;
}

static void
gbemol_mpd_set_property (GObject      *object,
                        guint         property_id,
                        const GValue *value,
                        GParamSpec   *pspec)
{
	GbemolMpd *obj = GBEMOL_MPD (object);

	switch (property_id)
	{
		case GBEMOL_MPD_HOST:
			g_free (obj->priv->host);
			obj->priv->host = g_value_dup_string (value);
			break;
		case GBEMOL_MPD_PASS:
			g_free (obj->priv->pass);
			obj->priv->pass = g_value_dup_string (value);
			break;
		case GBEMOL_MPD_PORT:
			obj->priv->port = g_value_get_int (value);
			break;
		case GBEMOL_MPD_TIMEOUT:
			obj->priv->timeout = g_value_get_float (value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    			break;
	}
}

static void
gbemol_mpd_get_property (GObject      *object,
                        guint         property_id,
                        GValue *value,
                        GParamSpec   *pspec)
{
	GbemolMpd *obj = GBEMOL_MPD (object);

	switch (property_id)
	{
		case GBEMOL_MPD_HOST:
			g_value_set_string (value, obj->priv->host);
			break;
		case GBEMOL_MPD_PASS:
			g_value_set_string (value, obj->priv->pass);
			break;
		case GBEMOL_MPD_PORT:
			g_value_set_int (value, obj->priv->port);
			break;
		case GBEMOL_MPD_TIMEOUT:
			g_value_set_float (value, obj->priv->timeout);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    			break;
	}
}

static void
gbemol_mpd_class_init (GObjectClass *g_class)
{
	GParamSpec *pspec;

  	g_class->set_property = gbemol_mpd_set_property;
  	g_class->get_property = gbemol_mpd_get_property;
	g_class->finalize = gbemol_mpd_finalize;

	pspec = g_param_spec_string ("host",
                               "MPD Host",
                               "Set MPD server host name",
                               "localhost" /* default value */,
                               G_PARAM_READWRITE);

	g_object_class_install_property (g_class,
                                   GBEMOL_MPD_HOST,
                                   pspec);

	pspec = g_param_spec_string ("pass",
                               "MPD Password",
                               "Set MPD server password",
                               NULL /* default value */,
                               G_PARAM_READWRITE);

	g_object_class_install_property (g_class,
                                   GBEMOL_MPD_PASS,
                                   pspec);

	pspec = g_param_spec_int ("port",
                               "MPD Port",
                               "Set MPD TCP/IP port to use",
                               0,
			       65535,
			       6600,
			       G_PARAM_READWRITE);

	g_object_class_install_property (g_class,
                                   GBEMOL_MPD_PORT,
                                   pspec);

	pspec = g_param_spec_float ("timeout",
                               "MPD Timeout",
                               "Set MPD connection timeout",
                               0,
			       65535,
			       1.0,
			       G_PARAM_READWRITE);

	g_object_class_install_property (g_class,
                                   GBEMOL_MPD_TIMEOUT,
                                   pspec);	

	/* Signals */
	gbemol_mpd_signals [STATE_CHANGED_SIGNAL] = 
		g_signal_new ("state_changed",
			      G_TYPE_FROM_CLASS (g_class),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, 
			      NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 
			      0);

	gbemol_mpd_signals [SONG_CHANGED_SIGNAL] = 
		g_signal_new ("song_changed",
			      G_TYPE_FROM_CLASS (g_class),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, 
			      NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 
			      0);
	gbemol_mpd_signals [PLAYLIST_CHANGED_SIGNAL] = 
		g_signal_new ("playlist_changed",
			      G_TYPE_FROM_CLASS (g_class),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, 
			      NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 
			      0);
	gbemol_mpd_signals [REFRESH_SIGNAL] = 
		g_signal_new ("refresh",
			      G_TYPE_FROM_CLASS (g_class),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, 
			      NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 
			      0);
		gbemol_mpd_signals [ERROR_SIGNAL] = 
		g_signal_new ("error",
			      G_TYPE_FROM_CLASS (g_class),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, 
			      NULL,
			      gbemol_cclosure_VOID__INT_INT_STRING,
			      G_TYPE_NONE, 
			      3,
			      G_TYPE_INT,
			      G_TYPE_INT,
			      G_TYPE_STRING);

}

static GbemolMpdStatus* gbemol_mpd_get_status (GbemolMpd* obj)
{
	if (!gbemol_mpd_check_permission (obj, "status"))
		return NULL;

	mpd_sendCurrentSongCommand (obj->priv->conn);
	
	if (gbemol_mpd_finish_and_handle (obj))
		return NULL;

	mpd_sendStatusCommand (obj->priv->conn);
	return mpd_getStatus (obj->priv->conn);
}



static void
gbemol_mpd_finalize (GObject *obj)
{
	GbemolMpd *mpd = GBEMOL_MPD (obj);

	mpd_closeConnection (mpd->priv->conn);
	g_free (mpd->priv->host);
	g_free (mpd->priv->pass);

	g_free (GBEMOL_MPD (obj)->priv);
}

static gboolean gbemol_mpd_refresh (GbemolMpd *obj)
{
	GbemolMpdStatus* status;
	gboolean signals [LAST_SIGNAL] = { FALSE }; /* Signals to be emitted */
	int i;

	if (!gbemol_mpd_check_permission (obj, "status"))
		return TRUE;

	status = gbemol_mpd_get_status (obj);

	signals [REFRESH_SIGNAL] = TRUE;

	if (status)
	{
		/* If there wasn't a status before, update everything */
		if (!obj->status)
		{
			signals [SONG_CHANGED_SIGNAL] = TRUE;
			signals [PLAYLIST_CHANGED_SIGNAL] = TRUE;
			signals [STATE_CHANGED_SIGNAL] = TRUE;
		}
		else
		{			
			/* Check if song has changed */
			if ((status->song < 0) || (status->songid != obj->status->songid))
				signals [SONG_CHANGED_SIGNAL] = TRUE;
		
			/* Check if playlist has changed */
			if (status->playlist != obj->status->playlist)
				signals [PLAYLIST_CHANGED_SIGNAL] = TRUE;
			
			/* State has changed? */
			if (status->state != obj->status->state)
				signals [STATE_CHANGED_SIGNAL] = TRUE;

			mpd_freeStatus (obj->status);
		}
	}

	obj->status = status;

	/* Emit the signals */
	for (i = 0; i < LAST_SIGNAL; i++)
		if (signals[i])
			g_signal_emit (obj, gbemol_mpd_signals [i], 0, NULL);

	return TRUE;

}

GbemolMpd* gbemol_mpd_new (const gchar *host, const gchar *pass, int port, float timeout)
{
	GbemolMpd *obj;

	obj = gbemol_mpd_new_with_defaults ();
	
	g_object_set (G_OBJECT (obj), 
			"host", host,
			"pass", pass,
			"port", port,
			"timeout", timeout,
			NULL);
	return obj;
}

GbemolMpd*
gbemol_mpd_new_with_defaults (void)
{
	GbemolMpd* obj;

	obj = GBEMOL_MPD (g_object_new (GBEMOL_TYPE_MPD, NULL));
	
	/* Add refresh function to the main loop */
	g_timeout_add (500, (GSourceFunc) gbemol_mpd_refresh, (gpointer) obj);

	return obj;
}

gboolean gbemol_mpd_is_connected (GbemolMpd *obj)
{
	return obj->priv->connected;
}

gchar* gbemol_mpd_get_version (GbemolMpd* obj)
{
	gchar* str;

	str = g_strdup_printf ("%d.%d.%d", obj->priv->conn->version[0], obj->priv->conn->version[1], 
			obj->priv->conn->version[2]);
	return str;
}

void gbemol_mpd_free_song (GbemolMpdSong *song)
{
	if (song)
		mpd_freeSong (song);
	song = NULL;
}

void gbemol_mpd_free_song_list (GList *songs)
{
	if (songs)
	{
		g_list_foreach (songs, (GFunc) mpd_freeSong, NULL);
		g_list_free (songs);
	}
	songs = NULL;
}

void gbemol_mpd_free_char_list (GList *l)
{
	if (l)
	{
		g_list_foreach (l, (GFunc) g_free, NULL);
		g_list_free (l);
	}
	l = NULL;
}

void gbemol_mpd_free_output_list (GList *l)
{
	if (l)
	{
		g_list_foreach (l, (GFunc) mpd_freeOutputElement, NULL);
		g_list_free (l);
	}
	l = NULL;
}

/* Property related methods */
void gbemol_mpd_set_host (GbemolMpd *obj, const gchar *host)
{
	g_object_set (G_OBJECT (obj), "host", host, NULL);
}

void gbemol_mpd_set_pass (GbemolMpd *obj, const gchar *pass)
{
	g_object_set (G_OBJECT (obj), "pass", pass, NULL);
}

void gbemol_mpd_set_port (GbemolMpd *obj, guint port)
{
	g_object_set (G_OBJECT (obj), "port", port, NULL);
}

void gbemol_mpd_set_timeout (GbemolMpd *obj, gfloat timeout)
{
	g_object_set (G_OBJECT (obj), "timeout", timeout, NULL);
}

gboolean gbemol_mpd_connect (GbemolMpd* obj)
{
	if (obj->priv->conn)
		mpd_closeConnection (obj->priv->conn);

	obj->priv->conn = mpd_newConnection (obj->priv->host, obj->priv->port, obj->priv->timeout);
	
	if (gbemol_mpd_finish_and_handle (obj))
		obj->priv->connected = FALSE;
	else
	{
		gbemol_mpd_get_not_commands_list (obj);
		if (obj->status)
			mpd_freeStatus (obj->status);
		obj->status = gbemol_mpd_get_status (obj);
		obj->priv->connected = TRUE;
		/* Emit a song-changed signal */
		g_signal_emit (obj, gbemol_mpd_signals [SONG_CHANGED_SIGNAL], 0, NULL);
	}

	return obj->priv->connected;
}

/* Connect */
gboolean gbemol_mpd_connect_and_authenticate (GbemolMpd* obj)
{
	if (obj->priv->conn)
		mpd_closeConnection (obj->priv->conn);

	obj->priv->conn = mpd_newConnection (obj->priv->host, obj->priv->port, obj->priv->timeout);
	
	if (gbemol_mpd_finish_and_handle (obj))
	{
		obj->priv->connected = FALSE;
		return FALSE;
	}
	else
		obj->priv->connected = TRUE;

	/* Not authenticated, yet connected, so return TRUE, but mark as disconnected */
	if (!gbemol_mpd_authenticate (obj))
		obj->priv->connected = FALSE;
	else
	{
		gbemol_mpd_get_not_commands_list (obj);
		if (obj->status)
			mpd_freeStatus (obj->status);
		obj->status = gbemol_mpd_get_status (obj);
		/* Emit a song-changed signal */
		g_signal_emit (obj, gbemol_mpd_signals [SONG_CHANGED_SIGNAL], 0, NULL);
	}

	return TRUE;
}

void gbemol_mpd_disconnect (GbemolMpd *obj)
{
	if (obj->priv->not_commands)
		gbemol_mpd_free_char_list (obj->priv->not_commands);
	obj->priv->not_commands = NULL;

	obj->priv->connected = FALSE;
	if (obj->priv->conn)
		mpd_closeConnection (obj->priv->conn);
	obj->priv->conn = NULL;
}

gboolean gbemol_mpd_authenticate (GbemolMpd *obj)
{
	mpd_sendPasswordCommand (obj->priv->conn, obj->priv->pass);
	return !gbemol_mpd_finish_and_handle (obj);
}


/*
 * Get the list of not allowed commands
 */
static void gbemol_mpd_get_not_commands_list (GbemolMpd *obj)
{
	gchar* command;

	/* if there's a previous list, free it first */
	mpd_sendNotCommandsCommand (obj->priv->conn);
	while ((command = mpd_getNextCommand (obj->priv->conn)))
		obj->priv->not_commands = g_list_append (obj->priv->not_commands, command);

	gbemol_mpd_finish_and_handle (obj);
}

/*
 * Finish a command and checks for errors, emits a signal for the error
 */
gboolean gbemol_mpd_finish_and_handle (GbemolMpd* obj)
{	
	if (gbemol_mpd_is_connected (obj))
		mpd_finishCommand (obj->priv->conn);

	if (obj->priv->conn->error)
	{
		switch (obj->priv->conn->error)
		{
			case MPD_ERROR_CONNCLOSED:
				obj->priv->connected = FALSE;
				break;
			case MPD_ERROR_TIMEOUT:
				obj->priv->connected = FALSE;
				break;
			case MPD_ERROR_UNKHOST:
				obj->priv->connected = FALSE;
				break;
			case MPD_ERROR_NORESPONSE:
				obj->priv->connected = FALSE;
				break;
			case MPD_ERROR_CONNPORT:
				obj->priv->connected = FALSE;
				break;
			case MPD_ERROR_NOTMPD:
				obj->priv->connected = FALSE;
				break;
			case MPD_ERROR_SYSTEM:
				obj->priv->connected = FALSE;
				break;
		}
		g_signal_emit (obj, gbemol_mpd_signals [ERROR_SIGNAL], 0, 
				obj->priv->conn->error, obj->priv->conn->errorCode, obj->priv->conn->errorStr, NULL);
		mpd_clearError (obj->priv->conn);
		return TRUE;
	}
	else
	{
		mpd_finishCommand (obj->priv->conn);
		return FALSE;
	}
}

/* 
 * Finishes a command and return TRUE if there are no errors, FALSE if there are
 * but doesn't treat them
 */
gboolean gbemol_mpd_finish_and_check (GbemolMpd *obj)
{
	mpd_finishCommand (obj->priv->conn);

	if (obj->priv->conn->error)
		return FALSE;
	else
		return TRUE;
}

/*
 * Check if user is allowed to execute command
 */
gboolean gbemol_mpd_check_permission (GbemolMpd *obj, gchar *command)
{
	GList* l = g_list_first(obj->priv->not_commands);

	if (!gbemol_mpd_is_connected (obj))
		return FALSE;

	while ((l = g_list_next (l)))
	{
		if (g_str_equal ((gchar *) l->data, command))
		{
			g_message ("DENIED: %s", (gchar *) l->data);
			return FALSE;
		}
	}
	return TRUE;	
}

void gbemol_mpd_player_next (GbemolMpd *obj)
{
	if (!gbemol_mpd_check_permission (obj, "next"))
		return;

	mpd_sendNextCommand (obj->priv->conn);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_player_previous (GbemolMpd *obj)
{
	if (!gbemol_mpd_check_permission (obj, "previous"))
			return;

	mpd_sendPrevCommand (obj->priv->conn);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_player_stop (GbemolMpd *obj)
{
	if (!gbemol_mpd_check_permission (obj, "stop"))
			return;

	mpd_sendStopCommand (obj->priv->conn);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_player_pause (GbemolMpd *obj)
{
	if (!gbemol_mpd_check_permission (obj, "pause"))
			return;

	if (obj->status->state == MPD_STATUS_STATE_PLAY)
		mpd_sendPauseCommand (obj->priv->conn, 1);
	else
		mpd_sendPauseCommand (obj->priv->conn, 0);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_player_play_song_by_id (GbemolMpd *obj, int id)
{

	if (!gbemol_mpd_check_permission (obj, "playid"))
			return;

	mpd_sendPlayIdCommand (obj->priv->conn, id);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_player_play_song_by_pos (GbemolMpd *obj, int pos)
{
	if (!gbemol_mpd_check_permission (obj, "play"))
		return;

	mpd_sendPlayCommand (obj->priv->conn, pos);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_set_random (GbemolMpd *obj, gboolean random)
{
	if (!gbemol_mpd_check_permission (obj, "random"))
		return;

	mpd_sendRandomCommand (obj->priv->conn, random);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_set_repeat (GbemolMpd *obj, gboolean repeat)
{
	if (!gbemol_mpd_check_permission (obj, "repeat"))
			return;

	mpd_sendRepeatCommand (obj->priv->conn, repeat);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_seek (GbemolMpd *obj, int id, int sec)
{
	if (!gbemol_mpd_check_permission (obj, "seekid"))
			return;

	mpd_sendSeekIdCommand (obj->priv->conn, id, sec);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_crossfade (GbemolMpd *obj, int fade)
{
	if (!gbemol_mpd_check_permission (obj, "crossfade"))
		return;

	mpd_sendCrossfadeCommand (obj->priv->conn, fade);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_set_volume (GbemolMpd *obj, int volume)
{
	if (!gbemol_mpd_check_permission (obj, "setvol"))
		return;
	mpd_sendSetvolCommand (obj->priv->conn, volume);
	gbemol_mpd_finish_and_handle (obj);
}

GbemolMpdSong*
gbemol_mpd_get_current_song (GbemolMpd* obj)
{
	if (!gbemol_mpd_check_permission (obj, "status") || !obj->status)
		return NULL;
		
	if (obj->status->state == MPD_STATUS_STATE_STOP)
		return NULL;

	if (obj->status->songid >= 0)
		return gbemol_mpd_playlist_get_song_by_id (obj, obj->status->songid);
	return NULL;
}

GList* gbemol_mpd_playlist_get_songs (GbemolMpd *obj)
{
	GList* songs = NULL;
	GbemolMpdInfo *info;
	GbemolMpdSong *song;

	if (!gbemol_mpd_check_permission (obj, "playlistinfo"))
			return NULL;

	mpd_sendPlaylistInfoCommand (obj->priv->conn, -1);

	while ((info = mpd_getNextInfoEntity (obj->priv->conn)))
	{
		if (info->type == MPD_INFO_ENTITY_TYPE_SONG)
		{
			song = mpd_songDup (info->info.song);
			songs = g_list_prepend (songs, (gpointer) song);
		}
		mpd_freeInfoEntity (info);
	}

	if (gbemol_mpd_finish_and_handle (obj))
		return NULL;
	else
		return g_list_reverse (songs);
}

GbemolMpdSong* gbemol_mpd_playlist_get_song_by_id (GbemolMpd *obj, gint id)
{
	GbemolMpdInfo *info;
	GbemolMpdSong *song;

	if ((id < 0) || !gbemol_mpd_check_permission (obj, "playlistid"))
		return NULL;

	mpd_sendPlaylistIdCommand (obj->priv->conn, id);
	info = mpd_getNextInfoEntity (obj->priv->conn);
	
	if (gbemol_mpd_finish_and_handle (obj) || !info)
		return NULL;

	if (info->type == MPD_INFO_ENTITY_TYPE_SONG)
	{
		song = info->info.song;
		info->info.song = NULL;
	} else
		song = NULL;

	mpd_freeInfoEntity (info);

	return song;			
}

/*
 * Return the current selected song (referenced by status->song)
 */
GbemolMpdSong* gbemol_mpd_playlist_get_selected_song (GbemolMpd *obj)
{
	GbemolMpdSong *song = NULL;
	GbemolMpdInfo *info;

	if (!gbemol_mpd_check_permission (obj, "status"))
			return NULL;

	mpd_sendCurrentSongCommand (obj->priv->conn);
	info = mpd_getNextInfoEntity (obj->priv->conn);

	if (info && (info->type == MPD_INFO_ENTITY_TYPE_SONG))
		song = mpd_songDup (info->info.song);

	mpd_freeInfoEntity (info);

	gbemol_mpd_finish_and_handle (obj);

	return song;
}
	
void gbemol_mpd_playlist_clear (GbemolMpd *obj)
{
	if (!gbemol_mpd_check_permission (obj, "clear"))
		return;

	mpd_sendClearCommand (obj->priv->conn);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_playlist_shuffle (GbemolMpd *obj)
{
	if (!gbemol_mpd_check_permission (obj, "shuffle"))
		return;

	mpd_sendShuffleCommand (obj->priv->conn);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_playlist_remove_song_by_id (GbemolMpd *obj, gint id)
{
	if (!gbemol_mpd_check_permission (obj, "deleteid"))
		return;

	mpd_sendDeleteIdCommand (obj->priv->conn, id);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_playlist_remove_song_by_pos (GbemolMpd *obj, gint pos)
{
	if (!gbemol_mpd_check_permission (obj, "delete"))
			return;

	mpd_sendDeleteCommand (obj->priv->conn, pos);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_playlist_add_song (GbemolMpd* obj, gchar* path)
{
	if (!gbemol_mpd_check_permission (obj, "add"))
		return;

	mpd_sendAddCommand (obj->priv->conn, path);
	gbemol_mpd_finish_and_handle (obj);
}

/* 
 * TRUE if the song is actually a stream 
 */
gboolean gbemol_mpd_song_is_stream (GbemolMpdSong* song)
{
	if (song->name && !song->artist && (song->time == MPD_SONG_NO_TIME))
		return TRUE;
	else
		return FALSE;
}

GList* gbemol_mpd_preview_playlist (GbemolMpd *obj, gchar *path)
{
	GList *songs = NULL;
	GbemolMpdSong* song;
	GbemolMpdInfo* info;

	if (!gbemol_mpd_check_permission (obj, "listplaylistinfo"))
			return NULL;

	mpd_sendListPlaylistInfoCommand (obj->priv->conn, path);

	while ((info = mpd_getNextInfoEntity (obj->priv->conn)))
	{
		if (info->type == MPD_INFO_ENTITY_TYPE_SONG)
		{
			song = info->info.song;
			info->info.song = NULL;
			songs = g_list_prepend (songs, song);
		}
		mpd_freeInfoEntity (info);
	}

	if (gbemol_mpd_finish_and_handle (obj))
		return NULL;
	else
		return g_list_reverse (songs);

}

void gbemol_mpd_load_playlist (GbemolMpd *obj, gchar *playlist)
{
	if (!gbemol_mpd_check_permission (obj, "load"))
		return;

	mpd_sendLoadCommand (obj->priv->conn, playlist);
	gbemol_mpd_finish_and_handle (obj);
}

void gbemol_mpd_delete_playlist (GbemolMpd *obj, gchar *path)
{
	if (!gbemol_mpd_check_permission (obj, "rm"))
		return;

	mpd_sendRmCommand (obj->priv->conn, path);
	gbemol_mpd_finish_and_handle (obj);
}

gboolean gbemol_mpd_save_playlist (GbemolMpd *obj, gchar *path)
{
	if (!gbemol_mpd_check_permission (obj, "save"))
		return FALSE;

	mpd_sendSaveCommand (obj->priv->conn, path);
	if (!gbemol_mpd_finish_and_check (obj))
	{
		/* Playlist already exists (probably) */
		gbemol_mpd_delete_playlist (obj, path);
		mpd_sendSaveCommand (obj->priv->conn, path);
		return gbemol_mpd_finish_and_handle (obj);
	}

	return TRUE;
}

GList* gbemol_mpd_get_playlists (GbemolMpd *obj)
{
	GList* pls = NULL;
	GbemolMpdInfo *info;
	gchar *list;

	if (!gbemol_mpd_check_permission (obj, "lsinfo"))
			return NULL;

	mpd_sendLsInfoCommand (obj->priv->conn, "/");

	while ((info = mpd_getNextInfoEntity (obj->priv->conn)))
	{
		if (info->type == MPD_INFO_ENTITY_TYPE_PLAYLISTFILE)
		{
			list = g_strdup (info->info.playlistFile->path);
			pls = g_list_prepend (pls, (gpointer) list);
		}
		mpd_freeInfoEntity (info);
	}

	if (gbemol_mpd_finish_and_handle (obj))
		return NULL;
	else
		return g_list_reverse (pls);
}

/*
 * Begin a queue of commands
 */
void gbemol_mpd_queue_start (GbemolMpd* obj)
{
	if (!gbemol_mpd_is_connected (obj))
		return;

	mpd_sendCommandListOkBegin (obj->priv->conn);
}

/*
 * Execute the queue
 */
void gbemol_mpd_queue_finish (GbemolMpd* obj)
{
	if (!gbemol_mpd_is_connected (obj))
		return;

	mpd_sendCommandListEnd (obj->priv->conn);
	gbemol_mpd_finish_and_handle (obj);
}

/*
 * Queue removing a song
 */
void gbemol_mpd_queue_remove_song (GbemolMpd* obj, gint id)
{
	if (!gbemol_mpd_check_permission (obj, "deleteid"))
		return;

	mpd_sendDeleteIdCommand (obj->priv->conn, id);
}

void gbemol_mpd_queue_add_song (GbemolMpd* obj, gchar* path)
{
	if (!gbemol_mpd_check_permission (obj, "add"))
		return;

	mpd_sendAddCommand (obj->priv->conn, path);
}

/*
 * Start field search
 */
void gbemol_mpd_search_field_start (GbemolMpd* obj, gint field)
{
	if (!gbemol_mpd_is_connected (obj))
		return;

	mpd_startFieldSearch (obj->priv->conn, field);
}

void gbemol_mpd_search_start (GbemolMpd* obj)
{
	if (!gbemol_mpd_is_connected (obj))
		return;

	mpd_startSearch (obj->priv->conn, TRUE);
}


/*
 * Add constraint to the search
 */
void gbemol_mpd_search_add_constraint (GbemolMpd* obj, gint tag, gchar* value)
{
	if (!gbemol_mpd_is_connected (obj))
		return;

	mpd_addConstraintSearch (obj->priv->conn, tag, value);
}

/*
 * Get the search results
 */
GList* gbemol_mpd_search_field_get_results (GbemolMpd* obj, gint tag)
{
	GList* results = NULL;
	gchar* str;

	if (!gbemol_mpd_is_connected (obj))
		return NULL;

	mpd_commitSearch (obj->priv->conn);
	while ((str = mpd_getNextTag (obj->priv->conn, tag)) != NULL)
		results = g_list_append (results, str);

	return results;
}

GList* gbemol_mpd_search_get_results (GbemolMpd* obj)
{
	GList* results = NULL;
	GbemolMpdInfo *info;
	GbemolMpdSong *song;

	if (!gbemol_mpd_is_connected (obj))
		return NULL;

	gbemol_mpd_search_commit (obj);

	while ((info = mpd_getNextInfoEntity (obj->priv->conn)) != NULL)
	{
		if (info->type == MPD_INFO_ENTITY_TYPE_SONG)
		{
			song = mpd_songDup (info->info.song);
			results = g_list_append (results, song);
		}
		mpd_freeInfoEntity (info);
	}

	return results;
}

void gbemol_mpd_search_commit (GbemolMpd* obj)
{
	if (!gbemol_mpd_is_connected (obj))
		return;

	mpd_commitSearch (obj->priv->conn);
}

GList* gbemol_mpd_database_get_all_songs (GbemolMpd* obj)
{
	GList* results = NULL;
	GbemolMpdInfo *info;
	GbemolMpdSong *song;

	if (!gbemol_mpd_is_connected (obj))
		return NULL;

	mpd_sendListallInfoCommand (obj->priv->conn, "/");

	while ((info = mpd_getNextInfoEntity (obj->priv->conn)) != NULL)
	{
		if (info->type == MPD_INFO_ENTITY_TYPE_SONG)
		{
			song = mpd_songDup (info->info.song);
			results = g_list_append (results, song);
		}
		mpd_freeInfoEntity (info);
	}

	if (gbemol_mpd_finish_and_handle (obj))
		return NULL;
	else
		return results;
}

void gbemol_mpd_database_update (GbemolMpd* obj, gchar *path)
{
	mpd_sendUpdateCommand (obj->priv->conn, path);
	gbemol_mpd_finish_and_handle (obj);
}

GList* gbemol_mpd_output_get_list (GbemolMpd* obj)
{
	GList *l = NULL;
	GbemolMpdOutput* out;

	mpd_sendOutputsCommand (obj->priv->conn);

	while ((out = mpd_getNextOutput (obj->priv->conn)))
		l = g_list_append (l, out);

	if (gbemol_mpd_finish_and_handle (obj))
		return NULL;
	else
		return l;
}

void gbemol_mpd_output_set_active (GbemolMpd* obj, gint id, gboolean active)
{
	if (active)
		mpd_sendEnableOutputCommand (obj->priv->conn, id);
	else
		mpd_sendDisableOutputCommand (obj->priv->conn, id);

	gbemol_mpd_finish_and_handle (obj);
}

gint gbemol_mpd_get_crossfade (GbemolMpd* obj)
{
	if (obj->status && gbemol_mpd_check_permission (obj, "status"))
		return obj->status->crossfade;
	else
		return -1;
}

gint gbemol_mpd_get_state (GbemolMpd* obj)
{
	if (obj->status && gbemol_mpd_check_permission (obj, "status"))
		return obj->status->state;
	else
		return -1;
}

gint gbemol_mpd_get_current_id (GbemolMpd* obj)
{
	if (obj->status && gbemol_mpd_check_permission (obj, "status"))
		return obj->status->songid;
	else
		return -1;
}

gint gbemol_mpd_get_total_time (GbemolMpd* obj)
{
	if (obj->status && gbemol_mpd_check_permission (obj, "status"))
		return obj->status->totalTime;
	else
		return -1;
}

GbemolMpdSong* gbemol_mpd_songdup(GbemolMpdSong* song)
{
	return mpd_songDup(song);	
}
